Rem
Rem $Header: rdbms/demo/aqmonitor.sql /main/7 2012/08/01 20:21:19 maba Exp $
Rem
Rem aqmonitor.sql
Rem
Rem Copyright (c) 2009, 2012, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      aqmonitor.sql - AQ performance monitor package
Rem
Rem    DESCRIPTION
Rem      This file contains the header and body of the dbms_aq_monitor 
Rem      package. It provides automatic monitoring to specified queues,
Rem      and will generate reports based on monitoring results.     
Rem
Rem    NOTES
Rem 
Rem    Necessary Parameter Setting before Run:
Rem    1. Use add_queue_owner to add one or multiple queue owners to the 
Rem       monitoring list. The monitor will monitor queues belonging to 
Rem       these queue owners.
Rem    2. Use set_log_dir to set a directory for dumping log files. You need 
Rem       to have read and write privileges to the directory. If not set, the 
Rem       default log directory is /tmp.
Rem
Rem    Rules and Restrictions:
Rem    1. Queue owner must be added before the monitor is started. Otherwise 
Rem       statistics for new owners will not be collected. This is due to 
Rem       DBMS_SCHEDULER characteristics. 
Rem    2. Once a monitor is started, all previously collected statistics will 
Rem       be cleared. When the monitor is stopped, collected statistics will 
Rem       not be cleared.
Rem    3. At most one monitor instance is allowed at all time. It means that, 
Rem       if start_monitor procedure is successfully executed (in any session),
Rem       a new monitor will be started and all previous monitoring results 
Rem       will be cleared.
Rem    4. The monitor instance works across multiple sessions. For example, 
Rem       you could start the monitor in one session, and stop it in another 
Rem       session. A monitor operation can be executed in any session and will 
Rem       have the same effect.
Rem    5. Reports could be generated without stopping the monitor. Generating 
Rem       reports does not affect ongoing monitoring. 
Rem    6. It supports Oracle Database version 10.2/11.1/11.2
Rem
Rem
Rem    See http://dbdev.us.oracle.com/twiki/bin/view/Main/AQPerformanceMonitor
Rem    or aqmonitorREADME.txt for more details.
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    maba        07/25/12 - fixed aqmonitor privilege issue in 12c
Rem    maba        10/13/11 - fix lrg5090550
Rem    jmadduku    02/10/11 - Grant Unlimited Tablespace with RESOURCE
Rem    maba        01/07/11 - fix 9944205, add clean corrupted msg
Rem    xingjin     08/10/10 - bug fix 9960101. 
Rem    xingjin     12/09/09 - bug fix 9198962.
Rem    xingjin     11/05/09 - Created
Rem


------------------------------------------------------------------------
--         
--       1. Create an AQ User
--
------------------------------------------------------------------------

CONNECT &dba_name/&dba_passwd as sysdba;
SET echo OFF
SET serveroutput OFF
SET verify OFF

PROMPT Create a user for monitor package...
ACCEPT aqmon_user PROMPT Username:;
ACCEPT aqmon_pass PROMPT Password:;

DROP user &aqmon_user cascade;
CREATE user &aqmon_user identified by &aqmon_pass;

GRANT connect, resource, unlimited tablespace TO &aqmon_user;
GRANT select on sys.obj$ to &aqmon_user;
GRANT select on sys.tab$ to &aqmon_user;
GRANT select on sys.user$ to &aqmon_user;
GRANT create any directory TO &aqmon_user;
GRANT execute on DBMS_WORKLOAD_REPOSITORY to &aqmon_user;
GRANT aq_user_role to &aqmon_user;
GRANT aq_administrator_role TO &aqmon_user;
GRANT execute on dbms_aqadm to &aqmon_user;
GRANT execute on dbms_aq to &aqmon_user;
GRANT execute on sys.dbms_aqadm_sys to &aqmon_user;
GRANT execute ON sys.dbms_aqadm_syscalls TO &aqmon_user;
GRANT execute ON sys.dbms_aq_inv TO &aqmon_user;
GRANT create any job to &aqmon_user; 

-- do not use select_catalog_role, which cannot be inherited in procedures.
GRANT select any dictionary to &aqmon_user; 

grant select any table to &aqmon_user;

grant create view to &aqmon_user;

CONNECT &aqmon_user/&aqmon_pass;
SET echo OFF

--Create types and views for the healthcheck component to check for corrupted
--persistent messages

CREATE TYPE aq$_corrupted_message
AS OBJECT (
  msgid           RAW(16),      --Message id  of a message producer or consumer
  subscriber_id   NUMBER,                      -- subscriber id of the consumer 
  corruption_type NUMBER                          -- type of message corruption
 );
/
CREATE TYPE aq$_corrupted_message_t AS TABLE OF aq$_corrupted_message;
/

CREATE OR REPLACE FUNCTION aq$_get_corrupted_messages (
  qt_schema   IN  VARCHAR2,
  qt_name    IN  VARCHAR2,
  qt_flags    IN  BINARY_INTEGER) RETURN aq$_corrupted_message_t
PIPELINED IS
 
  type rt is       REF CURSOR;
  qt_rc            rt;
  subid       NUMBER; 
  msgid       RAW(16);
  dequeue_iot_sel_stmt           VARCHAR2(10240);
  queue_table_corrupted_sel_stmt  VARCHAR2(10240);
  history_iot_corrupted_msg      VARCHAR2(10240);
  -- Mark bug-9944205
  time_iot_corrupted_msg_stmt    VARCHAR2(10240);
  dequeue_log_corrupted_msg_stmt VARCHAR2(10240);
  queue_rentention               NUMBER;
  subscriber_count               NUMBER;
  qname_stmt                     VARCHAR2(10240);
  qname_rc                       rt;
  curr_qname                     VARCHAR2(100);
  
BEGIN 
  
  IF bitand(qt_flags, 8) != 0 THEN
  	--SELECT recipients INTO queue_type FROM all_queue_tables WHERE queue_table = UPPER(qt_name);
  	--DBMS_OUTPUT.put_line(queue_type);
  	
  	-- whatever in the queue table should be in the history IOT, time management IOT
  	queue_table_corrupted_sel_stmt := ' (SELECT msgid FROM ' ||
  	dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||qt_name||'"') ||
    ' qt WHERE NOT EXISTS ' ||
    ' ( SELECT * FROM ' ||
    dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||
    ' his WHERE his.msgid = qt.msgid ))';
    
    -- If the queue have rentention, then all msg should be in the time management IOT
    -- If the msg has expiration, then it should be in the time management IOT
    -- If the msg has delay, then it's must be either in the time management IOT or in the dequeue IOT
    SELECT ret_time INTO queue_rentention FROM system.aq$_queues q, system.aq$_queue_tables qt 
    WHERE q.table_objno = qt.objno AND qt.name = UPPER(qt_name) AND q.name NOT LIKE 'AQ$%';
    
    IF queue_rentention > 0 THEN
      queue_table_corrupted_sel_stmt := queue_table_corrupted_sel_stmt ||
      ' UNION '||
      ' (SELECT msgid FROM '||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name||'"') ||
      ' qt WHERE NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_T'||'"') ||
      ' t WHERE t.msgid = qt.msgid ))';
    ELSE
    	queue_table_corrupted_sel_stmt := queue_table_corrupted_sel_stmt ||
      ' UNION '||
      ' (SELECT msgid FROM '||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name||'"') ||
      ' qt WHERE ( expiration IS NOT NULL AND NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_T'||'"') ||
      ' t WHERE t.msgid = qt.msgid ))' ||
      ' OR ( delay IS NOT NULL AND ( NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_T'||'"') ||
      ' t WHERE t.msgid = qt.msgid ) AND NOT EXISTS '||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||
      ' his WHERE his.msgid = qt.msgid ))))';
    END IF;
      -- if the queue table is multi-consumer, there should be entry in dequeue IOT
    IF bitand(qt_flags, 1) = 1 THEN
      queue_table_corrupted_sel_stmt := queue_table_corrupted_sel_stmt ||
      ' UNION '||
      ' (SELECT msgid FROM '||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name||'"') ||
      ' qt WHERE delay IS NULL AND NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_I'||'"') ||
      ' dq WHERE dq.msgid = qt.msgid ))';
    END IF;
    
--    DBMS_OUTPUT.put_line (queue_table_corrupted_sel_stmt);
--    DBMS_OUTPUT.put_line ('queue_table_corrupted_sel_stmt');
    
    OPEN qt_rc FOR queue_table_corrupted_sel_stmt;
    LOOP
      FETCH qt_rc INTO msgid;
      EXIT WHEN qt_rc%NOTFOUND;
      PIPE ROW (aq$_corrupted_message(msgid, 0, 1));
    END LOOP;
    
    -- whatever in the time management IOT, if the action is ready(action = 0), should be an entry in queue table
    -- Not handling time management IOT issues, time manager process will fix them
    
    -- whatever in the dequeue log IOT should be an entry in queue table, if the dequeue_time is NULL there should also be entries in the History IOT
    dequeue_log_corrupted_msg_stmt := ' (SELECT msgid, subscriber# FROM '||
    dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_L'||'"') ||
    ' dqlog WHERE NOT EXISTS ' ||
    ' ( SELECT * FROM ' ||
    dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||qt_name || '"') ||
    ' qt WHERE dqlog.msgid = qt.msgid ) group by msgid, subscriber#) '||
    'UNION (SELECT msgid, subscriber# FROM '||
    dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_L'||'"') ||
    ' dqlog WHERE dequeue_time IS NULL AND NOT EXISTS ' ||
    ' ( SELECT * FROM ' ||
    dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
    dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||
    ' his WHERE dqlog.msgid = his.msgid) group by msgid, subscriber#)';
    OPEN qt_rc FOR dequeue_log_corrupted_msg_stmt;
    LOOP
      FETCH qt_rc INTO msgid, subid;
      EXIT WHEN qt_rc%NOTFOUND;
      PIPE ROW (aq$_corrupted_message(msgid, subid, 5));
    END LOOP;
    
--    DBMS_OUTPUT.put_line (dequeue_log_corrupted_msg_stmt);
--    DBMS_OUTPUT.put_line ('dequeue_log_corrupted_msg_stmt');
    
    IF bitand(qt_flags, 1) = 1 THEN
      -- if multi-consumer queue, whatever in the dequeue IOT should be in queue table, and history IOT
      dequeue_iot_sel_stmt := ' (SELECT msgid, subscriber# FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_I'||'"') ||
      ' dqt WHERE NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name || '"') ||
      ' qt WHERE dqt.msgid = qt.msgid ) group by msgid, subscriber#)' ||
      ' UNION '||
      ' (SELECT msgid, subscriber# FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_I'||'"') ||
      ' dqt WHERE NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||
      ' his WHERE dqt.msgid = his.msgid ) group by msgid, subscriber#)';
      OPEN qt_rc FOR dequeue_iot_sel_stmt;
      LOOP
        FETCH qt_rc INTO msgid, subid;
        EXIT WHEN qt_rc%NOTFOUND;
        PIPE ROW (aq$_corrupted_message(msgid, subid, 2));
      END LOOP;
      
      -- history IOT corrupted message: message in history IOT should be in QT
      history_iot_corrupted_msg := ' (SELECT msgid, subscriber# FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||
      ' his WHERE NOT EXISTS ' ||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name || '"') ||
      ' qt WHERE his.msgid = qt.msgid ) GROUP BY msgid, subscriber#) ';
      
      -- If a history state for the subscriber indicates it is not dequeued
      -- a)	If not delay, a row for the (Subid, messageid) is either in the dequeue log or in the dequeue iot
      -- b)	If delay, then a row for the messageid is in the dequeue iot / or time management IOT
      history_iot_corrupted_msg := history_iot_corrupted_msg ||
      ' UNION ' ||
      ' (SELECT his.msgid, his.subscriber# FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||' his, '||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name || '"') ||
      ' qt WHERE his.msgid = qt.msgid AND his.dequeue_time IS NULL AND qt.delay IS NULL '||
      ' AND NOT EXISTS '||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_L'||'"') ||
      ' log WHERE his.msgid = log.msgid AND his.subscriber# = log.subscriber# ) '||
      ' AND NOT EXISTS '||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_I'||'"') ||
      ' dqt WHERE his.msgid = dqt.msgid AND his.subscriber# = dqt.subscriber# )) ';
      
      history_iot_corrupted_msg := history_iot_corrupted_msg ||
      ' UNION ' ||
      ' (SELECT his.msgid, his.subscriber# FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_H'||'"') ||' his, '||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||qt_name || '"') ||
      ' qt WHERE his.msgid = qt.msgid AND his.dequeue_time IS NULL AND qt.delay IS NOT NULL '||
      ' AND NOT EXISTS '||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_T'||'"') ||
      ' t WHERE his.msgid = t.msgid ) '||
      ' AND NOT EXISTS '||
      ' ( SELECT * FROM ' ||
      dbms_assert.enquote_name('"'||qt_schema||'"') || '.' ||
      dbms_assert.enquote_name('"'||'AQ$_' || qt_name || '_I'||'"') ||
      ' dqt WHERE his.msgid = dqt.msgid AND his.subscriber# = dqt.subscriber# )) ';

--      DBMS_OUTPUT.put_line(history_iot_corrupted_msg);
--      DBMS_OUTPUT.put_line('*****************');

      OPEN qt_rc FOR history_iot_corrupted_msg;
      LOOP
        FETCH qt_rc INTO msgid, subid;
        EXIT WHEN qt_rc%NOTFOUND;
        PIPE ROW (aq$_corrupted_message(msgid, subid, 3));
      END LOOP;
      
    END IF;
  END IF;
  
  RETURN;
END;
/

show errors

CREATE OR REPLACE VIEW AQHC_MESSAGES_HEALTHCHECK
AS 
SELECT t.schema OWNER, q.name QUEUE_NAME, t.name QUEUE_TABLE,
       s.msgid MSGID, s.subscriber_id SUBSCRIBER#,
       decode(s.corruption_type, 
       1, 'Queue Table corrupted',
       2, 'Metadata in Dequeue IOT corrupted',
       3, 'Metadata in History IOT corrupted',
       4, 'Metadata in Time Manager IOT corrupted',
       5, 'Metadata in Dequeue Log IOT corrupted') CORRUPTION_CAUSE 
FROM   system.aq$_queues q, system.aq$_queue_tables t,
       TABLE (aq$_get_corrupted_messages(t.schema, t.name, t.flags)) s
WHERE  q.table_objno = t.objno
AND    bitand(t.flags, 1) = 1 and q.usage!=1
/

------------------------------------------------------------------------
--         
--       2. Create Monitoring Package
--
------------------------------------------------------------------------

CREATE OR REPLACE DIRECTORY AQMON_LOGDIR AS '/tmp';


-- Initialize the monitor. Object created:
--   Tables:   1.  aqmon_pqueues_tab          --|
--             2.  aqmon_pqueues_accum_tab      |--> persistent queues
--             3.  aqmon_psubscribers_tab       |
--             4.  aqmon_pqmncache_tab        --|
--             5.  aqmon_bqueues_tab            --|
--             6.  aqmon_bqueues_accum_tab        |--> buffered queues  
--             7.  aqmon_bsubscribers_tab       --|  
--             8.  aqmon_queue_config_tab     --|
--             9.  aqmon_subscriber_config_tab  |--> general purpose
--             10. aqmon_queue_owners_tab       |
--             11. aqmon_params_tab           --| 
--             12. aqmon_subreg_tab   
--
--  Sequence:  1.  aqmon_monitor_seq
--
-- For all the created tables, there is a field flag
--   flag = 0 - for collected snapshots
--   flag = 1 - for manually generated snapshot 
--
--
-- A simple comparison
--                               11.2       11.1       10.2
-- gv$persistent_queues           Y          Y          N
-- gv$persistent_subscribers      Y          Y          N
-- gv$persistent_qmn_cache        Y          N          N
-- gv$buffered_queues             Y          Y          Y
-- gv$buffered_subscribers        Y          Y          Y
-- gv$subscr_registration_stats   Y          Y          N
--
CREATE OR REPLACE PROCEDURE prepare_monitoring AUTHID CURRENT_USER AS
BEGIN 

  -- create monitoring tables for (1) gv$persistent_queues,  
  -- (2) gv$persistent_subscribers and (3) dumping accumulative statistics 
  -- for persistent queues
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqueues_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;

  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_psubscribers_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;

  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqueues_accum_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;

  BEGIN
    -- table 1
    EXECUTE IMMEDIATE 'CREATE TABLE aqmon_pqueues_tab AS SELECT  ' || 
         ' * FROM gv$persistent_queues WHERE 1 > 2';   
    EXECUTE IMMEDIATE 'ALTER TABLE aqmon_pqueues_tab ADD ' ||
         ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
         ' aqmon_pqueues_tab TO PUBLIC';

    -- table 2
    EXECUTE IMMEDIATE 'CREATE TABLE aqmon_pqueues_accum_tab ' ||
              ' (INST_ID                   NUMBER NOT NULL, ' ||
              '  QUEUE_ID                  NUMBER NOT NULL, ' ||
              '  QUEUE_NAME                VARCHAR2(30),    ' ||
              '  ENQUEUED_MSGS             NUMBER,          ' ||
              '  DEQUEUED_MSGS             NUMBER,          ' ||
              '  BROWSED_MSGS              NUMBER,          ' ||
              '  ENQUEUED_EXPIRY_MSGS      NUMBER,          ' ||
              '  ENQUEUED_DELAY_MSGS       NUMBER,          ' ||
              '  MSGS_MADE_EXPIRED         NUMBER,          ' ||
              '  MSGS_MADE_READY           NUMBER,          ' ||
              '  ENQUEUE_RATE              NUMBER,          ' ||
              '  DEQUEUE_RATE              NUMBER,          ' ||
              '  AVG_TIME_PER_ENQUEUE      NUMBER,          ' ||
              '  AVG_TIME_PER_DEQUEUE      NUMBER,          ' ||
              '  TRANSFORMATION_TIME       NUMBER,          ' ||
              '  RULE_EVAL_TIME            NUMBER,          ' ||
              '  PENDING_MSGS              NUMBER,          ' ||
              '  LAST_ENQUEUE_TIME         TIMESTAMP,       ' ||
              '  LAST_DEQUEUE_TIME         TIMESTAMP,       ' ||
              '  LAST_TM_EXPIRY_TIME       TIMESTAMP,       ' ||
              '  LAST_TM_READY_TIME        TIMESTAMP,       ' ||
          -- columns for 11.2 or above
              '  AVG_CPU_TIME_PER_ENQUEUE  NUMBER,          ' ||
              '  AVG_CPU_TIME_PER_DEQUEUE  NUMBER,          ' ||
              '  ENQUEUE_TRANSACTIONS      NUMBER,          ' ||
              '  DEQUEUE_TRANSACTIONS      NUMBER,          ' ||
              '  EXECUTION_COUNT           NUMBER,          ' ||
              '  AVG_MSG_AGE               NUMBER,          ' ||
              '  DEQUEUED_MSG_LATENCY      NUMBER,          ' ||
          -- end columns for 11.2 or above
              '  TIME_DURATION             NUMBER,          ' ||
              '  FIRST_MONITOR_TIME        TIMESTAMP,       ' ||
              '  LAST_MONITOR_TIME         TIMESTAMP)       '; 
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
              ' aqmon_pqueues_accum_tab TO PUBLIC';

    -- table 3
    EXECUTE IMMEDIATE 'CREATE TABLE aqmon_psubscribers_tab AS SELECT  ' || 
           ' * FROM gv$persistent_subscribers WHERE 1 > 2';
    EXECUTE IMMEDIATE 'ALTER TABLE aqmon_psubscribers_tab ADD ' ||
           ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' ||
           ' aqmon_psubscribers_tab TO PUBLIC';

  EXCEPTION WHEN OTHERS THEN NULL;
    -- should work fine for 11.1 or 11.2 database
    -- may throw 'table or view does not exist' exception in 10.2 database   
  END;


  -- create a monitoring table for gv$persistent_qmn_cache
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqmncache_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  BEGIN
    EXECUTE IMMEDIATE 'CREATE TABLE aqmon_pqmncache_tab AS SELECT  ' || 
           ' * FROM gv$persistent_qmn_cache WHERE 1 > 2';
    EXECUTE IMMEDIATE 'ALTER TABLE aqmon_pqmncache_tab ADD ' ||
           ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
           ' aqmon_pqmncache_tab TO PUBLIC';
  EXCEPTION WHEN OTHERS THEN NULL;
    -- should work fine for 11.2 database 
    -- may throw 'table or view does not exist' exception in 10.2/11.1 database
  END;


  -- create a monitoring table for gv$buffered_queues 
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_bqueues_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_bqueues_tab AS SELECT  ' || 
       ' * FROM gv$buffered_queues WHERE 1 > 2';   
  EXECUTE IMMEDIATE 'ALTER TABLE aqmon_bqueues_tab ADD ' ||
       ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
       ' aqmon_bqueues_tab TO PUBLIC';


  -- create a table for dumping accumulative statistics for buffered queues
  BEGIN  
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_bqueues_accum_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_bqueues_accum_tab ' ||
            ' (INST_ID                   NUMBER NOT NULL, ' ||
            '  QUEUE_ID                  NUMBER NOT NULL, ' ||
            '  QUEUE_NAME                VARCHAR2(30),    ' ||
            '  ENQUEUED_MSGS             NUMBER,          ' ||
            '  DEQUEUED_MSGS             NUMBER,          ' ||
            '  SPILLED_MSGS              NUMBER,          ' ||
            '  EXPIRED_MSGS              NUMBER,          ' ||
            '  ENQUEUE_RATE              NUMBER,          ' ||
            '  DEQUEUE_RATE              NUMBER,          ' ||
            '  PENDING_MSGS              NUMBER,          ' ||
          -- columns for 11.2 or above
            '  AVG_TIME_PER_ENQUEUE      NUMBER,          ' ||
            '  AVG_TIME_PER_DEQUEUE      NUMBER,          ' ||
            '  AVG_CPU_TIME_PER_ENQUEUE  NUMBER,          ' ||
            '  AVG_CPU_TIME_PER_DEQUEUE  NUMBER,          ' ||
            '  TRANSFORMATION_TIME       NUMBER,          ' ||
            '  RULE_EVAL_TIME            NUMBER,          ' ||
            '  OLDEST_MSGID              RAW(16),         ' ||
            '  OLDEST_MSG_ENQTM          TIMESTAMP,       ' ||
            '  LAST_ENQUEUE_TIME         TIMESTAMP,       ' ||
            '  LAST_DEQUEUE_TIME         TIMESTAMP,       ' ||
          -- end columns for 11.2 or above
            '  TIME_DURATION             NUMBER,          ' ||
            '  FIRST_MONITOR_TIME        TIMESTAMP,       ' ||
            '  LAST_MONITOR_TIME         TIMESTAMP)       ';   
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
            ' aqmon_bqueues_accum_tab TO PUBLIC';


  -- create a monitoring table for gv$buffered_subscribers
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_bsubscribers_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_bsubscribers_tab AS SELECT  ' || 
         ' * FROM gv$buffered_subscribers WHERE 1 > 2';
  EXECUTE IMMEDIATE 'ALTER TABLE aqmon_bsubscribers_tab ADD ' ||
         ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' ||
         ' aqmon_bsubscribers_tab TO PUBLIC';


  -- create a config info table for all queues
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_queue_config_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_queue_config_tab ' ||
            ' (QUEUE_ID           NUMBER NOT NULL, ' ||
            '  QUEUE_NAME         VARCHAR2(30),    ' ||
            '  QUEUE_TABLE        VARCHAR2(30),    ' ||
            '  QUEUE_OWNER        VARCHAR2(30),    ' ||
            '  QUEUE_TYPE         VARCHAR2(20),    ' ||
            '  DATA_TYPE          VARCHAR2(7) ,    ' ||
            '  OBJECT_TYPE        VARCHAR2(61),    ' ||
            '  SORT_ORDER         VARCHAR2(22),    ' ||
            '  RECIPIENTS         VARCHAR2(8) ,    ' ||
            '  MESSAGE_GROUPING   VARCHAR2(13),    ' ||
            '  CONSTRAINT  pk_qconf  PRIMARY KEY (QUEUE_ID))'; 
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
            ' aqmon_queue_config_tab TO PUBLIC';


  -- create a config info table for all subscribers
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_subscriber_config_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_subscriber_config_tab ' || 
            ' (QUEUE_NAME         VARCHAR2(30),  ' ||
            '  QUEUE_TABLE        VARCHAR2(30),  ' ||
            '  CONSUMER_NAME      VARCHAR2(30),  ' ||
            '  TRANSFORMATION     VARCHAR2(61),  ' ||
            '  DELIVERY_MODE      VARCHAR2(22),  ' ||
            '  QUEUE_TO_QUEUE     VARCHAR2(5) ,  ' ||
            '  OWNER              VARCHAR2(30),  ' ||
            '  ADDRESS            VARCHAR2(1024))';
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
            ' aqmon_subscriber_config_tab TO PUBLIC';


  -- create a table for storing the monitoring list.
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_queue_owners_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_queue_owners_tab ' || 
         ' (QUEUE_OWNER    VARCHAR2(30) NOT NULL)';
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
         ' aqmon_queue_owners_tab TO PUBLIC';


  -- create a table for storing runtime parameters (so that multiple sessions
  -- will read the same values). Parameters include: 
  --
  --   APT_COLLECT_CONFIG_FREQ  : Frequency for collecting queue
  --                              configuration info (in second). 
  --
  --   APT_COLLECT_RUNTIME_FREQ : Frequency for collecting queue runtime
  --                              stats (in second). 
  --
  --   APT_COLLECT_ACCUM_FREQ   : Frequency for caculating and dumping 
  --                              accumulative statistics in log file 
  --                              (in second).
  --
  --   APT_DUMP_AWR_FREQ        : Frequency for generating AWR reports 
  --                              (in second).
  --
  --   APT_MONITOR_STATUS       : current monitor status
  -- 
  --   APT_BEGIN_SNAPSHOT_ID  
  --   APT_END_SNAPSHOT_ID      : Begin and end snapshot IDs, used for top 
  --                              SQL analysis.
  --  
  --   APT_BEGIN_MONITOR_TM    
  --   APT_END_MONITOR_TM       : Begin and end monitoring time. Similar to 
  --                              APT_BEGIN_SNAPSHOT_ID/APT_END_SNAPSHOT_ID.
  --
  --   APT_DATABASE_VERSION     : Oracle database version. It could be
  --                              10.2, or 11.1, or 11.2, or 0 (unknown 
  --                              version).
  -- 
  --   APT_DUMP_PLOTTING_FILE   : indicate whether dump each queue's runtime 
  --                              statistics into a separate file, which will 
  --                              be used by gnuplot for plotting.
  --                              It could be 1 (yes) or 0 (no).  
  --
  --   APT_SHARED_SUBSQL_LENGTH : 
  --   APT_SHARED_MEMORY_BOUND  :  The monitor will display SQLs that are 
  --                               similar (i.e., their share the same first
  --                               $APT_SHARED_SUBSQL_LENGTH$ characters) and
  --                               in total consume more than 
  --                               $APT_SHARED_MEMORY_BOUND$ MBytes memory
  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_params_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  EXECUTE IMMEDIATE 'CREATE TABLE aqmon_params_tab ' ||
         ' (PARAM_NAME    VARCHAR2(30) NOT NULL, '   ||
         '  PARAM_VALUE   NUMBER, '                  ||
         '  PARAM_VALUE2  TIMESTAMP, '               ||
         '  CONSTRAINT pk_params PRIMARY KEY (PARAM_NAME))';      
  EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
         ' aqmon_params_tab TO PUBLIC';


  BEGIN 
    EXECUTE IMMEDIATE 'DROP TABLE aqmon_subreg_tab';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  BEGIN
    EXECUTE IMMEDIATE 'CREATE TABLE aqmon_subreg_tab AS SELECT  ' || 
         ' * FROM gv$subscr_registration_stats WHERE 1 > 2';   
    EXECUTE IMMEDIATE 'ALTER TABLE aqmon_subreg_tab ADD ' ||
         ' (memseq NUMBER, monitor_time TIMESTAMP, flag NUMBER)';
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, DELETE, UPDATE on ' || 
         ' aqmon_subreg_tab TO PUBLIC';
  EXCEPTION WHEN OTHERS THEN NULL;
    -- should work fine for 11.1 or 11.2 database
    -- may throw 'table or view does not exist' exception in 10.2 database   
  END;


  -- create a sequence for collected snapshots
  BEGIN 
    EXECUTE IMMEDIATE 'DROP SEQUENCE aqmon_monitor_seq';
  EXCEPTION WHEN OTHERS THEN NULL;
  END;
  -- the max value is set to 99999, because in the log file a sequence
  -- number is given 5 characters for displaying.
  EXECUTE IMMEDIATE 'CREATE SEQUENCE aqmon_monitor_seq ' || 
            ' START WITH 1 MAXVALUE 99999 CYCLE NOCACHE';
  EXECUTE IMMEDIATE 'GRANT SELECT ON aqmon_monitor_seq TO PUBLIC';

END prepare_monitoring;
/
show errors

EXEC prepare_monitoring;  



-- Public Package for Users
CREATE OR REPLACE PACKAGE dbms_aq_monitor AUTHID CURRENT_USER AS
  
  -----------------------------------------------------------------------
  -- PART 1: set/get monitor configuration parameters
  -----------------------------------------------------------------------

  -- Add a queue owner to the monitoring list. The owner name will be 
  -- automatically capitalized. All queues belonging to that owner will 
  -- be monitored.
  PROCEDURE add_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL); 


  -- Remove a queue owner from the monitoring list. The owner name will be 
  -- automatically capitalized.
  PROCEDURE remove_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL);


  -- Set directory for dumping log files.
  --
  -- PARAM  : overwrite_file - whether to overwrite existing file if file
  --                           'aqmon_report.log' already exists in the log 
  --                           directory.
  PROCEDURE set_log_dir(dir_name       IN VARCHAR2 DEFAULT NULL,
                        overwrite_file IN BOOLEAN DEFAULT FALSE);


  -- Set monitoring parameters
  --
  -- PARAM  : A list of param_names is as follows.
  -- 
  ---------------------------------------------------------------------------
  --Param Name            Default Value  Usage
  ---------------------------------------------------------------------------
  --CONFIG_COLLECT_FREQ   300(seconds)   frequency for collecting
  --                                     queue configuration info
  --RUNTIME_COLLECT_FREQ  60(seconds)    frequency for collecting 
  --                                     queue runtime stats
  --AWR_DUMP_FREQ         300(seconds)   frequency for generating AWR 
  --                                     report
  --ACCUM_DUMP_FREQ       300(seconds)   frequency for dumping 
  --                                     accumulative stats in log file
  --FILE_PLOT             0              whether dump per-queue stats file
  --                                     for gnuplot. Value could be 0 or 1.
  --SHARED_SUBSQL_LENGTH  150  
  --SHARED_MEMEORY_BOUND  1 (MByte)      The monitor will display SQLs that
  --                                     are similar (i.e., their share the
  --                                     same first $APT_SHARED_SUBSQL_LENGTH$
  --                                     characters) and in total consume more
  --                                     than $APT_SHARED_MEMORY_BOUND$ MBytes
  --                                     memory
  ---------------------------------------------------------------------------
  PROCEDURE set_monitoring_param(param_name  IN VARCHAR2 DEFAULT NULL, 
                                 param_value IN NUMBER DEFAULT NULL);


  -- Display monitor configuration parameters.
  PROCEDURE show_config;



  -----------------------------------------------------------------------
  -- PART 2: monitor operations
  -----------------------------------------------------------------------

  -- Start the monitor. When started, the monitor will clear up all 
  -- previously collected statistics. It will also overwrite log files 
  -- in the log directory. 
  --
  -- PARAM  : start_time - number of minutes after which the monitor
  --                       will be started  
  PROCEDURE start_monitor(start_time IN NUMBER DEFAULT 0);


  -- Stop all statistics collecting processes. Existing statistics are 
  -- still kept in database.
  --
  -- PARAM  : stop_time - number of minutes after which the monitor
  --                      will be stopped
  PROCEDURE stop_monitor(stop_time IN NUMBER DEFAULT 0); 


  -- Clear all tables and data used by the monitor.
  PROCEDURE destroy_monitor;


  -- Clear scheduled job
  --
  -- PARAM  : job_name - could be 'start_monitor' or 'stop_monitor'
  PROCEDURE clear_scheduled_job(job_name IN VARCHAR2 DEFAULT NULL);  


  -- Generate a summary monitoring report for all monitored queues (on all 
  -- instances).
  -- 
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_report(final_collect IN BOOLEAN DEFAULT TRUE);


  -- Generate a detailed report for a specific queue on a specific instance.
  --
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_queue_report(queue_id       IN NUMBER DEFAULT NULL,
                             inst_number    IN NUMBER DEFAULT NULL,
                             final_collect  IN BOOLEAN DEFAULT TRUE,
                             single_logfile IN BOOLEAN DEFAULT FALSE);
  

  -- Generate a detailed report for each of the monitored queues on the 
  -- specified instance.
  -- 
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_all_queues_reports(inst_number    IN NUMBER DEFAULT NULL,
                                   final_collect  IN BOOLEAN DEFAULT TRUE,
                                   single_logfile IN BOOLEAN DEFAULT FALSE);

  
  -- Generate a report for subscr-registration stats.
  -- (valid for 11.1/11.2)  
  -- 
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_subreg_report(final_collect IN BOOLEAN DEFAULT TRUE);
  
  PROCEDURE hcheck_clean_queue_table_msg(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL); 
  
  --Check that the components of a queue table are VALID
  
  PROCEDURE hcheck_queue_table(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL);
 
  --Check for logically corrupted messages in a particular queue table
  --and provide suggestions for resolution

  PROCEDURE hcheck_queue_table_messages(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL);

  --Check relevant AQ init.ora parameters are set correctly 

  PROCEDURE hcheck_parameters;
 
  --Check that the AQ metadata in system.aq$_queues and system.aq$queue_tables is consistent with what is in obj$
  
  PROCEDURE hcheck_dictionary;
 
  --Check for errors in dba_queue_schedules and common propagation issues
 
  PROCEDURE hcheck_propagation_sender;
  
  --Check for issues associated with the propagation receiver processes
  
  PROCEDURE hcheck_propagation_receiver;
 
  --Check for issues associated with qmon processes
  
  PROCEDURE hcheck_qmon;
  
  --Check for issues associated with notification
  
  PROCEDURE hcheck_notification;
  
  --Execute all AQ healthcheck procedures
  
  PROCEDURE hcheck_all;

END dbms_aq_monitor;
/
show errors 



-- Utility package for AQ monitor
CREATE OR REPLACE PACKAGE dbms_aq_monitor_util AS

  -----------------------------------------------------------------------
  -- PART 1: set/get monitor configuration parameters
  -----------------------------------------------------------------------

  -- Add a queue owner to the monitoring list. The owner name will be 
  -- automatically capitalized. All queues belonging to that owner will 
  -- be monitored.
  PROCEDURE add_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL); 


  -- Remove a queue owner from the monitoring list. The owner name will be 
  -- automatically capitalized.
  PROCEDURE remove_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL);


  -- Set directory for dumping log files.
  --
  -- PARAM  : overwrite_file - whether to overwrite existing file if file
  --                           'aqmon_report.log' already exists in the log 
  --                           directory.
  PROCEDURE set_log_dir(dir_name       IN VARCHAR2 DEFAULT NULL,
                        overwrite_file IN BOOLEAN DEFAULT FALSE);


  -- Set monitoring parameters
  --
  -- PARAM  : A list of param_names is as follows.
  -- 
  ---------------------------------------------------------------------------
  --Param Name            Default Value  Usage
  ---------------------------------------------------------------------------
  --CONFIG_COLLECT_FREQ   300(seconds)   frequency for collecting
  --                                     queue configuration info
  --RUNTIME_COLLECT_FREQ  60(seconds)    frequency for collecting 
  --                                     queue runtime stats
  --AWR_DUMP_FREQ         300(seconds)   frequency for generating AWR 
  --                                     report
  --ACCUM_DUMP_FREQ       300(seconds)   frequency for dumping 
  --                                     accumulative stats in log file
  --FILE_PLOT             0              whether dump per-queue stats file
  --                                     for gnuplot. Value could be 0 or 1.
  --SHARED_SUBSQL_LENGTH  150  
  --SHARED_MEMEORY_BOUND  1 (MByte)      The monitor will display SQLs that
  --                                     are similar (i.e., their share the
  --                                     same first $APT_SHARED_SUBSQL_LENGTH$
  --                                     characters) and in total consume more
  --                                     than $APT_SHARED_MEMORY_BOUND$ MBytes
  --                                     memory
  ---------------------------------------------------------------------------
  PROCEDURE set_monitoring_param(param_name  IN VARCHAR2 DEFAULT NULL, 
                                 param_value IN NUMBER DEFAULT NULL);


  -- Display monitor configuration parameters.
  PROCEDURE show_config;



  -----------------------------------------------------------------------
  -- PART 2: monitor operations
  -----------------------------------------------------------------------

  -- Start the monitor. When started, the monitor will clear up all 
  -- previously collected statistics. It will also overwrite log files 
  -- in the log directory. 
  --
  -- PARAM  : start_time - number of minutes after which the monitor
  --                       will be started  
  PROCEDURE start_monitor(start_time IN NUMBER DEFAULT 0);


  -- Stop all statistics collecting processes. Existing statistics are 
  -- still kept in database.
  --
  -- PARAM  : stop_time - number of minutes after which the monitor
  --                      will be stopped
  PROCEDURE stop_monitor(stop_time IN NUMBER DEFAULT 0); 


  -- Clear all tables and data used by the monitor.
  PROCEDURE destroy_monitor;


  -- Clear scheduled job
  --
  -- PARAM  : job_name - could be 'start_monitor' or 'stop_monitor'
  PROCEDURE clear_scheduled_job(job_name IN VARCHAR2 DEFAULT NULL);  


  -- Generate a summary monitoring report for all monitored queues (on all 
  -- instances).
  --
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_report(final_collect IN BOOLEAN DEFAULT TRUE);


  -- Generate a detailed report for a specific queue on a specific instance.
  --
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_queue_report(queue_id       IN NUMBER DEFAULT NULL,
                             inst_number    IN NUMBER DEFAULT NULL,
                             final_collect  IN BOOLEAN DEFAULT TRUE,
                             single_logfile IN BOOLEAN DEFAULT FALSE);
  

  -- Generate a detailed report for each of the monitored queues on the 
  -- specified instance.
  -- 
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_all_queues_reports(inst_number    IN NUMBER DEFAULT NULL,
                                   final_collect  IN BOOLEAN DEFAULT TRUE,
                                   single_logfile IN BOOLEAN DEFAULT FALSE);


  -- Generate a report for subscr-registration stats.
  -- (valid for 11.1/11.2)  
  -- 
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_subreg_report(final_collect IN BOOLEAN DEFAULT TRUE);


  -----------------------------------------------------------------------
  -- PART3: PACKAGE PROCEDURES CALLED BY DBMS_SCHEDULER
  -----------------------------------------------------------------------

  -- Collect queue config info and dump it into corresponding tables.
  PROCEDURE collect_config;


  -- Collect queue runtime stats and dump it into corresponding tables.
  PROCEDURE collect_runtime;


  -- Immediately start the monitor  
  PROCEDURE start_monitor_immed;


  -- Immediately stop the monitor  
  PROCEDURE stop_monitor_immed;

  --Check that the components of a queue table are VALID
  
  PROCEDURE hcheck_queue_table(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL);
 
  --Check for logically corrupted messages in a particular queue table
  --and provide suggestions for resolution

  PROCEDURE hcheck_queue_table_messages(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL);

  -- clean potentially corrupted queue messages
  PROCEDURE hcheck_clean_queue_table_msg(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL);
  --Check relevant AQ init.ora parameters are set correctly 

  PROCEDURE hcheck_parameters;
 
  --Check that the AQ metadata in system.aq$_queues and system.aq$queue_tables is consistent with what is in obj$
  
  PROCEDURE hcheck_dictionary;
 
  --Check for errors in dba_queue_schedules and common propagation issues
 
  PROCEDURE hcheck_propagation_sender;
  
  --Check for issues associated with the propagation receiver processes
  
  PROCEDURE hcheck_propagation_receiver;
 
  --Check for issues associated with qmon processes
  
  PROCEDURE hcheck_qmon;
  
  --Check for issues associated with notification
  
  PROCEDURE hcheck_notification;
  
  --Execute all AQ healthcheck procedures
  
  PROCEDURE hcheck_all;

END dbms_aq_monitor_util;
/
show errors;



CREATE OR REPLACE PACKAGE BODY dbms_aq_monitor AS


  -----------------------------------------------------------------------
  -- PART 1: set/get monitor configuration parameters
  -----------------------------------------------------------------------

  -- Add a queue owner to the monitoring list. The owner name will be 
  -- automatically capitalized. All queues belonging to that owner will 
  -- be monitored.
  PROCEDURE add_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.add_queue_owner(owner_name);
  END add_queue_owner;
  

  -- Remove a queue owner from the monitoring list. The owner name will be 
  -- automatically capitalized.
  PROCEDURE remove_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.remove_queue_owner(owner_name);
  END remove_queue_owner;


  -- Set directory for dumping log files.
  --
  -- PARAM  : overwrite_file - whether to overwrite existing file if file
  --                           'aqmon_report.log' already exists in the log 
  --                           directory.
  PROCEDURE set_log_dir(dir_name       IN VARCHAR2 DEFAULT NULL,
                        overwrite_file IN BOOLEAN DEFAULT FALSE) AS
  BEGIN
    dbms_aq_monitor_util.set_log_dir(dir_name, overwrite_file);
  END set_log_dir;


  -- Set monitoring parameters
  --
  -- PARAM  : A list of param_names is as follows.
  -- 
  ---------------------------------------------------------------------------
  --Param Name            Default Value  Usage
  ---------------------------------------------------------------------------
  --CONFIG_COLLECT_FREQ   300(seconds)   frequency for collecting
  --                                     queue configuration info
  --RUNTIME_COLLECT_FREQ  60(seconds)    frequency for collecting 
  --                                     queue runtime stats
  --AWR_DUMP_FREQ         300(seconds)   frequency for generating AWR 
  --                                     report
  --ACCUM_DUMP_FREQ       300(seconds)   frequency for dumping 
  --                                     accumulative stats in log file
  --FILE_PLOT             0              whether dump per-queue stats file
  --                                     for gnuplot. Value could be 0 or 1.
  --SHARED_SUBSQL_LENGTH  150  
  --SHARED_MEMEORY_BOUND  1 (MByte)      The monitor will display SQLs that
  --                                     are similar (i.e., their share the
  --                                     same first $APT_SHARED_SUBSQL_LENGTH$
  --                                     characters) and in total consume more
  --                                     than $APT_SHARED_MEMORY_BOUND$ MBytes
  --                                     memory
  ---------------------------------------------------------------------------
  PROCEDURE set_monitoring_param(param_name  IN VARCHAR2 DEFAULT NULL, 
                                 param_value IN NUMBER DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.set_monitoring_param(param_name, param_value);
  END set_monitoring_param;


  -- Display monitor configuration parameters.
  PROCEDURE show_config AS
  BEGIN
    dbms_aq_monitor_util.show_config;
  END show_config;



  -----------------------------------------------------------------------
  -- PART 2: monitor operations
  -----------------------------------------------------------------------

  -- Start the monitor. When started, the monitor will clear up all 
  -- previously collected statistics. It will also overwrite log files 
  -- in the log directory. 
  --
  -- PARAM  : start_time - number of minutes after which the monitor
  --                       will be started  
  PROCEDURE start_monitor(start_time IN NUMBER DEFAULT 0) AS
  BEGIN
    dbms_aq_monitor_util.start_monitor(start_time);
  END start_monitor;


  -- Stop all statistics collecting processes. Existing statistics are 
  -- still kept in database.
  --
  -- PARAM  : stop_time - number of minutes after which the monitor
  --                      will be stopped
  PROCEDURE stop_monitor(stop_time IN NUMBER DEFAULT 0) AS
  BEGIN
    dbms_aq_monitor_util.stop_monitor(stop_time);
  END stop_monitor;


  -- Clear all tables and data used by the monitor.
  PROCEDURE destroy_monitor AS
  BEGIN
    dbms_aq_monitor_util.destroy_monitor;
  END;


  -- Clear scheduled job
  --
  -- PARAM  : job_name - could be 'start_monitor' or 'stop_monitor'
  PROCEDURE clear_scheduled_job(job_name IN VARCHAR2 DEFAULT NULL) AS 
  BEGIN
    dbms_aq_monitor_util.clear_scheduled_job(job_name);
  END clear_scheduled_job;


  -- Generate a summary monitoring report for all monitored queues (on all 
  -- instances).  
  --
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_report(final_collect IN BOOLEAN DEFAULT TRUE) AS
  BEGIN
    dbms_aq_monitor_util.get_report(final_collect);
  END get_report;
 

  -- Generate a detailed report for a specific queue on a specific instance.
  --
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_queue_report(queue_id       IN NUMBER DEFAULT NULL,
                             inst_number    IN NUMBER DEFAULT NULL,
                             final_collect  IN BOOLEAN DEFAULT TRUE,
                             single_logfile IN BOOLEAN DEFAULT FALSE) AS
  BEGIN 
    dbms_aq_monitor_util.get_queue_report(queue_id, inst_number, 
                                          final_collect, single_logfile);
  ENd get_queue_report;

  
  -- Generate a detailed report for each of the monitored queues on the 
  -- specified instance.
  -- 
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_all_queues_reports(inst_number    IN NUMBER DEFAULT NULL,
                                   final_collect  IN BOOLEAN DEFAULT TRUE,
                                   single_logfile IN BOOLEAN DEFAULT FALSE) AS
  BEGIN 
    dbms_aq_monitor_util.get_all_queues_reports(inst_number, 
               final_collect, single_logfile);
  END get_all_queues_reports;


  -- Generate a report for subscr-registration stats.
  -- (valid for 11.1/11.2)  
  -- 
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_subreg_report(final_collect IN BOOLEAN DEFAULT TRUE) AS
  BEGIN
    dbms_aq_monitor_util.get_subreg_report(final_collect);
  END get_subreg_report;
  
  --Check that the components of a queue table are VALID
  
  PROCEDURE hcheck_queue_table(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.hcheck_queue_table(owner_name, queue_table);
  END hcheck_queue_table;
 
  --Check for logically corrupted messages in a particular queue table
  --and provide suggestions for resolution

  PROCEDURE hcheck_queue_table_messages(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.hcheck_queue_table_messages(owner_name, queue_table);
  END hcheck_queue_table_messages;

  --Clean for logically corrupted message in a particular queue table
  PROCEDURE hcheck_clean_queue_table_msg(
  owner_name IN VARCHAR2 DEFAULT NULL,
  queue_table IN VARCHAR2 DEFAULT NULL) AS
  BEGIN
    dbms_aq_monitor_util.hcheck_clean_queue_table_msg(owner_name, queue_table);
  END hcheck_clean_queue_table_msg;
  --Check relevant AQ init.ora parameters are set correctly 

  PROCEDURE hcheck_parameters AS
  BEGIN
    dbms_aq_monitor_util.hcheck_parameters;
  END hcheck_parameters;

  --Check that the AQ metadata in system.aq$_queues and system.aq$queue_tables is consistent with what is in obj$  
  
  PROCEDURE hcheck_dictionary AS
  BEGIN
    dbms_aq_monitor_util.hcheck_dictionary;
  END hcheck_dictionary;
 
  --Check for errors in dba_queue_schedules and common propagation issues
 
  PROCEDURE hcheck_propagation_sender AS
  BEGIN
    dbms_aq_monitor_util.hcheck_propagation_sender;
  END hcheck_propagation_sender;
 
  --Check for issues associated with the propagation receiver processes 
  
  PROCEDURE hcheck_propagation_receiver AS
  BEGIN
    dbms_aq_monitor_util.hcheck_propagation_receiver;
  END hcheck_propagation_receiver;
 
  --Check for issues associated with qmon processes
  
  PROCEDURE hcheck_qmon AS
  BEGIN
    dbms_aq_monitor_util.hcheck_qmon;
  END hcheck_qmon;
  
  --Check for issues associated with notification
 
  PROCEDURE hcheck_notification AS
  BEGIN
    dbms_aq_monitor_util.hcheck_notification;
  END hcheck_notification;
  
  --Execute all AQ healthcheck procedures
  
  PROCEDURE hcheck_all AS
  BEGIN
    dbms_aq_monitor_util.hcheck_all;
  END hcheck_all;

END dbms_aq_monitor;
/
show errors;



CREATE OR REPLACE PACKAGE BODY dbms_aq_monitor_util AS    

  -----------------------------------------------------------------------
  -- CONST VARIABLES THAT DO NOT ALLOW MODIFICATION
  ----------------------------------------------------------------------- 

  FIVE_MINUTES                   PLS_INTEGER := 300; -- seconds in 5 minutes
  ONE_MINUTE                     PLS_INTEGER := 60;  -- seconds in 1 minute
  INFINITE_TIME                  PLS_INTEGER := 2147483647; -- infinity 

  -- Log filename for all-queues summary report.
  AQMON_REPORT_FNAME             VARCHAR2(1000) := 'aqmon_report.log';  
  AQMON_REPORT_OUTPUT            UTL_FILE.FILE_TYPE; 

  -- Log filename for the AQ healthcheck report.
  AQMON_HCREPORT_FNAME             VARCHAR2(1000) := 'aqmon_healthcheck_report.log';
  AQMON_HCREPORT_OUTPUT            UTL_FILE.FILE_TYPE; 

  -- Log filename for all-queues detailed report. 
  -- This is used when parameter single_logfile in get_queue_report or 
  -- get_all_queues_reports is set to true.
  -- Otherwise, the monitor will generate a unique log file as a specific
  -- queue's detailed report. The file is named as 'aqmon_queue_AB_XXYY.log',
  -- where AB is the instance number and XXYY is the queue id. 
  AQMON_ALLQUEUES_REP_FNAME      VARCHAR2(100) := 'aqmon_all_queues.log';


  -- Log filename for subscr-registration stats
  AQMON_SUBREG_FNAME             VARCHAR2(100) := 'aqmon_subreg.log';
  AQMON_SUBREG_OUTPUT            UTL_FILE.FILE_TYPE;


  -- Log directory name 
  -- NOTE: This is not the directory path.
  AQMON_LOGDIR                   VARCHAR2(30) := 'AQMON_LOGDIR'; 
  
  -- Number of top SQLs that are displayed.
  AQMON_TOP_N_SQL                PLS_INTEGER := 10;   

  -- Number of top segments that are displayed.
  AQMON_TOP_N_SEGSTAT            PLS_INTEGER := 5;

  -- Number of top queues that are displayed in the report.
  AQMON_TOP_N_QUEUE              PLS_INTEGER := 10;

  -- For a certain metric, only SQLs whose value is higher than 
  -- AQMON_TOP_PCT_SQL are displayed. By setting it to 0, we 
  -- neglect this criterion.
  AQMON_TOP_PCT_SQL              PLS_INTEGER := 0;    

  -- Max length for a queue owner
  AQMON_MAX_OWNER_LENGTH         PLS_INTEGER := 30;   

  -- Monitor status const
  AQMON_STATUS_NULL              PLS_INTEGER := 0; -- No monitor process
  AQMON_STATUS_START             PLS_INTEGER := 1; -- Monitor is started
  AQMON_STATUS_STOP              PLS_INTEGER := 2; -- Monitor is stopped


  -- Monitor jobnames
  AQMON_JOB_START_MONITOR        VARCHAR2(100) := 'AQMON_JOB_START_MONITOR';
  AQMON_JOB_STOP_MONITOR         VARCHAR2(100) := 'AQMON_JOB_STOP_MONITOR';
  AQMON_JOB_COLLECT_CONFIG       VARCHAR2(100) := 'AQMON_JOB_COLLECT_CONFIG';
  AQMON_JOB_COLLECT_RUNTIME      VARCHAR2(100) := 'AQMON_JOB_COLLECT_RUNTIME';
  AQMON_JOB_DUMP_AWR             VARCHAR2(100) := 'AQMON_JOB_DUMP_AWR';


  -----------------------------------------------------------------------
  -- PACKAGE PROCEDURES AND FUNCTIONS
  -- 
  -- NOTE: Procedures/functions starting with prvt_ are private
  --       procedures/functions. Others are public ones.
  -----------------------------------------------------------------------


  -----------------------------------------------------------------------
  -- Part 0: internal utility procedures/functions
  -----------------------------------------------------------------------

  -- Display information to users.
  PROCEDURE prvt_echo_line(str IN VARCHAR2) AS    
  BEGIN
    IF str IS NOT NULL THEN
      DBMS_OUTPUT.PUT_LINE(str);
    END IF;
  END prvt_echo_line;



  -- Display error info to users.
  --    Todo: may also raise an error
  PROCEDURE prvt_echo_error(str IN VARCHAR2) AS
  BEGIN
    IF str IS NOT NULL THEN
      DBMS_OUTPUT.PUT_LINE('ERROR: ' || str);
    END IF;
  END prvt_echo_error;



  -- Set a parameter value of NUMBER type in table aqmon_params_tab
  PROCEDURE prvt_set_param(parameter_name  IN VARCHAR2,
                           parameter_value IN NUMBER) AS
    stmt     VARCHAR2(200);
  BEGIN
    IF parameter_name IS NULL OR parameter_value IS NULL THEN
      RETURN;
    END IF;

    -- parameter check
    CASE 
      WHEN UPPER(parameter_name) = 'APT_COLLECT_CONFIG_FREQ' OR
           UPPER(parameter_name) = 'APT_COLLECT_RUNTIME_FREQ' OR
           UPPER(parameter_name) = 'APT_COLLECT_ACCUM_FREQ' OR
           UPPER(parameter_name) = 'APT_DUMP_AWR_FREQ' THEN
           IF parameter_value <= 0 THEN
             prvt_echo_error('Frequency value must be positive.');
             RETURN;
           END IF;       
      WHEN UPPER(parameter_name) = 'APT_DUMP_PLOTTING_FILE' THEN 
           IF parameter_value != 0 AND parameter_value != 1 THEN
             prvt_echo_error('Error parameter for dumping plotting file.');
             RETURN;
           END IF;       
      WHEN UPPER(parameter_name) = 'APT_SHARED_SUBSQL_LENGTH' OR 
           UPPER(parameter_name) = 'APT_SHARED_MEMORY_BOUND' THEN
           IF parameter_value <= 0 THEN
             prvt_echo_error('Parameter value must be positive.');
             RETURN;
           END IF;      
      ELSE NULL; 
    END CASE;      

    -- update table
    stmt := 'DELETE FROM aqmon_params_tab where PARAM_NAME=:1';
    EXECUTE IMMEDIATE stmt USING UPPER(parameter_name);

    stmt := 'INSERT INTO aqmon_params_tab (PARAM_NAME, PARAM_VALUE) ' ||
            ' VALUES(:1, :2)';
    EXECUTE IMMEDIATE stmt USING UPPER(parameter_name), parameter_value;

    EXECUTE IMMEDIATE 'COMMIT';
  END prvt_set_param;



  -- Set a parameter value of TIMESTAMP type in table aqmon_params_tab
  PROCEDURE prvt_set_param2(parameter_name  IN VARCHAR2,
                            parameter_value IN TIMESTAMP) AS
    stmt     VARCHAR2(200);
  BEGIN
    IF parameter_name IS NULL OR parameter_value IS NULL THEN
      RETURN;
    END IF;

    -- update table
    stmt := 'DELETE FROM aqmon_params_tab where PARAM_NAME=:1';
    EXECUTE IMMEDIATE stmt USING UPPER(parameter_name);

    stmt := 'INSERT INTO aqmon_params_tab(PARAM_NAME, PARAM_VALUE2) ' ||
            'VALUES(:1, :2)';
    EXECUTE IMMEDIATE stmt USING UPPER(parameter_name), parameter_value;

    EXECUTE IMMEDIATE 'COMMIT';
  END prvt_set_param2;



  -- Get a parameter value of NUMBER type from table aqmon_params_tab
  FUNCTION prvt_get_param(parameter_name IN VARCHAR2) 
  RETURN NUMBER IS
    result     NUMBER;
  BEGIN 
    IF parameter_name IS NULL THEN
      RETURN NULL;
    END IF;

    BEGIN
      SELECT PARAM_VALUE INTO result
        FROM aqmon_params_tab
       WHERE PARAM_NAME = UPPER(parameter_name);
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        CASE UPPER(parameter_name) 
          -- set default values
          WHEN 'APT_COLLECT_CONFIG_FREQ'  THEN result := INFINITE_TIME;
          WHEN 'APT_COLLECT_RUNTIME_FREQ' THEN result := ONE_MINUTE;
          WHEN 'APT_COLLECT_ACCUM_FREQ'   THEN result := FIVE_MINUTES;
          WHEN 'APT_DUMP_AWR_FREQ'        THEN result := FIVE_MINUTES;
          WHEN 'APT_MONITOR_STATUS'       THEN result := AQMON_STATUS_NULL;
          WHEN 'APT_BEGIN_SNAPSHOT_ID'    THEN result := -1;
          WHEN 'APT_END_SNAPSHOT_ID'      THEN result := -1;
          WHEN 'APT_DATABASE_VERSION'     THEN result := 0;
          WHEN 'APT_DUMP_PLOTTING_FILE'   THEN result := 0;
          WHEN 'APT_SHARED_SUBSQL_LENGTH' THEN result := 150;
          WHEN 'APT_SHARED_MEMORY_BOUND'  THEN result := 1;
          ELSE                                 result := NULL;
        END CASE;
    END;

    RETURN result;
  END prvt_get_param;



  -- Get a parameter value of TIMESTAMP type from table aqmon_params_tab
  FUNCTION prvt_get_param2(parameter_name IN VARCHAR2) 
  RETURN TIMESTAMP IS
    result     TIMESTAMP;
  BEGIN 
    IF parameter_name IS NULL THEN
      RETURN NULL;
    END IF;

    BEGIN
      SELECT PARAM_VALUE2 INTO result
        FROM aqmon_params_tab
       WHERE PARAM_NAME = UPPER(parameter_name);
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        CASE UPPER(parameter_name) 
          -- set default values
          WHEN 'APT_BEGIN_MONITOR_TM'    THEN  result := NULL;
          WHEN 'APT_END_MONITOR_TM'      THEN  result := NULL;
          ELSE                                 result := NULL;
        END CASE;
    END;

    RETURN result;
  END prvt_get_param2;



  -- Get database version
  PROCEDURE prvt_get_database_version AS
    db_version    VARCHAR2(500);
  BEGIN
    BEGIN 
      SELECT banner INTO db_version FROM v$version
       WHERE UPPER(banner) LIKE 'ORACLE%';
    EXCEPTION
      WHEN OTHERS THEN
        db_version := NULL;
    END;
    
    IF db_version IS NOT NULL THEN
      IF (instr(UPPER(db_version), 'RELEASE 10.2') > 0) THEN
        prvt_set_param('APT_DATABASE_VERSION', 10.2);
        RETURN;
      ELSIF (instr(UPPER(db_version), 'RELEASE 11.1') > 0) THEN
        prvt_set_param('APT_DATABASE_VERSION', 11.1);
        RETURN;
      ELSIF (instr(UPPER(db_version), 'RELEASE 11.2') > 0) THEN
        prvt_set_param('APT_DATABASE_VERSION', 11.2);
        RETURN;
      ELSIF (instr(UPPER(db_version), 'RELEASE 12.1') > 0) THEN
        prvt_set_param('APT_DATABASE_VERSION', 12.1);
        RETURN;
      END IF;
    END IF;

    prvt_set_param('APT_DATABASE_VERSION', 0);
    
  END prvt_get_database_version;



  -- Check the monitor configuration.
  FUNCTION prvt_check_monitor_config RETURN BOOLEAN IS
    cnt            NUMBER;
    error          BOOLEAN;
    test_file      UTL_FILE.FILE_TYPE;
  BEGIN
    -- 1. Check queue owners in the monitoring list.
    SELECT count(QUEUE_OWNER) INTO cnt FROM aqmon_queue_owners_tab;
    IF cnt = 0 THEN
      prvt_echo_error('No queue owner in the monitoring list.');
      RETURN FALSE;
    END IF;

    -- 2. Check database version
    prvt_get_database_version;
    cnt := prvt_get_param('APT_DATABASE_VERSION');
    IF cnt IS NULL or (cnt!=10.2 AND cnt!=11.1 AND cnt!=11.2 AND cnt!=12.1) THEN
      prvt_echo_error('Only Oracle Database 10.2/11.1/11.2/12.1 is supported.');
      RETURN FALSE;
    END IF;

    -- 3. Check the log directory and report name.
    error := FALSE;
    BEGIN    
      test_file := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_REPORT_FNAME, 'w');
      UTL_FILE.FCLOSE(test_file);      
    EXCEPTION
      WHEN OTHERS THEN
        -- fail to open or write to log file 
        prvt_echo_error('Cannot create log file in the log directory.');
        error := TRUE;
    END;  

    IF error THEN
      RETURN FALSE;
    END IF;

    -- 4. Check monitoring frequencies.
    cnt := prvt_get_param('APT_COLLECT_RUNTIME_FREQ');
    IF cnt IS NULL OR cnt <= 0 THEN
      prvt_echo_error('Frequency for collecting queue runtime '|| 
                      'information must be positive.');
      RETURN FALSE;
    END IF;


    cnt := prvt_get_param('APT_COLLECT_ACCUM_FREQ');
    IF cnt IS NULL OR cnt <= 0 THEN
      prvt_echo_error('Frequency for dumping accumulative '|| 
                      'statistics must be positive.');
      RETURN FALSE;
    END IF;


    cnt := prvt_get_param('APT_COLLECT_CONFIG_FREQ');
    IF cnt IS NULL OR cnt <= 0 THEN
      prvt_echo_error('Frequency for collecting queue configuration '|| 
                      'information must be positive.');
      RETURN FALSE;
    END IF;
 

    cnt := prvt_get_param('APT_DUMP_AWR_FREQ');
    IF cnt IS NULL OR cnt <= 0 THEN
      prvt_echo_error('Frequency for generating AWR reports '|| 
                      'must be positive.');
      RETURN FALSE;
    END IF;

    -- if all the above checks pass
    RETURN TRUE;

  END prvt_check_monitor_config;



  -- Stop and drop a job
  PROCEDURE prvt_stop_job(job_name IN VARCHAR2) AS
  BEGIN
    IF job_name IS NULL THEN
      RETURN;
    END IF;

    BEGIN dbms_scheduler.disable(job_name, force=>true); 
    EXCEPTION WHEN OTHERS THEN NULL;
    END;

    BEGIN dbms_scheduler.stop_job(job_name, force=>true);
    EXCEPTION WHEN OTHERS THEN NULL;
    END;

    BEGIN dbms_scheduler.drop_job(job_name); 
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
  END prvt_stop_job;



  -- Generate an interval string for DBMS_SCHEDULER
  -- (DBMS_SCHEDULER accepts only integer in its repeat_interval.)
  -- 
  -- PARAM  : sec - frequency value in seconds. 
  FUNCTION prvt_get_repeat_interval(sec IN NUMBER) RETURN VARCHAR2 IS
  BEGIN
    IF sec IS NULL OR sec <= 0 THEN
      RETURN NULL;
    END IF;

    IF sec < 999 THEN
      RETURN 'FREQ=SECONDLY; INTERVAL=' || round(sec);
    ELSIF sec < 999*60 THEN
      RETURN 'FREQ=MINUTELY; INTERVAL=' || round(sec/60);
    ELSIF sec < 999*60*60 THEN
      RETURN 'FREQ=HOURLY; INTERVAL=' || round(sec/60/60);
    ELSIF sec < 999*60*60*24 THEN
      RETURN 'FREQ=DAYLY; INTERVAL=' || round(sec/60/60/24);
    ELSIF sec < 999.0*60*60*24*30 THEN
      RETURN 'FREQ=MONTHLY; INTERVAL=' || round(sec/60/60/24/30);
    ELSE
      RETURN NULL;
    END IF;

  END prvt_get_repeat_interval;
  


  -- Replace special characters in a SQL name and enquote the name. 
  FUNCTION prvt_enquote_str(sql_name IN VARCHAR2) RETURN VARCHAR2 IS
  BEGIN
    IF sql_name IS NULL THEN
      RETURN NULL;
    ELSE
      RETURN dbms_assert.enquote_literal(replace(sql_name, '''', ''''''));
    END IF;
  END prvt_enquote_str;



  -----------------------------------------------------------------------
  -- Part 1: set/get monitor configuration parameters
  -----------------------------------------------------------------------

  -- Add a queue owner to the monitoring list. The owner name will be 
  -- automatically capitalized. All queues belonging to that owner will 
  -- be monitored.
  PROCEDURE add_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL) AS
    cnt     NUMBER  := 0;
    stmt    VARCHAR2(200);
  BEGIN
    IF owner_name IS NULL THEN
      prvt_echo_error('Cannot add a NULL queue owner.');
      RETURN;
    END IF;

    -- check owner_name length
    IF LENGTH(owner_name) > AQMON_MAX_OWNER_LENGTH THEN
      prvt_echo_error('Length of a queue owner cannot exceed ' ||
                      AQMON_MAX_OWNER_LENGTH || '.');
      RETURN;  
    END IF;  

    -- update AQMON_QUEUE_OWNERS_TAB with the given owner info.
    SELECT count(*) INTO cnt FROM aqmon_queue_owners_tab 
     WHERE QUEUE_OWNER = UPPER(owner_name);

    IF cnt > 0 THEN
      prvt_echo_line('Queue owner ' || UPPER(owner_name)
                     || ' is already in the monitoring list.');   
      RETURN;
    ELSE 
      stmt := 'INSERT INTO aqmon_queue_owners_tab VALUES (:1)';
      EXECUTE IMMEDIATE stmt USING UPPER(owner_name);
      EXECUTE IMMEDIATE 'COMMIT';
    END IF;

    SELECT count(*) INTO cnt FROM aqmon_queue_owners_tab 
     WHERE QUEUE_OWNER = UPPER(owner_name);

    IF cnt = 0 THEN
      prvt_echo_error('Cannot add queue owner ' || UPPER(owner_name)
                     || ' to the monitoring list.');   
    ELSE
      prvt_echo_line('Queue owner ' || UPPER(owner_name)
                     || ' is added to the monitoring list.');   
    END IF;

  END add_queue_owner;



  -- Remove a queue owner from the monitoring list. The owner name will be 
  -- automatically capitalized.
  PROCEDURE remove_queue_owner(owner_name IN VARCHAR2 DEFAULT NULL) AS 
    cnt     NUMBER  := 0;
    stmt    VARCHAR2(200);
  BEGIN
    IF owner_name IS NULL THEN
      prvt_echo_error('Cannot remove a NULL queue owner.');
      RETURN;
    END IF;

    -- check owner length
    IF LENGTH(owner_name) > AQMON_MAX_OWNER_LENGTH THEN
      prvt_echo_error('Length of a queue owner cannot exceed ' ||
                      AQMON_MAX_OWNER_LENGTH || '.');
      RETURN;  
    END IF;  

    SELECT count(*) INTO cnt FROM aqmon_queue_owners_tab 
     WHERE QUEUE_OWNER = UPPER(owner_name);

    IF cnt = 0 THEN
      prvt_echo_line('Queue owner ' || UPPER(owner_name)
                     || ' is not in the monitoring list.');
      RETURN;
    END IF;

    stmt := 'DELETE FROM aqmon_queue_owners_tab WHERE QUEUE_OWNER=:1';
    EXECUTE IMMEDIATE stmt USING UPPER(owner_name);
    EXECUTE IMMEDIATE 'COMMIT';

    SELECT count(*) INTO cnt FROM aqmon_queue_owners_tab 
     WHERE QUEUE_OWNER = UPPER(owner_name);

    IF cnt = 0 THEN
      prvt_echo_line('Queue owner ' || UPPER(owner_name)
                       || ' is removed from the monitoring list.');
    ELSE
      prvt_echo_error('Cannot remove queue owner ' || UPPER(owner_name)
                     || ' from the monitoring list.');     
    END IF;

  END remove_queue_owner;



  -- Set directory for dumping log files.
  --
  -- PARAM  : overwrite_file - whether to overwrite existing file if file
  --                           'aqmon_report.log' already exists in the log 
  --                           directory.
  
  
  PROCEDURE set_log_dir(dir_name       IN VARCHAR2 DEFAULT NULL,
                        overwrite_file IN BOOLEAN DEFAULT FALSE) AS
    test_file1    UTL_FILE.FILE_TYPE;
	test_file2    UTL_FILE.FILE_TYPE;
    old_path     VARCHAR2(4000);
    new_path     VARCHAR2(4000);
    error        BOOLEAN := FALSE;
  BEGIN
    IF dir_name IS NULL THEN
      prvt_echo_error('Log directory cannot be NULL.');
      RETURN;
    END IF;

    IF overwrite_file IS NULL THEN
      prvt_echo_error('Overwrite_file parameter cannot be NULL.');
      RETURN;
    END IF;

    BEGIN    
      SELECT DIRECTORY_PATH INTO old_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        old_path := NULL;
    END;

    EXECUTE IMMEDIATE 'CREATE OR REPLACE DIRECTORY ' || AQMON_LOGDIR ||
                      ' AS ' || prvt_enquote_str(dir_name);

    BEGIN
      SELECT DIRECTORY_PATH INTO new_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        new_path := NULL;
    END;

    -- Check whether the report files already exist in the log directory
	-- If they do not exist then an exception will be raised but we just 
	-- absorb that and raise any issues if we cannot write 
    BEGIN
      test_file1 := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_REPORT_FNAME, 'r');
	  test_file2 := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'r');
      IF UTL_FILE.IS_OPEN(test_file1) AND UTL_FILE.IS_OPEN(test_file2) AND NOT overwrite_file THEN
        prvt_echo_error('File ' || AQMON_REPORT_FNAME || ' already ' ||
                        'exists in directory ' || new_path || '.');
		prvt_echo_error('File ' || AQMON_HCREPORT_FNAME || ' already ' ||
                        'exists in directory ' || new_path || '.');						
        prvt_echo_line('Use set_log_dir(' || prvt_enquote_str(dir_name) || 
                       ', TRUE) to overwrite the existing files.');
        error := TRUE;
      END IF;
      UTL_FILE.FCLOSE(test_file1);
	  UTL_FILE.FCLOSE(test_file2);
    EXCEPTION 
      WHEN OTHERS THEN NULL;	  
    END;


    IF NOT error THEN 
      -- Check whether the new log directory is writable
      BEGIN
        test_file1 := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_REPORT_FNAME, 'w');
        UTL_FILE.FCLOSE(test_file1);
		test_file2 := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'w');
        UTL_FILE.FCLOSE(test_file2);
		error := FALSE;
      EXCEPTION
        WHEN OTHERS THEN
          -- fail to open or write to log file 
          prvt_echo_error('Cannot create log files in directory '||
                          new_path || '.');
          error := TRUE;
      END;
    END IF;

    IF error THEN 
      -- restore to previous log directory
      IF old_path IS NOT NULL THEN 
        EXECUTE IMMEDIATE 'CREATE OR REPLACE DIRECTORY ' || AQMON_LOGDIR ||
                          ' AS ' || prvt_enquote_str(old_path);
      END IF;
    ELSE
      prvt_echo_line('Set log directory to ' || new_path || '.');
    END IF;

    EXECUTE IMMEDIATE 'COMMIT';

  END set_log_dir;


  -- Set monitoring parameters
  --
  -- PARAM  : A list of param_names is as follows.
  -- 
  ---------------------------------------------------------------------------
  --Param Name            Default Value  Usage
  ---------------------------------------------------------------------------
  --CONFIG_COLLECT_FREQ   300(seconds)   frequency for collecting
  --                                     queue configuration info
  --RUNTIME_COLLECT_FREQ  60(seconds)    frequency for collecting 
  --                                     queue runtime stats
  --AWR_DUMP_FREQ         300(seconds)   frequency for generating AWR 
  --                                     report
  --ACCUM_DUMP_FREQ       300(seconds)   frequency for dumping 
  --                                     accumulative stats in log file
  --FILE_PLOT             0              whether dump per-queue stats file
  --                                     for gnuplot. Value could be 0 or 1.
  --SHARED_SUBSQL_LENGTH  150  
  --SHARED_MEMEORY_BOUND  1 (MByte)      The monitor will display SQLs that
  --                                     are similar (i.e., their share the
  --                                     same first $APT_SHARED_SUBSQL_LENGTH$
  --                                     characters) and in total consume more
  --                                     than $APT_SHARED_MEMORY_BOUND$ MBytes
  --                                     memory
  ---------------------------------------------------------------------------
  PROCEDURE set_monitoring_param(param_name  IN VARCHAR2 DEFAULT NULL, 
                                 param_value IN NUMBER DEFAULT NULL) AS

    num_maximum    NUMBER  := 999.0*60*60*24*30;
    succ           BOOLEAN;

    FUNCTION prvt_set_monitoring_freq(freq_name   IN VARCHAR2,
                                      freq_value  IN NUMBER,
                                      lower_bound IN NUMBER) 
    RETURN BOOLEAN IS
    BEGIN
      IF freq_value <= 0 THEN 
        prvt_echo_error('Frequency value must be positive.');     
        RETURN FALSE;
      ELSIF freq_value >= num_maximum THEN
        prvt_echo_error('Frequency value is too large.');
        RETURN FALSE;
      END IF;

      prvt_set_param(freq_name, round(freq_value));

      IF freq_value <= lower_bound THEN
        prvt_echo_line('Frequency value could be larger to ' || 
                       'save system resource.');
      END IF;

      RETURN TRUE;
    END prvt_set_monitoring_freq;  

  BEGIN     
    IF param_name IS NULL THEN
      prvt_echo_error('Parameter name cannot be NULL.');
      RETURN;
    END IF;

    IF param_value IS NULL THEN
      prvt_echo_error('Parameter value cannot be NULL.');
      RETURN;
    END IF;


    succ := TRUE;
    CASE UPPER(param_name)
      WHEN 'CONFIG_COLLECT_FREQ' THEN
        succ := prvt_set_monitoring_freq('APT_COLLECT_CONFIG_FREQ', 
                                         param_value, ONE_MINUTE);

      WHEN 'RUNTIME_COLLECT_FREQ' THEN 
        succ := prvt_set_monitoring_freq('APT_COLLECT_RUNTIME_FREQ', 
                                         param_value, 10);

      WHEN 'AWR_DUMP_FREQ' THEN
        succ := prvt_set_monitoring_freq('APT_DUMP_AWR_FREQ', 
                                         param_value, ONE_MINUTE);

      WHEN 'ACCUM_DUMP_FREQ' THEN
        succ := prvt_set_monitoring_freq('APT_COLLECT_ACCUM_FREQ', 
                                         param_value, ONE_MINUTE);

      WHEN 'FILE_PLOT' THEN
        IF param_value != 0 AND param_value != 1 THEN 
           prvt_echo_error('Parameter value for FILE_PLOT could only be 0' ||
                           ' or 1.');
           RETURN;
        END IF;
        prvt_set_param('APT_DUMP_PLOTTING_FILE', param_value);


      WHEN 'SHARED_SUBSQL_LENGTH' THEN
        IF param_value <= 0 THEN 
          prvt_echo_error('Parameter value for SHARED_SUBSQL_LENGTH must ' ||
                          'be positive.'); 
          RETURN;
        ELSIF param_value >= num_maximum THEN
          prvt_echo_error('Parameter value is too large.');
          RETURN;        
        END IF;  
        prvt_set_param('APT_SHARED_SUBSQL_LENGTH', round(param_value));


      WHEN 'SHARED_MEMORY_BOUND' THEN
        IF param_value <= 0 THEN 
          prvt_echo_error('Parameter value for SHARED_MEMORY_BOUND must ' ||
                          'be positive.'); 
          RETURN;
        ELSIF param_value >= num_maximum THEN
          prvt_echo_error('Parameter value is too large.');
          RETURN;        
        END IF;  
        prvt_set_param('APT_SHARED_MEMORY_BOUND', param_value);


      ELSE
        prvt_echo_error('Unrecognized parameter name ' || param_name || '.');
        RETURN;
    END CASE;

    IF succ THEN 
      prvt_echo_line('Parameter value is successfully set.');
    END IF;

  END set_monitoring_param;



  -- Display monitor configuration parameters.
  PROCEDURE show_config AS
    cnt           NUMBER;
    plural        VARCHAR2(2);
    logdir_path   VARCHAR2(4000);
    str           VARCHAR2(100);
    cursor owner_cs IS 
      SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab ORDER BY QUEUE_OWNER;
  BEGIN
    
    -- display monitor status
    cnt := prvt_get_param('APT_MONITOR_STATUS');
    IF cnt IS NOT NULL THEN
      CASE cnt
        WHEN AQMON_STATUS_NULL  THEN str := 'Monitor is not started.';
        WHEN AQMON_STATUS_START THEN str := 'Monitor is running.';
        WHEN AQMON_STATUS_STOP  THEN str := 'Monitor is stopped.';
        ELSE                         str := 'Unknown status.';
      END CASE;
    ELSE
      str := 'Unknown status.';
    END IF;
    prvt_echo_line('Monitor Status   : ' || str);
    

    -- display database version
    prvt_get_database_version;
    cnt := prvt_get_param('APT_DATABASE_VERSION');
    IF cnt IS NOT NULL THEN
      CASE cnt
        WHEN 10.2 THEN str := '10.2.';
        WHEN 11.1 THEN str := '11.1.';
        WHEN 11.2 THEN str := '11.2.';
        WHEN 12.1 THEN str := '12.1.';
        ELSE          str := 'Unknown version.';
      END CASE;
    ELSE
      str := 'Unknown version.';
    END IF;
    prvt_echo_line('Database version : ' || str);


    -- display owners in the monitoring list
    SELECT count(QUEUE_OWNER) INTO cnt
      FROM aqmon_queue_owners_tab;

    IF cnt = 0 THEN
      prvt_echo_line('No queue owner in the monitoring list.');
    ELSE
      IF cnt > 1 THEN  plural := 's';
      ELSE             plural := NULL;
      END IF;

      prvt_echo_line(cnt || ' queue owner' || plural 
                   || ' in the monitoring list:');
      prvt_echo_line('-------------');
      FOR i IN owner_cs LOOP
        prvt_echo_line('    ' || i.QUEUE_OWNER);
      END LOOP;
      prvt_echo_line('-------------');
    END IF;

    -- display log directory and files info
    BEGIN
      SELECT DIRECTORY_PATH INTO logdir_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        logdir_path := NULL;
    END;

    prvt_echo_line('Log directory : ' || logdir_path);
    prvt_echo_line('Summary report filename : ' || AQMON_REPORT_FNAME);


    -- display frequency info
    cnt := prvt_get_param('APT_COLLECT_RUNTIME_FREQ');
    IF cnt IS NOT NULL THEN   
      IF cnt > 1 THEN plural := 's';
      ELSE            plural := NULL;    
      END IF;
      prvt_echo_line('Runtime snapshots are taken every ' ||
                     cnt || ' second' || plural || '.');
    END IF;


    cnt := prvt_get_param('APT_COLLECT_ACCUM_FREQ');
    IF cnt IS NOT NULL THEN   
      IF cnt > 1 THEN plural := 's';
      ELSE            plural := NULL;    
      END IF;
      prvt_echo_line('Accumulative statistics are calculated every ' ||
                     cnt || ' second' || plural || '.');
    END IF;


    cnt := prvt_get_param('APT_COLLECT_CONFIG_FREQ');
    IF cnt IS NOT NULL THEN     
      IF cnt > 1 THEN plural := 's';
      ELSE            plural := NULL;    
      END IF;
      IF cnt = INFINITE_TIME THEN
        prvt_echo_line('Queue configuation statistics are taken at the ' ||
                       'monitor starting time (once).');
      ELSE
        prvt_echo_line('Queue configuation statistics are taken every ' ||
                       cnt || ' second' || plural || '.');
      END IF;
    END IF;


    cnt := prvt_get_param('APT_DUMP_AWR_FREQ');
    IF cnt IS NOT NULL THEN   
      IF cnt > 1 THEN plural := 's';
      ELSE            plural := NULL;    
      END IF;
      prvt_echo_line('AWR reports are taken every ' ||
                     cnt || ' second' || plural || '.');
    END IF;


    cnt := prvt_get_param('APT_DUMP_PLOTTING_FILE');
    IF cnt IS NOT NULL THEN   
      IF cnt = 1 THEN str := 'Yes';
      ELSE            str := 'No';    
      END IF;
      prvt_echo_line('Dump per-queue file for plotting: ' || str);      
    END IF;


    cnt := prvt_get_param('APT_SHARED_SUBSQL_LENGTH');
    IF cnt IS NOT NULL THEN   
      prvt_echo_line('Shared subsql length : ' || cnt);      
    END IF;


    cnt := prvt_get_param('APT_SHARED_MEMORY_BOUND');
    IF cnt IS NOT NULL THEN   
      IF cnt > 1 THEN plural := 's';
      ELSE            plural := NULL;    
      END IF;
      prvt_echo_line('Shared memory bound  : ' || cnt || ' MByte' || plural);
    END IF;


  END show_config;


 
  -----------------------------------------------------------------------
  -- Part 2: procedures called by DBMS_SCHEDULER
  ----------------------------------------------------------------------- 

  -- EXPORT PROCEDURE: collect queue config info and dump it into 
  --                   corresponding tables.
  -- NOTE: This procedure will be periodically called by DBMS_SCHEDULER. In
  --       that call, it does not recognize package variable values that 
  --       are changed in runtime.
  PROCEDURE collect_config AS     
    cnt            NUMBER;
    stmt           VARCHAR2(200);
 
    CURSOR coco_conf1 IS
      SELECT a.QID   QUEUE_ID,
             a.NAME   QUEUE_NAME,
             a.QUEUE_TABLE, 
             a.OWNER   QUEUE_OWNER,
             a.QUEUE_TYPE,
             b.TYPE   DATA_TYPE,
             b.OBJECT_TYPE,
             b.SORT_ORDER,
             b.RECIPIENTS, 
             b.MESSAGE_GROUPING
        FROM sys.dba_queues a, sys.dba_queue_tables b
       WHERE a.OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND a.QUEUE_TABLE = b.QUEUE_TABLE    
         AND a.QUEUE_TYPE = 'NORMAL_QUEUE';

    CURSOR coco_conf2 IS
      SELECT QUEUE_NAME, QUEUE_TABLE, CONSUMER_NAME, TRANSFORMATION,
             DELIVERY_MODE, QUEUE_TO_QUEUE, OWNER, ADDRESS
        FROM sys.dba_queue_subscribers
       WHERE OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab);

  BEGIN    
    -- update the queue config table
    FOR i in coco_conf1 LOOP
      SELECT count(QUEUE_ID) INTO cnt FROM aqmon_queue_config_tab
       WHERE QUEUE_ID = i.QUEUE_ID;

      IF cnt = 0 THEN
        stmt := 'INSERT INTO aqmon_queue_config_tab VALUES ' ||
                '  (:1, :2, :3, :4, :5, :6, :7, :8, :9, :10)';
        EXECUTE IMMEDIATE stmt USING i.QUEUE_ID, i.QUEUE_NAME, i.QUEUE_TABLE,
          i.QUEUE_OWNER, i.QUEUE_TYPE, i.DATA_TYPE, i.OBJECT_TYPE,
          i.SORT_ORDER, i.RECIPIENTS, i.MESSAGE_GROUPING;
      END IF;
    END LOOP;


    -- update the subscriber config table
    FOR i in coco_conf2 LOOP
      SELECT count(CONSUMER_NAME) INTO cnt 
        FROM aqmon_subscriber_config_tab
       WHERE QUEUE_NAME    = i.QUEUE_NAME
         AND QUEUE_TABLE   = i.QUEUE_TABLE
         AND CONSUMER_NAME = i.CONSUMER_NAME
         AND ADDRESS       = i.ADDRESS
         AND OWNER         = i.OWNER;

      IF cnt = 0 THEN
        stmt := 'INSERT INTO aqmon_subscriber_config_tab VALUES ' ||
                '  (:1, :2, :3, :4, :5, :6, :7, :8)';
        EXECUTE IMMEDIATE stmt USING i.QUEUE_NAME, i.QUEUE_TABLE, 
          i.CONSUMER_NAME, i.TRANSFORMATION, i.DELIVERY_MODE,
          i.QUEUE_TO_QUEUE, i.OWNER, i.ADDRESS;
      END IF;
    END LOOP;

    EXECUTE IMMEDIATE 'COMMIT';
  END collect_config;



  -- EXPORT PROCEDURE: collect queue runtime stats and dump it into 
  --                   corresponding tables.
  -- NOTE: This procedure will be periodically called by DBMS_SCHEDULER. In
  --       that call, it does not recognize package variable values that 
  --       are changed in runtime.
  PROCEDURE collect_runtime AS
    mytimestamp  TIMESTAMP;
    seq_num      NUMBER;
    v            VARCHAR2(4000);
    cnt          NUMBER;
    db_version   NUMBER;
  BEGIN 
    SELECT systimestamp INTO mytimestamp FROM DUAL;
    SELECT aqmon_monitor_seq.nextval INTO seq_num FROM DUAL;

    db_version := prvt_get_param('APT_DATABASE_VERSION');

    IF db_version IS NOT NULL AND db_version >= 11.1 THEN

      -- 1. collect from gv$persistent_subscribers
      -- NOTE: first collect from gv$persistent_subscribers, and then from
      --       gv$persistent_queues. Some fields in gv$persistent_queues
      --       are not timely updated.
      EXECUTE IMMEDIATE 'SELECT count(*) FROM gv$persistent_subscribers ' ||
        ' WHERE QUEUE_SCHEMA IS NOT NULL AND QUEUE_SCHEMA IN ' ||
        ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)' INTO cnt;
      IF cnt = 0 THEN
        v := 'INSERT INTO aqmon_psubscribers_tab (memseq, monitor_time,'||
             ' flag) VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
      ELSE
        v := 'INSERT INTO aqmon_psubscribers_tab SELECT a.*,' || seq_num 
             || ',''' || mytimestamp || ''', 0 FROM gv$persistent_subscribers'
             || ' a WHERE a.QUEUE_SCHEMA IS NOT NULL AND a.QUEUE_SCHEMA IN '
             || ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)';
      END IF;
      EXECUTE IMMEDIATE v;        


      -- 2. collect from gv$persistent_queues
      EXECUTE IMMEDIATE 'SELECT count(*) FROM gv$persistent_queues ' ||
        ' WHERE QUEUE_SCHEMA IS NOT NULL AND QUEUE_SCHEMA IN ' ||
        ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)' INTO cnt;
      IF cnt = 0 THEN
        v := 'INSERT INTO aqmon_pqueues_tab (memseq, monitor_time, flag) ' ||
             'VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
      ELSE
        v := 'INSERT INTO aqmon_pqueues_tab SELECT a.*, ' || seq_num  || ','''
             || mytimestamp || ''', 0 FROM gv$persistent_queues a' 
             || ' WHERE a.QUEUE_SCHEMA IS NOT NULL AND a.QUEUE_SCHEMA IN '
             || ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)';
      END IF;
      EXECUTE IMMEDIATE v;        
    

      -- 3. collect from gv$subscr_registration_stats
      EXECUTE IMMEDIATE 'SELECT count(*) FROM gv$subscr_registration_stats'
        INTO cnt;
      IF cnt = 0 THEN
        v := 'INSERT INTO aqmon_subreg_tab (memseq, monitor_time, flag) ' ||
             'VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
      ELSE
        v := 'INSERT INTO aqmon_subreg_tab SELECT a.*, ' || seq_num  || ','''
             || mytimestamp || ''', 0 FROM gv$subscr_registration_stats a';
      END IF;
      EXECUTE IMMEDIATE v;

    END IF;        

    -- 4. collect from gv$persistent_qmn_cache
    IF db_version IS NOT NULL AND db_version >= 11.2 THEN
      EXECUTE IMMEDIATE 'SELECT count(*) FROM gv$persistent_qmn_cache' ||
                        ' WHERE QUEUE_TABLE_ID IS NOT NULL AND' ||
                        ' QUEUE_TABLE_ID IN (SELECT DISTINCT ' ||
                        ' QUEUE_TABLE_ID FROM aqmon_pqueues_tab)' INTO cnt;
      IF cnt = 0 THEN
        v := 'INSERT INTO aqmon_pqmncache_tab (memseq, monitor_time, flag) ' ||
             'VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
      ELSE
        v := 'INSERT INTO aqmon_pqmncache_tab SELECT a.*,' || seq_num || 
             ',''' || mytimestamp || ''', 0 FROM gv$persistent_qmn_cache a' ||
             ' WHERE a.QUEUE_TABLE_ID IS NOT NULL AND a.QUEUE_TABLE_ID IN ' ||
             ' (SELECT DISTINCT QUEUE_TABLE_ID FROM aqmon_pqueues_tab)';
      END IF;
      EXECUTE IMMEDIATE v;        
    END IF;
 

    -- 5. collect from gv$buffered_subscribers
    SELECT count(*) INTO cnt FROM gv$buffered_subscribers
     WHERE QUEUE_SCHEMA IS NOT NULL 
       AND QUEUE_SCHEMA IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab);
    IF cnt = 0 THEN
      v := 'INSERT INTO aqmon_bsubscribers_tab (memseq, monitor_time, flag)'||
           'VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
    ELSE
      v := 'INSERT INTO aqmon_bsubscribers_tab SELECT a.*,' || seq_num 
           || ',''' || mytimestamp || ''', 0 FROM gv$buffered_subscribers a'
           || ' WHERE a.QUEUE_SCHEMA IS NOT NULL AND a.QUEUE_SCHEMA IN '
           || ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)';
    END IF;
    EXECUTE IMMEDIATE v;        


    -- 6. collect from gv$buffered_queues
    SELECT count(*) INTO cnt FROM gv$buffered_queues
     WHERE QUEUE_SCHEMA IS NOT NULL 
       AND QUEUE_SCHEMA IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab);
    IF cnt = 0 THEN
      v := 'INSERT INTO aqmon_bqueues_tab (memseq, monitor_time, flag) ' ||
           'VALUES (' || seq_num || ', ''' || mytimestamp || ''', 0)';
    ELSE
      v := 'INSERT INTO aqmon_bqueues_tab SELECT a.*, ' || seq_num  || ','''
           || mytimestamp || ''', 0 FROM gv$buffered_queues a' 
           || ' WHERE a.QUEUE_SCHEMA IS NOT NULL AND a.QUEUE_SCHEMA IN ' 
           || ' (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)';
    END IF;
    EXECUTE IMMEDIATE v;        

        
    EXECUTE IMMEDIATE 'COMMIT';

  END collect_runtime;
 


  -- EXPORT PROCEDURE: immediately stop the monitor
  -- NOTE: This procedure will be called by DBMS_SCHEDULER. It should not 
  --       change package variable values, as the change will not be 
  --       recognized by other sessions. Similarly, it cannot recognize
  --       changes made by other sessions.
  PROCEDURE stop_monitor_immed AS
    tmp          NUMBER;
    mytimestamp  TIMESTAMP;
  BEGIN 
    tmp := prvt_get_param('APT_MONITOR_STATUS');
    IF tmp IS NULL OR tmp = AQMON_STATUS_NULL THEN
      prvt_echo_error('Monitor is not started yet.');
      RETURN;
    END IF;

    -- take the last snapshot before stopping
    collect_config;
    collect_runtime;
    DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;

    prvt_stop_job(AQMON_JOB_COLLECT_CONFIG); 
    prvt_stop_job(AQMON_JOB_COLLECT_RUNTIME);
    prvt_stop_job(AQMON_JOB_DUMP_AWR);

    prvt_set_param('APT_MONITOR_STATUS', AQMON_STATUS_STOP);
    
    SELECT max(SNAP_ID) INTO tmp FROM sys.dba_hist_snapshot;
    IF tmp IS NULL THEN
       tmp := -1;
    END IF;
    prvt_set_param('APT_END_SNAPSHOT_ID', tmp);
    
    SELECT systimestamp INTO mytimestamp FROM DUAL;
    prvt_set_param2('APT_END_MONITOR_TM', mytimestamp);

    prvt_echo_line('Current monitor is stopped.');
  END stop_monitor_immed;



  -- EXPORT PROCEDURE: Immediately start the monitor. The monitor will clear
  --                   up all previously collected statistics. It will also 
  --                   overwrite log files in the log directory.  
  -- NOTE: This procedure will be called by DBMS_SCHEDULER. It should not 
  --       change package variable values, as the change will not be 
  --       recognized by other sessions. Similarly, it cannot recognize
  --       changes made by other sessions.
  --       Also, prvt_echo_error/prvt_echo_line may not properly print 
  --       information to the screen, as the procedure is called by 
  --       DBMS_SCHEDULER.
  PROCEDURE start_monitor_immed AS 
    jobname     VARCHAR2(200);
    freq        NUMBER;
    tmp         NUMBER;
    rept_intv   VARCHAR2(100);
    mytimestamp TIMESTAMP;
  BEGIN        
    -- check the monitor configuration parameters
    IF NOT prvt_check_monitor_config THEN
      prvt_echo_line('--');
      prvt_echo_line('Monitor is not started.');
      RETURN;
    END IF;
      
    -- stop current monitor processes if any
    prvt_set_param('APT_MONITOR_STATUS', AQMON_STATUS_START);
    stop_monitor_immed;

    -- truncate tables 
    -- NOTE: 1. In PL/SQL, in order to change the starting point of a sequence,
    --          one has to drop the sequence and then re-create it. But if we 
    --          re-create aqmon_monitor_seq here, there will be runtime error
    --          when using the sequeuence in other procedures. So we do not 
    --          process aqmon_monitor_seq here.
    --         
    --       2. Some tables (aqmon_queue_owners_tab, aqmon_params_tab) 
    --          containing system parameters are not truncated.
    BEGIN
      -- please use the following order
      -- table available in 10.2/11.1/11.2
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_bqueues_tab';          
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_bqueues_accum_tab';    
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_bsubscribers_tab';     
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_queue_config_tab';     
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_subscriber_config_tab';

      EXECUTE IMMEDIATE 'DELETE FROM aqmon_params_tab WHERE ' ||
         'PARAM_NAME=''APT_BEGIN_SNAPSHOT_ID'' OR ' ||
         'PARAM_NAME=''APT_END_SNAPSHOT_ID''   OR ' ||
         'PARAM_NAME=''APT_BEGIN_MONITOR_TM''  OR ' ||
         'PARAM_NAME=''APT_END_MONITOR_TM''';

      -- table available in 11.1/11.2
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_pqueues_tab';          
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_pqueues_accum_tab';    
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_psubscribers_tab';     

      -- table available in 11.2
      EXECUTE IMMEDIATE 'TRUNCATE TABLE aqmon_pqmncache_tab';        

    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    
    SELECT max(SNAP_ID) INTO tmp 
      FROM sys.dba_hist_snapshot;
    IF tmp IS NULL THEN
       tmp := -1;
    END IF;
    prvt_set_param('APT_BEGIN_SNAPSHOT_ID', tmp);

    SELECT systimestamp INTO mytimestamp FROM DUAL;
    prvt_set_param2('APT_BEGIN_MONITOR_TM', mytimestamp);

   
    -- set monitor status again, because status has been set to 
    -- AQMON_STATUS_STOP in stop_monitor_immed();
    prvt_set_param('APT_MONITOR_STATUS', AQMON_STATUS_START);

    -- collect configuration stats 
    jobname := AQMON_JOB_COLLECT_CONFIG;
    freq := prvt_get_param('APT_COLLECT_CONFIG_FREQ');
    IF freq IS NOT NULL AND freq > 0 THEN
      IF freq = INFINITE_TIME THEN
        -- collect only once (default)
        dbms_scheduler.create_job(
          job_name => jobname,
          job_type => 'PLSQL_BLOCK',
          job_action=> 'begin dbms_aq_monitor_util.collect_config; end;',
          enabled => false,
          start_date => systimestamp,
          auto_drop => true);
        dbms_scheduler.enable(jobname);       
      ELSE
        -- collect periodically
        rept_intv := prvt_get_repeat_interval(freq);
        IF rept_intv IS NOT NULL THEN
          dbms_scheduler.create_job(
            job_name => jobname,
            job_type => 'PLSQL_BLOCK',
            job_action=> 'begin dbms_aq_monitor_util.collect_config; end;',
            enabled => false,
            start_date => systimestamp,
            repeat_interval => rept_intv,
            auto_drop => false);
          dbms_scheduler.enable(jobname);
        END IF;
      END IF;
    END IF;

    -- periodically collect runtime stats from gv$persistent views
    jobname := AQMON_JOB_COLLECT_RUNTIME;
    freq := prvt_get_param('APT_COLLECT_RUNTIME_FREQ'); 
    IF freq IS NOT NULL AND freq > 0 THEN
      rept_intv := prvt_get_repeat_interval(freq);
      IF rept_intv IS NOT NULL THEN
        dbms_scheduler.create_job(
          job_name => jobname,
          job_type => 'PLSQL_BLOCK',
          job_action=> 'begin dbms_aq_monitor_util.collect_runtime; end;',
          enabled => false,
          start_date => systimestamp,
          repeat_interval => rept_intv,
          auto_drop => false);
        dbms_scheduler.enable(jobname);
      END IF;
    END IF;

    -- periodically generate AWR reports
    jobname := AQMON_JOB_DUMP_AWR;
    freq := prvt_get_param('APT_DUMP_AWR_FREQ');
    IF freq IS NOT NULL AND freq > 0 THEN
      rept_intv := prvt_get_repeat_interval(freq);
      IF rept_intv IS NOT NULL THEN
        dbms_scheduler.create_job(
          job_name => jobname,
          job_type => 'PLSQL_BLOCK',
          job_action=> 'begin DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT; end;',
          enabled => false,
          start_date => systimestamp,
          repeat_interval => rept_intv,
          auto_drop => false);
        dbms_scheduler.enable(jobname);
      END IF;
    END IF;

    prvt_echo_line('New monitor is started');

  END start_monitor_immed;



  -----------------------------------------------------------------------
  -- Part 3: monitor operations
  ----------------------------------------------------------------------- 

  -- EXPORT PROCEDURE: stop all statistics collecting processes. Existing 
  --                   statistics are still kept in database.
  --
  -- PARAM  : stop_time - number of minutes after which the monitor
  --                      will be stopped (must be an integer)
  PROCEDURE stop_monitor(stop_time IN NUMBER DEFAULT 0) AS 
    jobname       VARCHAR2(200);
    mytimestamp   TIMESTAMP;
    plural        VARCHAR2(2);
  BEGIN
    IF stop_time IS NULL OR stop_time < 0 THEN
      prvt_echo_error('Time value must be positive or 0.');
      RETURN;
    END IF;

    IF stop_time > 2000000*1440.0 THEN
      -- if longer than 2000000 days, or 5479 years
      prvt_echo_error('Time value is too large.');
      RETURN;
    ELSE    
      EXECUTE IMMEDIATE 'SELECT systimestamp+' || (stop_time/1440.0) || 
                        ' FROM DUAL' INTO mytimestamp;
    END IF;

    BEGIN
      jobname := AQMON_JOB_STOP_MONITOR;
      dbms_scheduler.create_job(
        job_name => jobname,
        job_type => 'PLSQL_BLOCK',
        job_action=> 'begin dbms_aq_monitor_util.stop_monitor_immed; end;',
        enabled => false,
        start_date => mytimestamp,
        auto_drop => true);
      dbms_scheduler.enable(jobname);
    
      IF stop_time > 0 THEN
        IF stop_time > 1 THEN plural := 's';
        ELSE                  plural := NULL;
        END IF;
        prvt_echo_line('Monitor is scheduled to stop after ' || stop_time || 
                       ' minute' || plural || '.');
      ELSE
        prvt_echo_line('Monitor is stopped.');
      END IF;
    EXCEPTION
      WHEN OTHERS THEN
        IF sqlcode = -27477 THEN  -- job aleady exists exception
          prvt_echo_error('A STOP_MONITOR job has been scheduled.');
          prvt_echo_line('Use clear_scheduled_job(''stop_monitor'') to ' || 
                         'remove the existing job.');
        ELSE RAISE;
        END IF;
    END;

  END stop_monitor;



  -- EXPORT PROCEDURE: Start the monitor. When started, the monitor will 
  --                   clear up all previously collected statistics. It 
  --                   will also overwrite log files in the log directory. 
  --
  -- PARAM  : start_time - number of minutes after which the monitor
  --                       will be started  
  PROCEDURE start_monitor(start_time IN NUMBER DEFAULT 0) AS 
    jobname        VARCHAR2(200);
    mytimestamp    TIMESTAMP;
    plural         VARCHAR2(2);
  BEGIN
    IF start_time IS NULL OR start_time < 0 THEN
      prvt_echo_error('Time value must be positive or 0.');
      RETURN;
    END IF;

    IF NOT prvt_check_monitor_config THEN
      prvt_echo_line('--');
      prvt_echo_line('Monitor is not started.');
      RETURN;
    END IF;

    IF start_time > 2000000*1440.0 THEN
      -- if longer than 2000000 days, or 5479 years
      prvt_echo_error('Time value is too large.');
      RETURN;
    ELSE    
      EXECUTE IMMEDIATE 'SELECT systimestamp+' || (start_time/1440.0) || 
                        ' FROM DUAL' INTO mytimestamp;
    END IF;

    BEGIN
      jobname := AQMON_JOB_START_MONITOR;
      dbms_scheduler.create_job(
        job_name => jobname,
        job_type => 'PLSQL_BLOCK',
        job_action=> 'begin dbms_aq_monitor_util.start_monitor_immed; end;',
        enabled => false,
        start_date => mytimestamp,
        auto_drop => true);
      dbms_scheduler.enable(jobname);
    
      IF start_time > 0 THEN
        IF start_time > 1 THEN plural := 's';
        ELSE                   plural := NULL;
        END IF;
        prvt_echo_line('Monitor is scheduled to start after ' || start_time || 
                       ' minute' || plural || '.');
      ELSE
        prvt_echo_line('New monitor is started.');
      END IF;
    EXCEPTION
      WHEN OTHERS THEN
        IF sqlcode = -27477 THEN  -- job aleady exists exception
          prvt_echo_error('A START_MONITOR job has been scheduled.');
          prvt_echo_line('Use clear_scheduled_job(''start_monitor'') to ' || 
                         'remove the existing job.');
        ELSE RAISE;
        END IF;
    END;

  END start_monitor;



  -- EXPORT PROCEDURE: Clear scheduled job
  --
  -- PARAM  : job_name - could be 'start_monitor' or 'stop_monitor'
  PROCEDURE clear_scheduled_job(job_name IN VARCHAR2 DEFAULT NULL) IS
  BEGIN
    IF job_name IS NULL THEN
      RETURN;
    END IF;
 
    CASE UPPER(job_name)
      WHEN 'START_MONITOR' THEN prvt_stop_job(AQMON_JOB_START_MONITOR);
      WHEN 'STOP_MONITOR'  THEN prvt_stop_job(AQMON_JOB_STOP_MONITOR);
      ELSE                      prvt_echo_error('Unrecognized job name.');
                                RETURN;
    END CASE;      

    prvt_echo_line('Successfully clear scheduled job.');
  END clear_scheduled_job;  
                      


  -- Clear all tables and data used by the monitor.
  PROCEDURE destroy_monitor AS
  BEGIN 
    stop_monitor_immed;
    BEGIN 
      -- table available in 10.2/11.1/11.2
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_bqueues_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_bqueues_accum_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_bsubscribers_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_queue_config_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_subscriber_config_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_queue_owner_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_params_tab';      
      EXECUTE IMMEDIATE 'DROP SEQUENCE aqmon_monitor_seq';

      -- table available in 11.1/11.2
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqueues_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqueues_accum_tab';
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_psubscribers_tab';

      -- table available in 11.2
      EXECUTE IMMEDIATE 'DROP TABLE aqmon_pqmncache_tab';

      prvt_echo_line('Monitor is successfully destroyed.');

    EXCEPTION WHEN OTHERS THEN NULL;
    END;    
  END destroy_monitor; 



  -- Collect top SQL and segment statistics from dba tables/views.
  -- (this procedure is general for 10.2 and 11.1 and 11.2)
  --  
  -- PARAM  : inst_number      - instance number
  --          queue_table_name - keyword to determine whether a certain 
  --                             sql command is related to the given queue
  --          output_file      - file handler for output
  PROCEDURE prvt_get_sql_seg(inst_number      IN NUMBER,
                             queue_table_name IN VARCHAR2,
                             output_file      IN UTL_FILE.FILE_TYPE) AS
    ustos        NUMBER := 1000000;  --  const: microsecond to second

    myDBTIME     NUMBER; 
    myDBCPU      NUMBER;
    myDBID       NUMBER;
    myIOWTM      NUMBER;
    myGets       NUMBER;
    myPHYR       NUMBER;
    myPHYW       NUMBER;

    cnt          NUMBER;

    tmp_cgfc     NUMBER;
    tmp_dbfc     NUMBER;
    tmp_rgfc     NUMBER;

    mon_status   NUMBER;
    mon_beginsnp NUMBER;
    mon_endsnp   NUMBER;

    plural             VARCHAR2(2);
    shared_subsql_len  NUMBER;
    shared_memory_bnd  NUMBER;

    -- Cursor sql1: SQL ordered by Elapsed Time
    --   vdbtime   : total database time (in microsecond)
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_sql1 (vdbtime NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                      vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT to_char(sqt.elat/ustos,'9,999,999,990.00')         || ' ' ||
             to_char(sqt.execs,'9,999,999,999')                 || ' ' ||
             lpad(decode(sqt.execs, 0, ' ', null, ' '
                  , to_char(sqt.elat/sqt.execs/ustos,'9,999,990.00'))
                  , 13)                                         || ' ' ||
             lpad(decode(vdbtime, 0, ' ', null, ' '
                  , to_char(sqt.elat/vdbtime*100,'990.00')), 7) || ' ' ||
             lpad(decode(sqt.elat,0, ' ' ,null, ' '
                  , to_char(sqt.cput/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(decode(sqt.elat, 0, ' ', null, ' '
                  , to_char(sqt.uiot/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(sqt.sql_id, 15)                              || '  ' ||
             decode(sqt.module, null, null
                  , rpad('Module: ' || sqt.module,66))                 ||
             to_char(nvl(substr(regexp_replace(st.sql_text,'(\s)+',' '), 1, 
                  2000), to_clob(' ** SQL Text not available ** '))) aa
        FROM (SELECT sql_id
                   , max(module)              module
                   , sum(disk_reads_delta)    reads
                   , sum(executions_delta)    execs
                   , sum(cpu_time_delta)      cput
                   , sum(elapsed_time_delta)  elat 
                   , sum(iowait_delta)        uiot   
                FROM sys.dba_hist_sqlstat
               WHERE dbid               = vdbid
                 AND instance_number    = vinst_num
                 AND vbid               < snap_id
                 AND snap_id            <= veid
               GROUP BY sql_id
             ) sqt
           , sys.dba_hist_sqltext st
       WHERE st.sql_id(+) = sqt.sql_id
         AND st.dbid      = vdbid
         AND instr(st.sql_text, vkeyword) > 0
         AND   decode(vdbtime, 0, 2, null, 2, 100*sqt.elat/vdbtime)
             > decode(vdbtime, 0, 1, null, 2, AQMON_TOP_PCT_SQL)
       ORDER BY sqt.elat desc, sqt.sql_id;
     


    -- Cursor sql2: SQL ordered by CPU Time
    --   vdbcpu    : total database CPU time (in microsecond)
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_sql2 (vdbcpu NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                      vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT to_char(sqt.cput/ustos,'999,990.00')               || ' ' ||
             to_char(sqt.execs,'999,999,999')                   || ' ' ||
             lpad(decode(sqt.execs, 0, ' ', null, ' '
                  , to_char(sqt.cput/sqt.execs/ustos,'99,990.00'))
                  , 10)                                         || ' ' ||
             lpad(decode(vdbcpu, 0, ' ', null, ' '
                  , to_char(sqt.cput/vdbcpu*100,'990.00')), 7)  || ' ' ||
             lpad(to_char(sqt.elat/ustos, '999,990.00'), 11)    || ' ' ||
             lpad(decode(sqt.elat,0, ' ' ,null, ' '
                  , to_char(sqt.cput/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(decode(sqt.elat, 0, ' ', null, ' '
                  , to_char(sqt.uiot/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(sqt.sql_id, 15)                              || '  ' ||
             decode(sqt.module, null, null
                  , rpad('Module: ' || sqt.module,66))                 ||
             to_char(nvl(substr(regexp_replace(st.sql_text,'(\s)+',' '), 1, 
                  2000), to_clob(' ** SQL Text not available ** '))) aa
        FROM (SELECT sql_id
                   , max(module)              module
                   , sum(buffer_gets_delta)   gets
                   , sum(executions_delta)    execs
                   , sum(cpu_time_delta)      cput
                   , sum(elapsed_time_delta)  elat 
                   , sum(iowait_delta)        uiot   
                FROM sys.dba_hist_sqlstat
               WHERE dbid               = vdbid
                 AND instance_number    = vinst_num
                 AND vbid               < snap_id
                 AND snap_id            <= veid
               GROUP BY sql_id
             ) sqt
           , sys.dba_hist_sqltext st
       WHERE st.sql_id(+) = sqt.sql_id
         AND st.dbid      = vdbid
         AND instr(st.sql_text, vkeyword) > 0
         AND   decode(vdbcpu, 0, 2, null, 2, 100*sqt.cput/vdbcpu)
             > decode(vdbcpu, 0, 1, null, 2, AQMON_TOP_PCT_SQL)
       ORDER BY sqt.cput desc, sqt.sql_id;



    -- Cursor sql3: SQL ordered by User I/O Wait Time
    --   viowtm    : total user I/O wait time (in microsecond)
    --   vdbtime   : total database time (in microsecond)
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_sql3 (viowtm NUMBER, vdbtime NUMBER, vdbid NUMBER, 
           vinst_num NUMBER, vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT to_char(sqt.uiot/ustos,'999,990.00')               || ' ' ||
             to_char(sqt.execs,'999,999,999')                   || ' ' ||
             lpad(decode(sqt.execs, 0, ' ', null, ' '
                  , to_char(sqt.uiot/sqt.execs/ustos,'99,990.00'))
                  , 10)                                         || ' ' ||
             lpad(decode(viowtm, 0, ' ', null, ' '
                  , to_char(sqt.uiot/viowtm*100,'990.00')), 7)  || ' ' ||
             lpad(to_char(sqt.elat/ustos, '999,990.00'), 11)    || ' ' ||
             lpad(decode(sqt.elat,0, ' ' ,null, ' '
                  , to_char(sqt.cput/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(decode(sqt.elat, 0, ' ', null, ' '
                  , to_char(sqt.uiot/sqt.elat*100,'990.00')), 7)|| ' ' ||
             lpad(sqt.sql_id, 15)                              || '  ' ||
             decode(sqt.module, null, null
                  , rpad('Module: ' || sqt.module,66))                 ||
             to_char(nvl(substr(regexp_replace(st.sql_text,'(\s)+',' '), 1, 
                  2000), to_clob(' ** SQL Text not available ** '))) aa
        FROM (SELECT sql_id
                   , max(module)              module
                   , sum(executions_delta)    execs
                   , sum(disk_reads_delta)    reads
                   , sum(elapsed_time_delta)  elat 
                   , sum(cpu_time_delta)      cput
                   , sum(iowait_delta)        uiot   
                FROM sys.dba_hist_sqlstat
               WHERE dbid               = vdbid
                 AND instance_number    = vinst_num
                 AND vbid               < snap_id
                 AND snap_id            <= veid
               GROUP BY sql_id
             ) sqt
           , sys.dba_hist_sqltext st
       WHERE st.sql_id(+) = sqt.sql_id
         AND st.dbid      = vdbid
         AND instr(st.sql_text, vkeyword) > 0
         AND   decode(sqt.elat, 0, 2, null, 2, 100*sqt.uiot/sqt.elat)
             > decode(sqt.elat, 0, 1, null, 2, AQMON_TOP_PCT_SQL)
         AND   decode(vdbtime, 0, 2, null, 2, 100*sqt.elat/vdbtime)
             > decode(vdbtime, 0, 1, null, 2, AQMON_TOP_PCT_SQL)
       ORDER BY sqt.uiot desc, sqt.sql_id;
     


    -- Cursor sql4: SQL ordered by Buffer Gets
    --   vgets     : total buffer gets
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_sql4 (vgets NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                      vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT lpad(case when sqt.gets <= 99999999 
                       then to_char(sqt.gets,'99,999,999')
                       else to_char(round(sqt.gets/1000), 'FM99,999,999')||'K'
                  end, 11)                                           || ' ' ||
             lpad(case when sqt.execs <= 99999999
                       then to_char(sqt.execs,'99,999,999') 
                       else to_char(round(sqt.execs/1000), 'FM99,999,999')||'K'
                  end, 11)                                           || ' ' ||
             lpad(decode(sqt.execs, 0, ' ', null, ' '
                  , case when sqt.gets/sqt.execs <= 999999
                         then to_char(sqt.gets/sqt.execs,'999,990.00')
                         else to_char(round(sqt.gets/sqt.execs/1000), 
                              'FM99,999,999') || 'K'
                    end),11)                                         || ' ' ||
             lpad(decode(vgets, 0, ' ', null, ' ' 
                  , to_char(sqt.gets/vgets*100,'990.00')), 7)        || ' ' ||
             lpad(to_char(sqt.elat/ustos, '999,990.00')       ,11)   || ' ' ||
             lpad(decode(sqt.elat,0, ' ' ,null, ' '
                  , to_char(sqt.cput/sqt.elat*100,'990.00')), 7)     || ' ' ||
             lpad(decode(sqt.elat, 0, ' ', null, ' '
                  , to_char(sqt.uiot/sqt.elat*100,'990.00')), 7)     || ' ' ||
             lpad(sqt.sql_id, 15)                                   || '  ' ||
             decode(sqt.module, null, null
                  , rpad('Module: ' || sqt.module,66))                      ||
             to_char(nvl(substr(regexp_replace(st.sql_text,'(\s)+',' '), 1, 
                  2000), to_clob(' ** SQL Text not available ** '))) aa
        FROM (SELECT sql_id
                   , max(module)              module
                   , sum(buffer_gets_delta)   gets
                   , sum(executions_delta)    execs
                   , sum(elapsed_time_delta)  elat 
                   , sum(cpu_time_delta)      cput
                   , sum(iowait_delta)        uiot   
                FROM sys.dba_hist_sqlstat
               WHERE dbid               = vdbid
                 AND instance_number    = vinst_num
                 AND vbid               < snap_id
                 AND snap_id            <= veid
               GROUP BY sql_id
             ) sqt
           , sys.dba_hist_sqltext st
       WHERE st.sql_id(+) = sqt.sql_id
         AND st.dbid      = vdbid
         AND instr(st.sql_text, vkeyword) > 0
         AND   decode(vgets, 0, 2, null, 2, 100*sqt.gets/vgets)
             > decode(vgets, 0, 1, null, 2, AQMON_TOP_PCT_SQL)
       ORDER BY sqt.gets desc, sqt.sql_id;


   
    -- Cursor seg1: Segments by Logical Reads
    --   vgets     : total buffer gets
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_seg1(vgets NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                     vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT lpad(decode(n.owner, null, ' '
                  , (case when length(n.owner) < 21
                          then n.owner
                          else substr(n.owner, length(n.owner)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.tablespace_name, null, ' '
                  , (case when length(n.tablespace_name) < 21
                          then n.tablespace_name
                          else substr(n.tablespace_name,
                                      length(n.tablespace_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.object_name, null, ' '
                  , (case when length(n.object_name) < 21
                          then n.object_name
                          else substr(n.object_name, length(n.object_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.subobject_name, null, ' '
                  , (case when length(n.subobject_name) < 21
                          then n.subobject_name
                          else substr(n.subobject_name,
                                      length(n.subobject_name)-19)
                     end)), 20) || '   ' || -- max length 30 
             lpad(decode(n.object_type, null, ' '
                  , n.object_type), 18) || '   ' || -- max length 18
             lpad(case when r.logical_reads  <= 999999999 
                       then to_char(r.logical_reads,'999,999,999')
                       else to_char(round(r.logical_reads/1000), 
                                    'FM99,999,999') || 'K'
                   end,11) || '   ' ||
             lpad(decode(vgets, 0, ' ', null, ' '
                  , to_char(logical_reads/vgets*100, '990.00')), 7)  aa
        FROM sys.dba_hist_seg_stat_obj n
           , (SELECT e.dataobj#
                   , e.obj#
                   , e.dbid
                   , sum(logical_reads_delta)  logical_reads
                FROM sys.dba_hist_seg_stat e
               WHERE vbid              < snap_id
                 AND snap_id          <= veid
                 AND e.dbid            = vdbid
                 AND e.instance_number = vinst_num
               GROUP BY dataobj#, obj#, dbid) r
       WHERE n.dataobj# = r.dataobj#
         AND n.obj#     = r.obj#
         AND n.dbid     = r.dbid
         AND instr(decode(n.object_name, null, ' ', n.object_name), 
                   vkeyword) > 0
         AND r.logical_reads > 0
       ORDER BY logical_reads desc, object_name, owner, subobject_name;



    -- Cursor seg2: Segments by Physical Reads
    --   vphyr     : total physical reads
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_seg2(vphyr NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                     vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT lpad(decode(n.owner, null, ' '
                  , (case when length(n.owner) < 21
                          then n.owner
                          else substr(n.owner, length(n.owner)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.tablespace_name, null, ' '
                  , (case when length(n.tablespace_name) < 21
                          then n.tablespace_name
                          else substr(n.tablespace_name,
                                      length(n.tablespace_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.object_name, null, ' '
                  , (case when length(n.object_name) < 21
                          then n.object_name
                          else substr(n.object_name, length(n.object_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.subobject_name, null, ' '
                  , (case when length(n.subobject_name) < 21
                          then n.subobject_name
                          else substr(n.subobject_name,
                                      length(n.subobject_name)-19)
                     end)), 20) || '   ' || -- max length 30 
             lpad(decode(n.object_type, null, ' '
                  , n.object_type), 18) || '   ' || -- max length 18
             lpad(case when r.physical_reads  <= 999999999 
                       then to_char(r.physical_reads,'999,999,999')
                       else to_char(round(r.physical_reads/1000), 
                                    'FM99,999,999') || 'K'
                  end,11) || '   ' ||
             lpad(decode(vphyr, 0, ' ', null, ' '
                  , to_char(physical_reads/vphyr*100, '990.00')), 7)  aa
        FROM sys.dba_hist_seg_stat_obj n
           , (SELECT e.dataobj#
                   , e.obj#
                   , e.dbid
                   , sum(physical_reads_delta)  physical_reads
                FROM sys.dba_hist_seg_stat e
               WHERE vbid              < snap_id
                 AND snap_id          <= veid
                 AND e.dbid            = vdbid
                 AND e.instance_number = vinst_num
               GROUP BY dataobj#, obj#, dbid) r
       WHERE n.dataobj# = r.dataobj#
         AND n.obj#     = r.obj#
         AND n.dbid     = r.dbid
         AND instr(decode(n.object_name, null, ' ', n.object_name), 
                   vkeyword) > 0
         AND r.physical_reads > 0
       ORDER BY physical_reads desc, object_name, owner, subobject_name;



    -- Cursor seg3: Segments by Physical Writes
    --   vphyw     : total physical writes
    --   vdbid     : database id
    --   vinst_num : instance number 
    --   vbid      : begin snapshot id
    --   veid      : end snapshot id
    --   vkeyword  : keyword for SQL filtering
    CURSOR pgss_seg3(vphyw NUMBER, vdbid NUMBER, vinst_num NUMBER, 
                     vbid NUMBER, veid NUMBER, vkeyword VARCHAR2) IS 
      SELECT lpad(decode(n.owner, null, ' '
                  , (case when length(n.owner) < 21
                          then n.owner
                          else substr(n.owner, length(n.owner)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.tablespace_name, null, ' '
                  , (case when length(n.tablespace_name) < 21
                          then n.tablespace_name
                          else substr(n.tablespace_name,
                                      length(n.tablespace_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.object_name, null, ' '
                  , (case when length(n.object_name) < 21
                          then n.object_name
                          else substr(n.object_name, length(n.object_name)-19)
                     end)), 20) || '   ' || -- max length 30
             lpad(decode(n.subobject_name, null, ' '
                  , (case when length(n.subobject_name) < 21
                          then n.subobject_name
                          else substr(n.subobject_name,
                                      length(n.subobject_name)-19)
                     end)), 20) || '   ' || -- max length 30 
             lpad(decode(n.object_type, null, ' '
                  , n.object_type), 18) || '   ' || -- max length 18
             lpad(case when r.physical_writes  <= 999999999 
                       then to_char(r.physical_writes,'999,999,999')
                       else to_char(round(r.physical_writes/1000), 
                                    'FM99,999,999') || 'K'
                  end, 11) || '   ' ||
             lpad(decode(vphyw, 0, ' ', null, ' '
                  , to_char(physical_writes/vphyw*100, '990.00')), 7)  aa
        FROM sys.dba_hist_seg_stat_obj n
           , (SELECT e.dataobj#
                   , e.obj#
                   , e.dbid
                   , sum(physical_writes_delta)  physical_writes
                FROM sys.dba_hist_seg_stat e
               WHERE vbid              < snap_id
                 AND snap_id          <= veid
                 AND e.dbid            = vdbid
                 AND e.instance_number = vinst_num
               GROUP BY dataobj#, obj#, dbid) r
       WHERE n.dataobj# = r.dataobj#
         AND n.obj#     = r.obj#
         AND n.dbid     = r.dbid
         AND instr(decode(n.object_name, null, ' ', n.object_name), 
                   vkeyword) > 0
         AND r.physical_writes > 0
       ORDER BY physical_writes desc, object_name, owner, subobject_name;



    -- Cursor grsql1: group similar SQL ordered by their in total shared memory
    --   vdbid       : database id
    --   vinst_num   : instance number 
    --   vbid        : begin snapshot id
    --   veid        : end snapshot id
    --   vkeyword    : keyword for SQL filtering
    --   vsubsql_len : $APT_SHARED_SUBSQL_LENGTH$
    --   vmem_bnd    : $APT_SHARED_MEMORY_BOUND$ 
    CURSOR pgss_grpsql1 (vdbid NUMBER, vinst_num NUMBER, vbid NUMBER, 
                         veid NUMBER, vkeyword VARCHAR2, vsubsql_len NUMBER,
                         vmem_bnd NUMBER) IS 
      SELECT lpad(case when sum(sa.sharable_mem)  <= 999999999 
                       then to_char(sum(sa.sharable_mem), '999,999,999')
                       else to_char(round(sum(sa.sharable_mem)/1000), 
                                    'FM999,999,999,999') || 'K'
                   end,19)                            || ' ' ||
             lpad(count(*), 14)                     || '   ' ||
             nvl(sa.module, '(null)') || ' || ' || nvl(sa.action, '(null)') || 
             ' || ' || substr(to_char(st.sql_text), 1, vsubsql_len) aa
        FROM v$sqlarea sa, sys.dba_hist_sqltext st
       WHERE sa.sql_id IN (SELECT sql_id
                             FROM sys.dba_hist_sqlstat
                            WHERE dbid               = vdbid
                              AND instance_number    = vinst_num
                              AND vbid               < snap_id
                              AND snap_id            <= veid
                            GROUP BY sql_id)
         AND st.sql_id = sa.sql_id
         AND st.dbid   = vdbid
         AND instr(decode(to_char(st.sql_text), null, ' ', to_char(st.sql_text)), 
                   vkeyword) > 0  
       GROUP BY sa.module, sa.action, 
                substr(to_char(st.sql_text), 1, vsubsql_len)
      HAVING sum(sa.sharable_mem) > vmem_bnd
         AND count(*) > 1
       ORDER BY sum(sa.sharable_mem) desc, count(*) desc;


  BEGIN
    IF inst_number IS NULL OR inst_number < 0 OR 
       queue_table_name IS NULL THEN 
      RETURN;
    END IF;

    IF NOT UTL_FILE.IS_OPEN(output_file) THEN
      prvt_echo_error('No valid log file.');
      RETURN;
    END IF;

    -- If the monitor is still running, set the latest snap_id to 
    -- APT_END_SNAPSHOT_ID
    mon_status := prvt_get_param('APT_MONITOR_STATUS');
    IF mon_status IS NOT NULL AND mon_status = AQMON_STATUS_START THEN      
      SELECT max(SNAP_ID) INTO mon_endsnp
        FROM sys.dba_hist_snapshot;
      IF mon_endsnp IS NULL THEN
        mon_endsnp := -1;
      END IF;
      prvt_set_param('APT_END_SNAPSHOT_ID', mon_endsnp);
    END IF;

    mon_beginsnp := prvt_get_param('APT_BEGIN_SNAPSHOT_ID');
    mon_endsnp   := prvt_get_param('APT_END_SNAPSHOT_ID');

    IF mon_beginsnp IS NULL OR mon_endsnp IS NULL OR 
       mon_beginsnp < 0 OR mon_endsnp < 0  OR mon_beginsnp >= mon_endsnp 
    THEN
      prvt_echo_error('No valid database snapshots: begin_snapid=' ||
                      mon_beginsnp || ', end_snapid=' ||mon_endsnp ||'.');
      RETURN;
    END IF;


    -- Get current database info
    BEGIN
      SELECT d.dbid INTO myDBID FROM v$database d;
    EXCEPTION
      WHEN OTHERS THEN
        myDBID := -1;
    END;


    -- Get Time Model diff'd values
    BEGIN
      SELECT to_char(sum(decode(e.stat_name
                         ,'DB time',e.value - nvl(b.value,0),0))) 
           , to_char(sum(decode(e.stat_name
                         ,'DB CPU',e.value - nvl(b.value,0),0)))
        INTO myDBTIME, myDBCPU
        FROM sys.dba_hist_sys_time_model e
           , sys.dba_hist_sys_time_model b
       WHERE e.snap_id         = mon_endsnp
         AND b.snap_id  (+)    = mon_beginsnp
         AND e.instance_number = inst_number
         AND e.dbid            = myDBID
         AND e.instance_number = b.instance_number (+)
         AND e.dbid            = b.dbid            (+)
         AND e.stat_id         = b.stat_id         (+)
         AND e.stat_name       = b.stat_name       (+);
    EXCEPTION
      WHEN OTHERS THEN
        myDBTIME := 0;
        myDBCPU := 0;
    END;

 
    BEGIN
      SELECT to_char(sum(case when e.wait_class = 'User I/O'
                              then e.time_waited_micro-
                                   nvl(b.time_waited_micro,0)
                              else 0
                         end)) 
        INTO myIOWTM
        FROM sys.dba_hist_system_event e
           , sys.dba_hist_system_event b
       WHERE e.snap_id         = mon_endsnp
         AND b.snap_id  (+)    = mon_beginsnp
         AND e.instance_number = inst_number
         AND e.dbid            = myDBID
         AND e.instance_number = b.instance_number (+)
         AND e.dbid            = b.dbid            (+)
         AND e.event_id        = b.event_id        (+)
         AND e.event_name      = b.event_name      (+)
         AND e.wait_class     != 'Idle';
    EXCEPTION
      WHEN OTHERS THEN
        myIOWTM := 0;
    END;


    BEGIN
      SELECT sum(decode(e.stat_name
                 ,'consistent gets from cache',e.value - nvl(b.value,0),0))
           , sum(decode(e.stat_name
                 ,'db block gets from cache',e.value - nvl(b.value,0),0))
           , sum(decode(e.stat_name
                 ,'recovery block gets from cache',e.value-nvl(b.value,0),0))
           , sum(decode(e.stat_name
                 ,'physical reads',e.value - nvl(b.value,0),0))       
           , sum(decode(e.stat_name
                 ,'physical writes',e.value - nvl(b.value,0),0))        
        INTO tmp_cgfc, tmp_dbfc, tmp_rgfc, myPHYR, myPHYW
        FROM sys.dba_hist_sysstat b
           , sys.dba_hist_sysstat e
       WHERE e.snap_id         = mon_endsnp
         AND b.snap_id  (+)    = mon_beginsnp
         AND e.instance_number = inst_number
         AND e.dbid            = myDBID
         AND e.instance_number = b.instance_number (+)
         AND e.dbid            = b.dbid            (+)
         AND e.stat_id         = b.stat_id         (+)
         AND e.stat_name       = b.stat_name       (+);
    EXCEPTION
      WHEN OTHERS THEN
        tmp_cgfc := 0;
        tmp_dbfc := 0;
        tmp_rgfc := 0;
        myPHYR   := 0;
        myPHYW   := 0;
    END;
   
    myGets := tmp_cgfc + tmp_dbfc + tmp_rgfc;

    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, 'SQL Statistics');
    UTL_FILE.PUT_LINE(output_file, '  DB/Inst: ' || myDBID || '/' 
                                   || inst_number);
    UTL_FILE.PUT_LINE(output_file, '  Snaps: ' || mon_beginsnp
                                   || '-' || mon_endsnp);
    UTL_FILE.PUT_LINE(output_file, ' ');

    -- Go through cursor sql1
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'SQL ordered by Elapsed Time'); 
    UTL_FILE.PUT_LINE(output_file, '  %Total - Elapsed Time  as a percentage '
                                               || 'of Total DB time');
    UTL_FILE.PUT_LINE(output_file, '  %CPU   - CPU Time      as a percentage '
                                               || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, '  %IO    - User I/O Time as a percentage '
                                               || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file, lpad('Elapsed Time', 46));
    UTL_FILE.PUT_LINE(output_file, 
             lpad('Elapsed Time(s)', 17) || ' ' ||
             lpad('Executions', 14)      || ' ' || 
             lpad('per Exec(s)', 13)     || ' ' ||
             lpad('%Total', 7)           || ' ' ||
             lpad('%CPU', 7)             || ' ' ||  
             lpad('%IO', 7)              || ' ' ||
             lpad('SQL ID', 15)         || '  ' || 
             'SQL Module/Text');
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_sql1(myDBTIME, myDBID, inst_number, mon_beginsnp, 
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SQL);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor sql2
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'SQL ordered by CPU Time'); 
    UTL_FILE.PUT_LINE(output_file, '  %Total - CPU Time      as a percentage '
                                               || 'of Total DB CPU');
    UTL_FILE.PUT_LINE(output_file, '  %CPU   - CPU Time      as a percentage '
                                               || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, '  %IO    - User I/O Time as a percentage '
                                               || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('CPU per', 35) || lpad('Elapsed', 20));  
    UTL_FILE.PUT_LINE(output_file, 
             lpad('CPU Time(s)', 11)     || ' ' ||
             lpad('Executions', 12)      || ' ' || 
             lpad('Exec(s)', 10)         || ' ' ||
             lpad('%Total', 7)           || ' ' ||
             lpad('Times(s)', 11)        || ' ' ||
             lpad('%CPU', 7)             || ' ' ||  
             lpad('%IO', 7)              || ' ' ||
             lpad('SQL ID', 15)         || '  ' || 
             'SQL Module/Text');
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_sql2(myDBCPU, myDBID, inst_number, mon_beginsnp, 
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SQL);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor sql3
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'SQL ordered by User I/O Wait Time'); 
    UTL_FILE.PUT_LINE(output_file, '  %Total - User I/O Time as a percentage '
                                            || 'of Total User I/O Wait time');
    UTL_FILE.PUT_LINE(output_file, '  %CPU   - CPU Time      as a percentage '
                                            || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, '  %IO    - User I/O Time as a percentage '
                                            || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('User I/O', 11) || lpad('UIO Per', 24)
                                  || lpad('Elapsed', 20));  
    UTL_FILE.PUT_LINE(output_file, 
             lpad('Time(s)', 11)         || ' ' ||
             lpad('Executions', 12)      || ' ' || 
             lpad('Exec(s)', 10)         || ' ' ||
             lpad('%Total', 7)           || ' ' ||
             lpad('Times(s)', 11)        || ' ' ||
             lpad('%CPU', 7)             || ' ' ||  
             lpad('%IO', 7)              || ' ' ||
             lpad('SQL ID', 15)         || '  ' || 
             'SQL Module/Text');
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_sql3(myIOWTM, myDBTIME, myDBID, inst_number, 
                         mon_beginsnp, mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SQL);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor sql4
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'SQL ordered by Buffer Gets'); 
    UTL_FILE.PUT_LINE(output_file, '  %Total - Buffer Gets   as a percentage '
                                           || 'of Total Buffer Gets');
    UTL_FILE.PUT_LINE(output_file, '  %CPU   - CPU Time      as a percentage ' 
                                           || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, '  %IO    - User I/O Time as a percentage '
                                           || 'of Elapsed Time');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('Buffer', 11) || lpad('Gets Per', 24)
                                || lpad('Elapsed', 20));  
    UTL_FILE.PUT_LINE(output_file, 
             lpad('Gets', 11)            || ' ' ||
             lpad('Executions', 11)      || ' ' || 
             lpad('Exec(s)', 11)         || ' ' ||
             lpad('%Total', 7)           || ' ' ||
             lpad('Times(s)', 11)        || ' ' ||
             lpad('%CPU', 7)             || ' ' ||  
             lpad('%IO', 7)              || ' ' ||
             lpad('SQL ID', 15)         || '  ' || 
             'SQL Module/Text');
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_sql4(myGets, myDBID, inst_number, mon_beginsnp,
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SQL);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');


    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, 'Segment Statistics');
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor seg1
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'Segments by Logical Reads'); 
    UTL_FILE.PUT_LINE(output_file, '  % Total - percentage of logical reads');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('Tablespace', 43) || '   ' ||
             lpad('Object', 20)     || '   ' ||
             lpad('SubObject', 20)  || '   ' ||
             lpad('Object', 18)     || '   ' ||         
             lpad('Logical', 11));  

    UTL_FILE.PUT_LINE(output_file, 
             lpad('Owner', 20)       || '   ' ||
             lpad('Name', 20)        || '   ' || 
             lpad('Name', 20)        || '   ' ||
             lpad('Name', 20)        || '   ' ||
             lpad('Type', 18)        || '   ' ||  
             lpad('Reads', 11)       || '   ' ||
             lpad('%Total', 7));
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_seg1(myGets, myDBID, inst_number, mon_beginsnp,
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SEGSTAT);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor seg2
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'Segments by Physical Reads'); 
    UTL_FILE.PUT_LINE(output_file, '  % Total - percentage of physical reads');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('Tablespace', 43) || '   ' ||
             lpad('Object', 20)     || '   ' || 
             lpad('SubObject', 20)  || '   ' ||
             lpad('Object', 18)     || '   ' ||
             lpad('Physical', 11)); 

    UTL_FILE.PUT_LINE(output_file, 
             lpad('Owner', 20)      || '   ' ||
             lpad('Name', 20)       || '   ' ||
             lpad('Name', 20)       || '   ' ||
             lpad('Name', 20)       || '   ' ||
             lpad('Type', 18)       || '   ' ||  
             lpad('Reads', 11)      || '   ' ||
             lpad('%Total', 7));
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_seg2(myPHYR, myDBID, inst_number, mon_beginsnp,
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;

        EXIT WHEN (cnt >= AQMON_TOP_N_SEGSTAT);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Go through cursor seg3
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 'Segments by Physical Writes'); 
    UTL_FILE.PUT_LINE(output_file, '  % Total - percentage of physical' 
                                                || ' writes');
    UTL_FILE.PUT_LINE(output_file, ' ');

    UTL_FILE.PUT_LINE(output_file,  
             lpad('Tablespace', 43) || '   ' ||
             lpad('Object', 20)     || '   ' || 
             lpad('SubObject', 20)  || '   ' ||
             lpad('Object', 18)     || '   ' || 
             lpad('Physical', 11));  
    UTL_FILE.PUT_LINE(output_file, 
             lpad('Owner', 20)      || '   ' ||
             lpad('Name', 20)       || '   ' || 
             lpad('Name', 20)       || '   ' ||
             lpad('Name', 20)       || '   ' ||
             lpad('Type', 18)       || '   ' ||  
             lpad('Writes', 11)     || '   ' ||
             lpad('%Total', 7));
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_seg3(myPHYW, myDBID, inst_number, mon_beginsnp,
                         mon_endsnp, queue_table_name) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;
  
        EXIT WHEN (cnt >= AQMON_TOP_N_SEGSTAT);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');



    -- Display SQLs that are similar (i.e., their share the same first
    -- $APT_SHARED_SUBSQL_LENGTH$ characters) and in total consume more than 
    -- $APT_SHARED_MEMORY_BOUND$ MBytes memory
    shared_subsql_len := prvt_get_param('APT_SHARED_SUBSQL_LENGTH');
    shared_memory_bnd := prvt_get_param('APT_SHARED_MEMORY_BOUND');
    IF shared_memory_bnd > 1 THEN    plural := 's';
    ELSE                             plural := NULL;
    END IF;

    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, 'Group of Similar SQLs Consuming In ' 
                                || 'Total More than ' || shared_memory_bnd  
                                || ' MByte' || plural || ' Memory');
    UTL_FILE.PUT_LINE(output_file, ' ');

    -- Go through cursor grpsql1
    UTL_FILE.PUT_LINE(output_file, rpad('-', 80, '-'));
    UTL_FILE.PUT_LINE(output_file, 
             'Total Shared Memory      Number of   ' || 
             'Module || Action || First ' || shared_subsql_len ||
             ' Chars of SQLs');
    UTL_FILE.PUT_LINE(output_file, '            (Bytes)   Similar SQLs');
    UTL_FILE.PUT_LINE(output_file, ' ');

    BEGIN
      cnt := 0;
      FOR i in pgss_grpsql1(myDBID, inst_number, mon_beginsnp, mon_endsnp, 
                 queue_table_name, shared_subsql_len, shared_memory_bnd) LOOP
        UTL_FILE.PUT_LINE(output_file, i.aa);  
        UTL_FILE.PUT_LINE(output_file, ' ');
        cnt := cnt+1;
 
        EXIT WHEN (cnt >= 50);     
      END LOOP;
    EXCEPTION WHEN OTHERS THEN NULL;
    END;
    UTL_FILE.PUT_LINE(output_file, ' ');

    
  END prvt_get_sql_seg;



  -- Adjust collected data before generating reports. That is,
  -- generate a "first" snapshot with flag=1 (when necessary), for
  --   1. the given queue 
  --   2. each subscriber of the queue
  --   3. queue table for the queue (if it is a persistent queue)
  -- 
  -- Invoked in the following procedures:
  --   1. prvt_get_pqueue_report
  --   2. prvt_get_bqueue_report
  --   3. get_report  
  PROCEDURE prvt_postrun_adjust(db_version       IN NUMBER,
                                inst_number      IN NUMBER,
                                queue_iden       IN NUMBER, 
                                persistent_queue IN BOOLEAN) AS
    -- min memseq for the monitor
    min_memseq        NUMBER; 

    mymemseq          NUMBER;
    mymonitor_time    TIMESTAMP;
    line_bq           aqmon_bqueues_tab%ROWTYPE;
    line_bs           aqmon_bsubscribers_tab%ROWTYPE;
    v                 VARCHAR2(4000);    
    v2                VARCHAr2(4000);

    -- columns in aqmon_pqueues_tab (for dynamical SQLs);
    pq_queue_id            NUMBER;
    pq_memseq              NUMBER;
    pq_flag                NUMBER;
    pq_queue_schema        VARCHAR2(30);
    pq_queue_name          VARCHAR2(30);

    -- columns in aqmon_psubscribers_tab (for dynamical SQLs);
    ps_queue_id            NUMBER;
    ps_memseq              NUMBER;
    ps_flag                NUMBER;
    ps_queue_schema        VARCHAR2(30);
    ps_queue_name          VARCHAR2(30);
    ps_subscriber_id       NUMBER;
    ps_subscriber_name     VARCHAR2(30);
    ps_subscriber_address  VARCHAR2(1024);
    ps_protocol            NUMBER;
    ps_subscriber_type     VARCHAR2(30);

    -- columns in aqmon_pqmncache_tab (for dynamical SQLs);
    pqmn_queue_table_id    NUMBER;
    pqmn_memseq            NUMBER;
    pqmn_flag              NUMBER;
    my_queue_table_id      NUMBER;

    TYPE ppa_refcur IS REF CURSOR;
    ppa_c1 ppa_refcur;
    ppa_c2 ppa_refcur;


    -----------------------------------------------
    -- for BUFFERED queues/subscribers
    -----------------------------------------------

    -- for a buffered queue
    -- get the immediately previous memseq before upperbound
    CURSOR ppa_bq1(upperbound NUMBER) IS 
      SELECT memseq, MONITOR_TIME 
        FROM aqmon_bqueues_tab
       WHERE memseq = (SELECT max(memseq) 
                         FROM aqmon_bqueues_tab
                        WHERE memseq < upperbound);


    -- cursor for looping through different buffered subscribers
    CURSOR ppa_bs1(vINST_ID NUMBER, vQUEUE_ID NUMBER) IS
      SELECT SUBSCRIBER_ID, SUBSCRIBER_NAME
        FROM aqmon_bsubscribers_tab 
       WHERE INST_ID  = ppa_bs1.vINST_ID
         AND QUEUE_ID = ppa_bs1.vQUEUE_ID  
       GROUP BY SUBSCRIBER_ID, SUBSCRIBER_NAME;


    -- for a buffered subscriber
    -- get the immediately previous memseq before upperbound
    CURSOR ppa_bs2(upperbound NUMBER) IS 
      SELECT memseq, MONITOR_TIME 
        FROM aqmon_bsubscribers_tab
       WHERE memseq = (SELECT max(memseq) 
                         FROM aqmon_bsubscribers_tab
                        WHERE memseq < upperbound);

  BEGIN
    IF db_version IS NULL OR inst_number IS NULL OR queue_iden IS NULL OR 
       persistent_queue IS NULL THEN
      RETURN;
    END IF;

    IF persistent_queue THEN
      --
      -- deal with persistent queues and persistent subscribers
      --

      IF db_version >= 11.1 THEN

        -- P1. deal with persistent queue stats
        --
        -- get the first snapshot for the given queue
        v := 'SELECT QUEUE_ID, QUEUE_SCHEMA, QUEUE_NAME, memseq, FLAG' || 
             '  FROM aqmon_pqueues_tab' ||
             ' WHERE INST_ID  = ' || inst_number ||
             '   AND QUEUE_ID = ' || queue_iden  ||
             '   AND memseq   = (SELECT min(memseq) ' ||
             '                     FROM aqmon_pqueues_tab ' ||
             '                    WHERE INST_ID  = ' || inst_number ||
             '                      AND QUEUE_ID = ' || queue_iden || ')';
        OPEN ppa_c1 for v;
        LOOP 
          FETCH ppa_c1 INTO pq_queue_id, pq_queue_schema, pq_queue_name, 
                            pq_memseq, pq_flag;
          EXIT;
        END LOOP;
        CLOSE ppa_c1;

        EXECUTE IMMEDIATE 'SELECT min(memseq) FROM aqmon_pqueues_tab'
          INTO min_memseq;
        IF min_memseq IS NULL THEN
          min_memseq := -1;
        END IF;

        -- If this queue appeared in gv$persistent_queues after monitoring 
        -- starts, we generate a 'first-snapshot' with flag=1
        IF (pq_queue_id IS NOT NULL) AND (min_memseq > 0) AND 
           (pq_memseq > min_memseq) AND (pq_flag = 0) THEN

          -- get the immediately previous memseq before pq_memseq
          v := 'SELECT memseq, MONITOR_TIME FROM aqmon_pqueues_tab ' ||
               ' WHERE memseq = (SELECT max(memseq) FROM aqmon_pqueues_tab ' ||
               ' WHERE memseq < ' || pq_memseq || ')';
          OPEN ppa_c1 for v;
          LOOP 
            FETCH ppa_c1 INTO mymemseq, mymonitor_time;
            EXIT;
          END LOOP;
          CLOSE ppa_c1;

          v := 'INSERT INTO aqmon_pqueues_tab (INST_ID, QUEUE_ID, ' ||
               ' QUEUE_SCHEMA, QUEUE_NAME, MEMSEQ, MONITOR_TIME, FLAG)' ||
               ' VALUES (:1, :2, :3, :4, :5, :6, 1)';
          EXECUTE IMMEDIATE v USING inst_number, pq_queue_id, 
            pq_queue_schema, pq_queue_name, mymemseq, mymonitor_time;
        END IF; 



        -- P2. deal with persistent subscriber stats
        -- 
        -- loop through different subscribers
        v := 'SELECT SUBSCRIBER_ID, SUBSCRIBER_NAME ' ||
             '  FROM aqmon_psubscribers_tab ' ||
             ' WHERE INST_ID  = ' || inst_number ||
             '   AND QUEUE_ID = ' || queue_iden || 
             '   AND SUBSCRIBER_ID IS NOT NULL' ||
             ' GROUP BY SUBSCRIBER_ID, SUBSCRIBER_NAME';
        OPEN ppa_c1 for v;
        LOOP 
          FETCH ppa_c1 INTO ps_subscriber_id, ps_subscriber_name;
          EXIT WHEN ppa_c1%NOTFOUND;

          -- get the first snapshot for the given subscriber
          v2 := 'SELECT QUEUE_ID, QUEUE_SCHEMA, QUEUE_NAME, ' ||
            '       SUBSCRIBER_ADDRESS, PROTOCOL, SUBSCRIBER_TYPE,' ||
            '       memseq, FLAG' ||	
            '  FROM aqmon_psubscribers_tab ' ||
            ' WHERE INST_ID        = ' || inst_number ||
            '   AND QUEUE_ID       = ' || queue_iden ||
            '   AND SUBSCRIBER_ID  = ' || ps_subscriber_id ||
            '   AND memseq = (SELECT min(memseq) ' ||
            '    FROM aqmon_psubscribers_tab ' ||
            '   WHERE INST_ID         = ' || inst_number ||
            '     AND QUEUE_ID        = ' || queue_iden ||
            '     AND SUBSCRIBER_ID   = ' || ps_subscriber_id || ')';
          OPEN ppa_c2 for v2;
          LOOP 
            FETCH ppa_c2 INTO ps_queue_id, ps_queue_schema, ps_queue_name,
                              ps_subscriber_address, ps_protocol, 
                              ps_subscriber_type, ps_memseq, ps_flag;
            EXIT;
          END LOOP;
          CLOSE ppa_c2;

          EXECUTE IMMEDIATE 'SELECT min(memseq) FROM aqmon_psubscribers_tab'
            INTO min_memseq;
          IF min_memseq IS NULL THEN
            min_memseq := -1;
          END IF;

          -- If this subscriber appeared in gv$persistent_subscribers after 
          -- monitoring starts, we generate a 'first-snapshot' with flag=1
          IF (ps_queue_id IS NOT NULL) AND (min_memseq > 0) AND 
             (ps_memseq > min_memseq) AND (ps_flag = 0) THEN

            -- get the immediately previous memseq before ps_memseq
            v2 := 'SELECT memseq, MONITOR_TIME FROM aqmon_psubscribers_tab ' ||
                  ' WHERE memseq = (SELECT max(memseq) FROM ' ||
                  ' aqmon_psubscribers_tab WHERE memseq<' || ps_memseq || ')';
            OPEN ppa_c2 for v2;
            LOOP 
              FETCH ppa_c2 INTO mymemseq, mymonitor_time;
              EXIT;
            END LOOP;
            CLOSE ppa_c2;
 
            v2 := 'INSERT INTO aqmon_psubscribers_tab (INST_ID, QUEUE_ID, ' ||
                  ' QUEUE_SCHEMA, QUEUE_NAME, SUBSCRIBER_ID,' ||
                  ' SUBSCRIBER_NAME, SUBSCRIBER_ADDRESS, PROTOCOL,' ||
                  ' SUBSCRIBER_TYPE, MEMSEQ, MONITOR_TIME, FLAG) VALUES ' ||	
                  ' (:1, :2, :3, :4, :5, :6, :7, :8, :9, :10, :11, 1)';
            EXECUTE IMMEDIATE v2 USING inst_number, ps_queue_id,  
              ps_queue_schema, ps_queue_name, ps_subscriber_id, 
              ps_subscriber_name, ps_subscriber_address, ps_protocol,
              ps_subscriber_type, mymemseq, mymonitor_time;
          END IF; 

        END LOOP; -- loop for different subscribers
        CLOSE ppa_c1;

      END IF; -- db_version >= 11.1



      -- P3. deal with persistent QMON cache stats
      --     (valid for 11.2) 
      IF db_version >= 11.2 THEN

        -- identify queue_table_id for the given queue
        v := 'SELECT distinct(QUEUE_TABLE_ID) FROM aqmon_pqueues_tab' ||
             ' WHERE INST_ID  = ' || inst_number ||
             '   AND QUEUE_ID = ' || queue_iden  ||
             '   AND QUEUE_TABLE_ID IS NOT NULL';  
        OPEN ppa_c1 for v;
        LOOP
          FETCH ppa_c1 INTO my_queue_table_id;
          EXIT; 
        END LOOP;
        CLOSE ppa_c1;

        -- get the first snapshot for the given queue_table_id
        v := 'SELECT QUEUE_TABLE_ID, memseq, FLAG FROM aqmon_pqmncache_tab' ||
             ' WHERE INST_ID        = ' || inst_number ||
             '   AND QUEUE_TABLE_ID = ' || my_queue_table_id ||
             '   AND memseq         = (SELECT min(memseq) ' ||
             '     FROM aqmon_pqmncache_tab ' ||
             '    WHERE INST_ID   = ' || inst_number ||
             '      AND QUEUE_TABLE_ID = ' || my_queue_table_id || ')';
        OPEN ppa_c1 for v;
        LOOP 
          FETCH ppa_c1 INTO pqmn_queue_table_id, pqmn_memseq, pqmn_flag;
          EXIT;
        END LOOP;
        CLOSE ppa_c1;
 
        EXECUTE IMMEDIATE 'SELECT min(memseq) FROM aqmon_pqmncache_tab' 
          INTO min_memseq;
        IF min_memseq IS NULL THEN
          min_memseq := -1;
        END IF;


        -- If this queue table appeared in gv$persistent_qmn_cache after 
        -- monitoring starts, we generate a 'first-snapshot' with flag=1
        IF (pqmn_queue_table_id IS NOT NULL) AND (min_memseq > 0) AND 
           (pqmn_memseq > min_memseq) AND (pqmn_flag = 0) THEN

          -- get the immediately previous memseq before pqmn_memseq
          v := 'SELECT memseq, MONITOR_TIME FROM aqmon_pqmncache_tab' ||
               ' WHERE memseq = (SELECT max(memseq) FROM ' ||
               ' aqmon_pqmncache_tab WHERE memseq < ' || pqmn_memseq || ')';

          OPEN ppa_c1 for v;
          LOOP 
            FETCH ppa_c1 INTO mymemseq, mymonitor_time;
            EXIT;
          END LOOP;
          CLOSE ppa_c1;

          v := 'INSERT INTO aqmon_pqmncache_tab (INST_ID, QUEUE_TABLE_ID, ' ||
               ' MEMSEQ, MONITOR_TIME, FLAG) VALUES (:1, :2, :3, :4, 1)';
          EXECUTE IMMEDIATE v USING inst_number, my_queue_table_id, mymemseq, 
            mymonitor_time;
        END IF; 
      END IF; -- db_version >= 11.2

    ELSE
      --
      -- deal with buffered queues and buffered subscribers
      --

      -- B1. deal with buffered queue stats
      --
      -- get the first snapshot for the given queue
      BEGIN
        SELECT * into line_bq 
          FROM aqmon_bqueues_tab
         WHERE INST_ID  = inst_number
           AND QUEUE_ID = queue_iden
           AND memseq   = (SELECT min(memseq) 
                             FROM aqmon_bqueues_tab 
                            WHERE INST_ID  = inst_number
                              AND QUEUE_ID = queue_iden);
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          line_bq := NULL;
        WHEN TOO_MANY_ROWS THEN 
          -- this should not happen, unless the table 
          -- is manually manipulated.
          line_bq := NULL; 
      END;

      SELECT min(memseq) INTO min_memseq FROM aqmon_bqueues_tab;
      IF min_memseq IS NULL THEN
        min_memseq := -1;
      END IF;

      -- If this queue appeared in gv$buffered_queues after monitoring starts,
      -- we generate a 'first-snapshot' with flag=1
      IF (line_bq.QUEUE_ID IS NOT NULL) AND (min_memseq > 0) AND 
         (line_bq.memseq > min_memseq) AND (line_bq.flag = 0) THEN

        -- get the immediately previous memseq before line.memseq
        FOR i in ppa_bq1(line_bq.memseq) LOOP
          mymemseq       := i.memseq;
          mymonitor_time := i.MONITOR_TIME;
          EXIT;
        END LOOP;

        v := 'INSERT INTO aqmon_bqueues_tab (INST_ID, QUEUE_ID, ' ||
             ' QUEUE_SCHEMA, QUEUE_NAME, MEMSEQ, MONITOR_TIME, FLAG) ' || 
             ' VALUES(:1, :2, :3, :4, :5, :6, 1)';
        EXECUTE IMMEDIATE v USING line_bq.INST_ID, line_bq.QUEUE_ID, 
             line_bq.QUEUE_SCHEMA, line_bq.QUEUE_NAME, mymemseq, 
             mymonitor_time;
      END IF; 


      -- B2. deal with persistent subscriber stats
      -- 
      -- loop through different subscribers
      FOR i in ppa_bs1(inst_number, queue_iden) LOOP

        -- get the first snapshot for the given subscriber
        BEGIN
          SELECT * into line_bs 
            FROM aqmon_bsubscribers_tab
           WHERE INST_ID        = inst_number
             AND QUEUE_ID       = queue_iden
             AND SUBSCRIBER_ID  = i.SUBSCRIBER_ID
             AND SUBSCRIBER_NAME= i.SUBSCRIBER_NAME
             AND memseq         = (SELECT min(memseq) 
                                     FROM aqmon_bsubscribers_tab 
                                    WHERE INST_ID         = inst_number
                                      AND QUEUE_ID        = queue_iden
                                      AND SUBSCRIBER_ID   = i.SUBSCRIBER_ID
                                      AND SUBSCRIBER_NAME = i.SUBSCRIBER_NAME);
        EXCEPTION
          WHEN NO_DATA_FOUND THEN
            line_bs := NULL;
          WHEN TOO_MANY_ROWS THEN 
            -- this should not happen, unless the table is
            -- manually manipulated.
            line_bs := NULL; 
        END;

        SELECT min(memseq) INTO min_memseq FROM aqmon_bsubscribers_tab;
        IF min_memseq IS NULL THEN
          min_memseq := -1;
        END IF;

        -- If this subscriber appeared in gv$buffered_subscribers after 
        -- monitoring starts, we generate a 'first-snapshot' with flag=1
        IF (line_bs.QUEUE_ID IS NOT NULL) AND (min_memseq > 0) AND 
           (line_bs.memseq > min_memseq) AND (line_bs.flag = 0) THEN

          -- get the immediately previous memseq before line.memseq
          FOR j in ppa_bs2(line_bs.memseq) LOOP
            mymemseq       := j.memseq;
            mymonitor_time := j.MONITOR_TIME;
            EXIT;
          END LOOP;

          v := 'INSERT INTO aqmon_bsubscribers_tab (INST_ID, QUEUE_ID, ' ||
               ' QUEUE_SCHEMA, QUEUE_NAME, SUBSCRIBER_ID, SUBSCRIBER_NAME,' ||
               ' SUBSCRIBER_ADDRESS, PROTOCOL, SUBSCRIBER_TYPE,	' ||	
               ' MEMSEQ, MONITOR_TIME, FLAG) VALUES ' ||
               ' (:1, :2, :3, :4, :5, :6, :7, :8, :9, :10, :11, 1)';
          EXECUTE IMMEDIATE v USING line_bs.INST_ID, line_bs.QUEUE_ID, 
               line_bs.QUEUE_SCHEMA, line_bs.QUEUE_NAME, 
               line_bs.SUBSCRIBER_ID, line_bs.SUBSCRIBER_NAME, 
               line_bs.SUBSCRIBER_ADDRESS, line_bs.PROTOCOL, 
               line_bs.SUBSCRIBER_TYPE, mymemseq, mymonitor_time;
        END IF; 
      END LOOP; -- FOR i in ppa_bs1 LOOP

    END IF; -- IF persistent_queue

    EXECUTE IMMEDIATE 'COMMIT';

  END prvt_postrun_adjust;



  -- Get subscribers info for a persistent queue.
  -- (valid for database 11.1 or 11.2)
  PROCEDURE prvt_get_psubscribers(db_version  IN NUMBER,
                                  inst_number IN NUMBER,
                                  queue_iden  IN NUMBER,
                                  output_file IN UTL_FILE.FILE_TYPE) AS

    ps_subscriber_id               NUMBER;
    ps_subscriber_name             VARCHAR2(30);
    ps_subscriber_address          VARCHAR2(1024);
    ps_protocol                    NUMBER;
    ps_subscriber_type             VARCHAR2(30);

    -- for 11.1 
    pre_monitor_time               TIMESTAMP;
    pre_enqueued_msgs              NUMBER;
    pre_dequeued_msgs              NUMBER;
    pre_browsed_msgs               NUMBER;
    pre_expired_msgs               NUMBER;

    curr_monitor_time              TIMESTAMP;
    curr_enqueued_msgs             NUMBER;
    curr_dequeued_msgs             NUMBER;
    curr_browsed_msgs              NUMBER;
    curr_expired_msgs              NUMBER;
    curr_memseq                    NUMBER;

    my_time_duration               NUMBER;
    my_enqueued_msgs_td            NUMBER;
    my_dequeued_msgs_td            NUMBER;
    my_browsed_msgs_td             NUMBER;
    my_expired_msgs_td             NUMBER;
    my_enqueue_rate_td             NUMBER; 
    my_dequeue_rate_td             NUMBER; 

    my_pending_msgs_accum          NUMBER; 
    my_deq_msg_latency_accum       NUMBER;
    my_last_enqtm_accum            TIMESTAMP;
    my_last_deqtm_accum            TIMESTAMP;
    -- end for 11.1
  
    -- for 11.2 or above  
    pre_elapsed_dequeue_time       NUMBER;
    pre_dequeue_cpu_time           NUMBER;
    pre_dequeue_trans              NUMBER;
    pre_execution_count            NUMBER;
    
    curr_elapsed_dequeue_time      NUMBER;
    curr_dequeue_cpu_time          NUMBER;
    curr_dequeue_trans             NUMBER;
    curr_execution_count           NUMBER;

    my_avg_time_per_deq_td         NUMBER;
    my_cpu_time_per_deq_td         NUMBER;
    my_dequeue_trans_td            NUMBER;
    my_execution_count_td          NUMBER;       

    my_avg_msg_age_accum           NUMBER;
    -- end for 11.2 or above 

    title_line1        VARCHAR2(1000); 
    title_line2_p1     VARCHAR2(1000);
    title_line3_p1     VARCHAR2(1000);

    title_line2_p2     VARCHAR2(1000);
    title_line3_p2     VARCHAR2(1000);
 
    first_row          BOOLEAN := TRUE;
    stxt               VARCHAR2(2000);   
    stxt2              VARCHAR2(2000); 
    tmpVar             VARCHAR2(2000);
    tmpTime            TIMESTAMP;
    output_line        VARCHAR2(1000);
    sub_num            NUMBER; -- number of subscribers

    TYPE ps_refcur IS REF CURSOR;
    ps_c1 ps_refcur;
    ps_c2 ps_refcur;

  BEGIN
    -- check input parameters
    IF db_version IS NULL OR inst_number IS NULL OR queue_iden IS NULL 
       OR db_version < 11.1 THEN
      RETURN;
    END IF;   

    IF NOT UTL_FILE.IS_OPEN(output_file) THEN
      prvt_echo_error('No valid log file.');
      RETURN;
    END IF;

    IF db_version >= 11.2 THEN 

      title_line1 := lpad(' Per-Period Metrics ', 77, '-') || 
                     lpad(' | ', 64, '-') ||
                     lpad(' Accumulative Metrics ', 59, '-') || 
                     lpad(' | ', 39, '-') ||
                     lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Browsed   Expired       Enqueue       Dequeue   Avg Time per      Avg CPU Time   Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)    Dequeue(us)   per Dequeue(us)     Trans';

      title_line2_p2 := '   Execution  |  Pending   Avg Msg   Dequeue Msg              Last Enqueue Time              Last Dequeue Time  |  Timestamp';
      title_line3_p2 := '       Count  |     Msgs    Age(s)    Latency(s)                 (UTC timezone)                 (UTC timezone)  |  ';

    ELSIF db_version = 11.1 THEN

      title_line1 := lpad(' Per-Period Metrics ', 50, '-') || 
                     lpad(' | ', 36, '-') ||
                     lpad(' Accumulative Metrics ', 54, '-') || 
                     lpad(' | ', 34, '-') ||
                     lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Browsed   Expired       Enqueue       Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)';

      title_line2_p2 := '  |  Pending   Dequeue Msg              Last Enqueue Time              Last Dequeue Time  |  Timestamp';
      title_line3_p2 := '  |     Msgs    Latency(s)                 (UTC timezone)                 (UTC timezone)  |  ';

    END IF;

    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    
    EXECUTE IMMEDIATE 'SELECT count(distinct SUBSCRIBER_ID) ' ||
                      '  FROM aqmon_psubscribers_tab ' ||
                      ' WHERE INST_ID   = ' || inst_number ||
                      '   AND QUEUE_ID  = ' || queue_iden ||
                      '   AND SUBSCRIBER_ID IS NOT NULL'
      INTO sub_num;

    IF sub_num = 0 THEN
      tmpVar := 'No subscriber found.';
    ELSIF sub_num = 1 THEN
      tmpVar := '1 subscriber found.';
    ELSE
      tmpVar := sub_num || ' subscribers found.';
    END IF;

    UTL_FILE.PUT_LINE(output_file, 'Subscriber Statistics : ' || tmpVar);
    UTL_FILE.PUT_LINE(output_file, ' ');

    -- loop through different subscribers
    stxt := 'SELECT SUBSCRIBER_ID, SUBSCRIBER_NAME, SUBSCRIBER_ADDRESS, ' ||
            '       PROTOCOL, SUBSCRIBER_TYPE ' ||
            '  FROM aqmon_psubscribers_tab ' ||
            ' WHERE INST_ID  = ' || inst_number ||
            '   AND QUEUE_ID = ' || queue_iden || 
            '   AND SUBSCRIBER_ID IS NOT NULL' ||
            ' GROUP BY SUBSCRIBER_ID, SUBSCRIBER_NAME, SUBSCRIBER_ADDRESS,' ||
            '       PROTOCOL, SUBSCRIBER_TYPE ' ||
            ' ORDER BY SUBSCRIBER_ID, SUBSCRIBER_NAME';
    OPEN ps_c1 for stxt;
    LOOP
      FETCH ps_c1 INTO ps_subscriber_id, ps_subscriber_name, 
                       ps_subscriber_address, ps_protocol, ps_subscriber_type;
      EXIT WHEN ps_c1%NOTFOUND;     
    
      IF ps_c1%ROWCOUNT > 1 THEN 
        UTL_FILE.PUT_LINE(output_file, ' ');
        UTL_FILE.PUT_LINE(output_file, ' ');
        UTL_FILE.PUT_LINE(output_file, ' ');
      END IF;

      UTL_FILE.PUT_LINE(output_file, '  Subscriber ID       : ' || 
                                        ps_subscriber_id);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Name     : ' || 
                                        ps_subscriber_name);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Address  : ' || 
                                        ps_subscriber_address);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Protocol : ' || 
                                        ps_protocol);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Type     : ' || 
                                        ps_subscriber_type);

      -- get FIRST_ACTIVITY_TIME
      EXECUTE IMMEDIATE 'SELECT min(FIRST_ACTIVITY_TIME) ' ||
        '  FROM aqmon_psubscribers_tab ' ||
        ' WHERE INST_ID         = ' || inst_number ||
        '   AND QUEUE_ID        = ' || queue_iden ||
        '   AND SUBSCRIBER_ID   = ' || ps_subscriber_id ||
        '   AND FIRST_ACTIVITY_TIME IS NOT NULL'
      INTO tmpTime;
 
      UTL_FILE.PUT_LINE(output_file, '  First Activity Time : ' || tmpTime); 
      UTL_FILE.PUT_LINE(output_file, ' ');    
     
      first_row := TRUE;
      -- loop through runtime stats cursor
      stxt2 := 'SELECT ENQUEUED_MSGS, DEQUEUED_MSGS, BROWSED_MSGS, ' ||
               '       EXPIRED_MSGS, LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
               '       DEQUEUED_MSG_LATENCY, MEMSEQ, MONITOR_TIME ' ||
               '  FROM aqmon_psubscribers_tab ' ||
               ' WHERE INST_ID       = ' || inst_number ||
               '   AND QUEUE_ID      = ' || queue_iden ||
               '   AND SUBSCRIBER_ID = ' || ps_subscriber_id ||
               ' ORDER BY memseq';
      OPEN ps_c2 for stxt2;
      LOOP 
        FETCH ps_c2 INTO curr_enqueued_msgs, curr_dequeued_msgs, 
                         curr_browsed_msgs, curr_expired_msgs, 
                         my_last_enqtm_accum, my_last_deqtm_accum, 
                         my_deq_msg_latency_accum, curr_memseq, 
                         curr_monitor_time;
        EXIT WHEN ps_c2%NOTFOUND;

        curr_memseq              := nvl(curr_memseq, -1); 
        curr_enqueued_msgs       := nvl(curr_enqueued_msgs, 0); 
        curr_dequeued_msgs       := nvl(curr_dequeued_msgs, 0); 
        curr_browsed_msgs        := nvl(curr_browsed_msgs, 0); 
        curr_expired_msgs        := nvl(curr_expired_msgs, 0); 
        my_deq_msg_latency_accum := nvl(my_deq_msg_latency_accum, 0); 
                   

        -- fetch metrics for 11.2 or above
        IF db_version >= 11.2 THEN  
          BEGIN
            tmpVar := 'SELECT DEQUEUE_TRANSACTIONS, EXECUTION_COUNT, ' ||
                      '       ELAPSED_DEQUEUE_TIME, DEQUEUE_CPU_TIME, ' ||
                      '       AVG_MSG_AGE' ||
                      '  FROM aqmon_psubscribers_tab ' ||
                      ' WHERE INST_ID       = ' || inst_number ||
                      '   AND QUEUE_ID      = ' || queue_iden ||
                      '   AND SUBSCRIBER_ID = ' || ps_subscriber_id ||
                      '   AND memseq        = ' || curr_memseq;
            EXECUTE IMMEDIATE tmpVar INTO curr_dequeue_trans, 
              curr_execution_count, curr_elapsed_dequeue_time, 
              curr_dequeue_cpu_time, my_avg_msg_age_accum;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          curr_dequeue_trans        := nvl(curr_dequeue_trans, 0);
          curr_execution_count      := nvl(curr_execution_count, 0);
          curr_elapsed_dequeue_time := nvl(curr_elapsed_dequeue_time, 0);
          curr_dequeue_cpu_time     := nvl(curr_dequeue_cpu_time, 0);
          my_avg_msg_age_accum      := nvl(my_avg_msg_age_accum, 0);

        END IF;

        IF first_row THEN 
          first_row := FALSE;

          UTL_FILE.PUT_LINE(output_file, title_line1);
          UTL_FILE.PUT_LINE(output_file, title_line2_p1 || title_line2_p2 );
          UTL_FILE.PUT_LINE(output_file, title_line3_p1 || title_line3_p2 );
          UTL_FILE.PUT_LINE(output_file, ' ');
        ELSE       
          -- unit: second
          my_time_duration     := (CAST(curr_monitor_time as DATE)-
                                   CAST(pre_monitor_time as DATE))*86400.0; 

          my_enqueued_msgs_td  := curr_enqueued_msgs - pre_enqueued_msgs;
          my_dequeued_msgs_td  := curr_dequeued_msgs - pre_dequeued_msgs;
          my_browsed_msgs_td   := curr_browsed_msgs  - pre_browsed_msgs;
          my_expired_msgs_td   := curr_expired_msgs  - pre_expired_msgs; 
          
          IF my_time_duration = 0 THEN
            my_enqueue_rate_td := 0;
            my_dequeue_rate_td := 0;
          ELSE 
            my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
            my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
          END IF;
        
          -- accum metrics
          my_pending_msgs_accum := curr_enqueued_msgs-curr_dequeued_msgs;


          -- metrics for 11.2 or above
          IF db_version >= 11.2 THEN  
            my_dequeue_trans_td   := curr_dequeue_trans  -pre_dequeue_trans;
            my_execution_count_td := curr_execution_count-pre_execution_count;

            IF my_dequeued_msgs_td = 0 THEN 
              my_avg_time_per_deq_td := 0;
              my_cpu_time_per_deq_td := 0;
            ELSE
              my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                      pre_elapsed_dequeue_time)/my_dequeued_msgs_td;
              my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                      pre_dequeue_cpu_time)/my_dequeued_msgs_td;
            END IF;

          END IF;
          -- end metrics for 11.2 or above

          IF db_version >= 11.2 THEN 
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_browsed_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'),14)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'),17)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(my_dequeue_trans_td, 9)                          || ' ' ||
              lpad(my_execution_count_td, 11)                     || '  |' ||
              lpad(my_pending_msgs_accum, 9)                        || ' ' ||
              lpad(my_avg_msg_age_accum, 9)                         || ' ' ||
              lpad(my_deq_msg_latency_accum, 13)                    || ' ' ||
              lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || '  |  ' ||
              curr_monitor_time;
          ELSIF db_version = 11.1 THEN
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_browsed_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td,'FM9999999.00'),13)   || ' ' ||
              lpad(to_char(my_dequeue_rate_td,'FM9999999.00'),13) || '  |' ||
              lpad(my_pending_msgs_accum, 9)                        || ' ' ||
              lpad(my_deq_msg_latency_accum, 13)                    || ' ' ||
              lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || '  |  ' ||
              curr_monitor_time;
          ELSE
            tmpVar := NULL;
          END IF;
   
          output_line := lpad(curr_memseq, 5) || ' ' ||   
                         lpad(to_char(my_time_duration, 'FM999999'), 6) || 
                         ' ' || tmpVar;

          UTL_FILE.PUT_LINE(output_file, output_line);          
        END IF; -- end if first_row ... else ...

        pre_monitor_time    := curr_monitor_time;
        pre_enqueued_msgs   := curr_enqueued_msgs;
        pre_dequeued_msgs   := curr_dequeued_msgs;
        pre_browsed_msgs    := curr_browsed_msgs;
        pre_expired_msgs    := curr_expired_msgs;

        IF db_version >= 11.2 THEN  
          pre_elapsed_dequeue_time := curr_elapsed_dequeue_time;
          pre_dequeue_cpu_time     := curr_dequeue_cpu_time;
          pre_dequeue_trans        := curr_dequeue_trans;
          pre_execution_count      := curr_execution_count;
        END IF;

      END LOOP; -- loop through runtime stats for a certain subscriber
      CLOSE ps_c2; 

    END LOOP; -- loop through different subscribers 
    CLOSE ps_c1;
   
    UTL_FILE.PUT_LINE(output_file, ' ');
  END prvt_get_psubscribers;



  -- Get info from gv$persistent_qmn_cache.
  -- (valid for 11.2)
  PROCEDURE prvt_get_pqmncache(db_version       IN NUMBER,
                               inst_number      IN NUMBER,
                               queue_table_iden IN NUMBER, 
                               output_file      IN UTL_FILE.FILE_TYPE) AS 
    first_row                       BOOLEAN := TRUE;

    pre_monitor_time                TIMESTAMP;
    pre_tmgr_rows                   NUMBER;
    pre_tmgr_ela_time               NUMBER;
    pre_tmgr_cpu_time               NUMBER;
    pre_deqlog_rows                 NUMBER;
    pre_deqlog_ela_time             NUMBER;
    pre_deqlog_cpu_time             NUMBER;
    pre_idxcle_cnt                  NUMBER;
    pre_idxcle_ela_time             NUMBER;
    pre_idxcle_cpu_time             NUMBER;

    curr_monitor_time               TIMESTAMP;
    curr_last_tmgr_time             TIMESTAMP;
    curr_last_deqlog_time           TIMESTAMP;
    curr_last_idxcle_time           TIMESTAMP;
    curr_memseq                     NUMBER;
    curr_tmgr_rows                  NUMBER;
    curr_tmgr_ela_time              NUMBER;
    curr_tmgr_cpu_time              NUMBER;
    curr_deqlog_rows                NUMBER;
    curr_deqlog_ela_time            NUMBER;
    curr_deqlog_cpu_time            NUMBER;
    curr_idxcle_cnt                 NUMBER;
    curr_idxcle_ela_time            NUMBER;
    curr_idxcle_cpu_time            NUMBER;

    my_time_duration                NUMBER;
    my_tmgr_rows_td                 NUMBER;
    my_tmgr_ela_time_per_row_td     NUMBER;
    my_tmgr_cpu_time_per_row_td     NUMBER;
    my_deqlog_rows_td               NUMBER;
    my_deqlog_ela_time_per_row_td   NUMBER;
    my_deqlog_cpu_time_per_row_td   NUMBER;
    my_idxcle_cnt_td                NUMBER;
    my_idxcle_ela_time_per_cnt_td   NUMBER;
    my_idxcle_cpu_time_per_cnt_td   NUMBER;

    title_line1_p1 VARCHAR2(1000) := ' Snap   Time   Time Manager   Time Manager Elapsed   Time Manager CPU   Last Time Manager Processing   Deq Log    Deq Log Elapsed   Deq Log CPU Time';
    title_line2_p1 VARCHAR2(1000) := '  Seq    (s)           Rows       Time per Row(us)   Time per Row(us)            Time (UTC timezone)      Rows   Time per Row(us)        per Row(us)';

    title_line1_p2 VARCHAR2(1000) := '        Last Deq Log Processing   Index Cleanup   Index Cleanup Elapsed    Index Cleanup CPU        Last Index Cleanup Time  |  Timestamp';
    title_line2_p2 VARCHAR2(1000) := '            Time (UTC timezone)           Count      Time per Count(us)   Time per Count(us)                 (UTC timezone)  |  ';

    stxt               VARCHAR2(1000);
    tmpVar             VARCHAR2(1000);
    tmpTime            TIMESTAMP;
    output_line        VARCHAR2(1000);
    
    TYPE qmn_refcur IS REF CURSOR;
    qmn_c1 qmn_refcur;

  BEGIN
    -- check input parameters
    IF db_version IS NULL OR inst_number IS NULL OR 
       queue_table_iden IS NULL OR db_version < 11.2 THEN
      RETURN;
    END IF;   

    IF NOT UTL_FILE.IS_OPEN(output_file) THEN
      prvt_echo_error('No valid log file.');
      RETURN;
    END IF;

    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, 'Queue Table Statistics');
    UTL_FILE.PUT_LINE(output_file, ' ');
     
    first_row := TRUE;

    -- loop through different memseq
    stxt := 'SELECT TMGR_ROWS_PROCESSED, TMGR_ELAPSED_TIME, TMGR_CPU_TIME,' ||
            '  LAST_TMGR_PROCESSING_TIME, DEQLOG_ROWS_PROCESSED,' ||
            '  DEQLOG_PROCESSING_ELAPSED_TIME, DEQLOG_PROCESSING_CPU_TIME,' ||
            '  LAST_DEQLOG_PROCESSING_TIME, INDEX_CLEANUP_COUNT,' ||
            '  INDEX_CLEANUP_ELAPSED_TIME, INDEX_CLEANUP_CPU_TIME,' ||
            '  LAST_INDEX_CLEANUP_TIME, MEMSEQ, MONITOR_TIME' ||
            '  FROM aqmon_pqmncache_tab ' ||
            ' WHERE INST_ID        = ' || inst_number ||
            '   AND QUEUE_TABLE_ID = ' || queue_table_iden ||
            ' ORDER BY memseq';

    OPEN qmn_c1 for stxt;
    LOOP
      FETCH qmn_c1 INTO curr_tmgr_rows, curr_tmgr_ela_time, 
        curr_tmgr_cpu_time, curr_last_tmgr_time, curr_deqlog_rows,
        curr_deqlog_ela_time, curr_deqlog_cpu_time, curr_last_deqlog_time,
        curr_idxcle_cnt, curr_idxcle_ela_time, curr_idxcle_cpu_time, 
        curr_last_idxcle_time, curr_memseq, curr_monitor_time;
      EXIT WHEN qmn_c1%NOTFOUND;
 
      curr_memseq              := nvl(curr_memseq, -1);
      curr_tmgr_rows           := nvl(curr_tmgr_rows, 0);
      curr_tmgr_ela_time       := nvl(curr_tmgr_ela_time, 0);
      curr_tmgr_cpu_time       := nvl(curr_tmgr_cpu_time, 0);
      curr_deqlog_rows         := nvl(curr_deqlog_rows, 0);
      curr_deqlog_ela_time     := nvl(curr_deqlog_ela_time, 0);
      curr_deqlog_cpu_time     := nvl(curr_deqlog_cpu_time, 0);
      curr_idxcle_cnt          := nvl(curr_idxcle_cnt, 0);
      curr_idxcle_ela_time     := nvl(curr_idxcle_ela_time, 0);
      curr_idxcle_cpu_time     := nvl(curr_idxcle_cpu_time, 0);

      IF first_row THEN 

        UTL_FILE.PUT_LINE(output_file, lpad('-', 305, '-'));
        UTL_FILE.PUT_LINE(output_file, title_line1_p1 || title_line1_p2 );
        UTL_FILE.PUT_LINE(output_file, title_line2_p1 || title_line2_p2 );
        UTL_FILE.PUT_LINE(output_file, ' ');
        first_row := FALSE;

      ELSE       
        -- unit: second
        my_time_duration := (CAST(curr_monitor_time as DATE)-
                             CAST(pre_monitor_time as DATE))*86400.0; 

        my_tmgr_rows_td := curr_tmgr_rows-pre_tmgr_rows;
        IF my_tmgr_rows_td = 0 THEN
          my_tmgr_ela_time_per_row_td := 0;
          my_tmgr_cpu_time_per_row_td := 0;
        ELSE
          my_tmgr_ela_time_per_row_td := 
             (curr_tmgr_ela_time-pre_tmgr_ela_time)/my_tmgr_rows_td;
          my_tmgr_cpu_time_per_row_td := 
             (curr_tmgr_cpu_time-pre_tmgr_cpu_time)/my_tmgr_rows_td;
        END IF;

        my_deqlog_rows_td := curr_deqlog_rows-pre_deqlog_rows;
        IF my_deqlog_rows_td = 0 THEN
          my_deqlog_ela_time_per_row_td := 0;
          my_deqlog_cpu_time_per_row_td := 0;
        ELSE
          my_deqlog_ela_time_per_row_td := 
             (curr_deqlog_ela_time-pre_deqlog_ela_time)/my_deqlog_rows_td;
          my_deqlog_cpu_time_per_row_td := 
             (curr_deqlog_cpu_time-pre_deqlog_cpu_time)/my_deqlog_rows_td;
        END IF;        
   
        my_idxcle_cnt_td := curr_idxcle_cnt-pre_idxcle_cnt;
        IF my_idxcle_cnt_td = 0 THEN
          my_idxcle_ela_time_per_cnt_td := 0;
          my_idxcle_cpu_time_per_cnt_td := 0;
        ELSE
          my_idxcle_ela_time_per_cnt_td := 
             (curr_idxcle_ela_time-pre_idxcle_ela_time)/my_idxcle_cnt_td;
          my_idxcle_cpu_time_per_cnt_td := 
             (curr_idxcle_cpu_time-pre_idxcle_cpu_time)/my_idxcle_cnt_td;
        END IF; 
 
        tmpVar := lpad(my_tmgr_rows_td, 14)   || ' ' ||
          lpad(to_char(my_tmgr_ela_time_per_row_td*10000, 
               'FM99999999999.00'),22) || ' ' || -- centi-sec to micro-sec
          lpad(to_char(my_tmgr_cpu_time_per_row_td*10000, 
               'FM99999999999.00'),18) || ' ' || -- centi-sec to micro-sec
          lpad(nvl(to_char(curr_last_tmgr_time), ' '), 30)   || ' ' ||
          lpad(my_deqlog_rows_td, 9)          || ' ' ||
          lpad(to_char(my_deqlog_ela_time_per_row_td*10000, 
               'FM99999999999.00'),18) || ' ' || -- centi-sec to micro-sec
          lpad(to_char(my_deqlog_cpu_time_per_row_td*10000, 
               'FM99999999999.00'),18) || ' ' || -- centi-sec to micro-sec
          lpad(nvl(to_char(curr_last_deqlog_time), ' '), 30) || ' ' ||
          lpad(my_idxcle_cnt_td, 15)           || ' ' ||
          lpad(to_char(my_idxcle_ela_time_per_cnt_td*10000, 
               'FM99999999999.00'),23) || ' ' || -- centi-sec to micro-sec
          lpad(to_char(my_idxcle_cpu_time_per_cnt_td*10000, 
               'FM99999999999.00'),20) || ' ' || -- centi-sec to micro-sec
          lpad(nvl(to_char(curr_last_idxcle_time), ' '), 30) ||'  |  '||
          curr_monitor_time;

        output_line := lpad(curr_memseq, 5) || ' ' ||   
                       lpad(to_char(my_time_duration, 'FM999999'), 6) || 
                       ' ' || tmpVar;

        UTL_FILE.PUT_LINE(output_file, output_line);          
      END IF; -- end if first_row

      pre_monitor_time      := curr_monitor_time;
      pre_tmgr_rows         := curr_tmgr_rows;
      pre_tmgr_ela_time     := curr_tmgr_ela_time;
      pre_tmgr_cpu_time     := curr_tmgr_cpu_time;
      pre_deqlog_rows       := curr_deqlog_rows;
      pre_deqlog_ela_time   := curr_deqlog_ela_time;
      pre_deqlog_cpu_time   := curr_deqlog_cpu_time;
      pre_idxcle_cnt        := curr_idxcle_cnt;
      pre_idxcle_ela_time   := curr_idxcle_ela_time;
      pre_idxcle_cpu_time   := curr_idxcle_cpu_time;

    END LOOP;
    CLOSE qmn_c1;

    UTL_FILE.PUT_LINE(output_file, ' ');

  END prvt_get_pqmncache;



  -- Generate detailed statistics report for a persistent queue.
  -- (valid for 10.1, 11.1, 11.2)
  PROCEDURE prvt_get_pqueue_report(db_version     IN NUMBER,
                                   inst_number    IN NUMBER,
                                   queue_iden     IN NUMBER,
                                   single_logfile IN BOOLEAN) AS
    first_row BOOLEAN := TRUE;

    -- for 11.1
    first_monitor_time           TIMESTAMP;
    first_enqueued_msgs          NUMBER;
    first_dequeued_msgs          NUMBER;
    first_browsed_msgs           NUMBER;
    first_enqueued_expiry_msgs   NUMBER;
    first_enqueued_delay_msgs    NUMBER;
    first_msgs_made_expired      NUMBER;
    first_msgs_made_ready        NUMBER;
    first_elapsed_enqueue_time   NUMBER;
    first_elapsed_dequeue_time   NUMBER;
    first_elapsed_trans_time     NUMBER;
    first_elapsed_rule_eval_time NUMBER;
        
    pre_monitor_time             TIMESTAMP;
    pre_enqueued_msgs            NUMBER;
    pre_dequeued_msgs            NUMBER;
    pre_browsed_msgs             NUMBER;
    pre_enqueued_expiry_msgs     NUMBER;
    pre_enqueued_delay_msgs      NUMBER;
    pre_msgs_made_expired        NUMBER;
    pre_msgs_made_ready          NUMBER;
    pre_elapsed_enqueue_time     NUMBER;
    pre_elapsed_dequeue_time     NUMBER;
    pre_elapsed_trans_time       NUMBER;
    pre_elapsed_rule_eval_time   NUMBER;    

    curr_memseq                  NUMBER;
    curr_monitor_time            TIMESTAMP;
    curr_enqueued_msgs           NUMBER;
    curr_dequeued_msgs           NUMBER;
    curr_browsed_msgs            NUMBER;
    curr_enqueued_expiry_msgs    NUMBER;
    curr_enqueued_delay_msgs     NUMBER;
    curr_msgs_made_expired       NUMBER;
    curr_msgs_made_ready         NUMBER;
    curr_elapsed_enqueue_time    NUMBER;
    curr_elapsed_dequeue_time    NUMBER;
    curr_elapsed_trans_time      NUMBER;
    curr_elapsed_rule_eval_time  NUMBER;    

    my_time_duration             NUMBER;
    my_enqueued_msgs_td          NUMBER;
    my_dequeued_msgs_td          NUMBER;
    my_browsed_msgs_td           NUMBER;
    my_enqueued_expiry_msgs_td   NUMBER;
    my_enqueued_delay_msgs_td    NUMBER;
    my_msgs_made_expired_td      NUMBER;
    my_msgs_made_ready_td        NUMBER;
    my_enqueue_rate_td           NUMBER; 
    my_dequeue_rate_td           NUMBER; 
    my_avg_time_per_enq_td       NUMBER;
    my_avg_time_per_deq_td       NUMBER;
    my_trans_time_td             NUMBER;
    my_rule_eval_time_td         NUMBER;

    -- the following metrics show accumulative stats, not in a certain 
    -- time duration
    my_pending_msgs_accum        NUMBER; 
    my_last_enqtm_accum          TIMESTAMP;
    my_last_deqtm_accum          TIMESTAMP;
    my_last_tm_expitm_accum      TIMESTAMP;
    my_last_tm_readtm_accum      TIMESTAMP;
    -- end for 11.1

    -- for 11.2 or above
    first_enqueue_cpu_time       NUMBER;
    first_dequeue_cpu_time       NUMBER;
    first_enqueue_trans          NUMBER;
    first_dequeue_trans          NUMBER;
    first_execution_count        NUMBER;

    pre_enqueue_cpu_time         NUMBER;
    pre_dequeue_cpu_time         NUMBER;
    pre_enqueue_trans            NUMBER;
    pre_dequeue_trans            NUMBER;
    pre_execution_count          NUMBER;

    curr_enqueue_cpu_time        NUMBER;
    curr_dequeue_cpu_time        NUMBER;
    curr_enqueue_trans           NUMBER;
    curr_dequeue_trans           NUMBER;
    curr_execution_count         NUMBER;

    my_enqueue_trans_td          NUMBER;
    my_dequeue_trans_td          NUMBER;
    my_execution_count_td        NUMBER;       
    my_cpu_time_per_enq_td       NUMBER;
    my_cpu_time_per_deq_td       NUMBER;

    my_avg_msg_age_accum         NUMBER;
    my_deq_msg_latency_accum     NUMBER;
    -- end for 11.2 or above 


    title_line1       VARCHAR2(1000); 
    title_line2_p1    VARCHAR2(1000);
    title_line3_p1    VARCHAR2(1000);

    title_line2_p2    VARCHAR2(1000);
    title_line3_p2    VARCHAR2(1000);

    title_line2_p3    VARCHAR2(1000);
    title_line3_p3    VARCHAR2(1000);

    title_line2_p4    VARCHAR2(1000);
    title_line3_p4    VARCHAR2(1000);


    output_line        VARCHAR2(1000);
    tmpVar             VARCHAR2(1000);
    tmpInterval        NUMBER;        
    queue_filename     VARCHAR2(200);
    queue_file         UTL_FILE.FILE_TYPE;  
    plot_filename      VARCHAR2(200);
    plot_file          UTL_FILE.FILE_TYPE;      
    conf_line          aqmon_queue_config_tab%ROWTYPE;
    max_memseq         NUMBER;    
    queue_table_iden   NUMBER;
    first_act_time     TIMESTAMP;
    freq               NUMBER;
    plural             VARCHAR2(2);
    myaccumfreq        NUMBER;
    dump_plottingfile  NUMBER;
    tmpTimeDuration    NUMBER;

    TYPE pq_refcur IS REF CURSOR;
    pq_myC1 pq_refcur;    

  
  BEGIN         
    IF db_version IS NULL OR inst_number IS NULL OR queue_iden IS NULL 
       OR single_logfile IS NULL THEN
      RETURN;
    END IF;   

    -- adjust collected data (e.g., first snapshot) before analysis
    prvt_postrun_adjust(db_version, inst_number, queue_iden, TRUE);


    -- open log file 1
    IF single_logfile THEN 
      BEGIN
        queue_file := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_ALLQUEUES_REP_FNAME,
                                     'a', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;   
    ELSE
      queue_filename := 'aqmon_queue_' || inst_number || '_' || 
                        queue_iden || '.log';
      BEGIN
        queue_file := UTL_FILE.FOPEN(AQMON_LOGDIR, queue_filename, 'w', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;
    END IF;

    IF NOT UTL_FILE.IS_OPEN(queue_file) THEN
      prvt_echo_error('Cannot create log file in the log directory.');  
      RETURN;
    END IF;


    -- open log file 2 if necessary
    dump_plottingfile := prvt_get_param('APT_DUMP_PLOTTING_FILE');
    IF dump_plottingfile IS NOT NULL AND dump_plottingfile = 1 
       AND db_version >= 11.1 THEN
      plot_filename := 'aqmon_queue_' || inst_number || '_' || 
                       queue_iden || '.' || db_version || 
                       '.persistent.gnuplot';
      BEGIN
        plot_file := UTL_FILE.FOPEN(AQMON_LOGDIR, plot_filename, 'w', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;

      IF NOT UTL_FILE.IS_OPEN(plot_file) THEN
        prvt_echo_error('Cannot create log file in the log directory.');  
        RETURN;
      END IF;
    END IF;


    IF db_version >= 11.2 THEN 

      title_line1  := lpad(' Per-Period Metrics ', 133, '-') || 
                      lpad(' | ', 121, '-') ||
                      lpad(' Accumulative Metrics ', 91, '-') || 
                      lpad(' | ', 69, '-') ||
                      lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Browsed      Enqueued     Enqueued   Msgs Made   Msgs Made       Enqueue       Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs   Expiry Msgs   Delay Msgs     Expired       Ready   Rate(msg/s)   Rate(msg/s)';

      title_line2_p2 := '   Avg Time per   Avg Time per      Avg CPU Time      Avg CPU Time   Enqueue   Dequeue   Execution   Transformation   Rule Eval  |  ';
      title_line3_p2 := '    Enqueue(us)    Dequeue(us)   per Enqueue(us)   per Dequeue(us)     Trans     Trans       Count         Time(us)    Time(us)  |  ';

      title_line2_p3 := 'Pending   Avg Msg   Dequeue Msg              Last Enqueue Time              Last Dequeue Time       Last Time Manager Expiry';
      title_line3_p3 := '   Msgs    Age(s)    Latency(s)                 (UTC timezone)                 (UTC timezone)            Time (UTC timezone)';

      title_line2_p4 := '   Last Time Manager Ready Time  |  Timestamp';
      title_line3_p4 := '                 (UTC timezone)  |  ';

    ELSIF db_version = 11.1 THEN

      title_line1  := lpad(' Per-Period Metrics ', 101, '-') || 
                      lpad(' | ', 85, '-') ||
                      lpad(' Accumulative Metrics ', 79, '-') || 
                      lpad(' | ', 57, '-') ||
                      lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Browsed      Enqueued     Enqueued   Msgs Made   Msgs Made       Enqueue       Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs   Expiry Msgs   Delay Msgs     Expired       Ready   Rate(msg/s)   Rate(msg/s)';

      title_line2_p2 := '   Avg Time per   Avg Time per   Transformation   Rule Eval  |  ';
      title_line3_p2 := '    Enqueue(us)    Dequeue(us)         Time(us)    Time(us)  |  ';

      title_line2_p3 := 'Pending              Last Enqueue Time              Last Dequeue Time       Last Time Manager Expiry';
      title_line3_p3 := '   Msgs                 (UTC timezone)                 (UTC timezone)            Time (UTC timezone)';

      title_line2_p4 := '   Last Time Manager Ready Time  |  Timestamp';
      title_line3_p4 := '                 (UTC timezone)  |  ';

    END IF;

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, lpad('%', 50, '%') || '  Report Start  '
                               || lpad('%', 50, '%'));  
    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, 'Persistent Queue Statistics');
    UTL_FILE.PUT_LINE(queue_file, ' ');
      
    BEGIN
      SELECT * INTO conf_line FROM aqmon_queue_config_tab
       WHERE QUEUE_ID = queue_iden;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        conf_line := NULL;
      WHEN TOO_MANY_ROWS THEN 
        -- this should not happen, unless the table 
        -- is manually manipulated.
        conf_line := NULL; 
    END;
   

    IF conf_line.QUEUE_ID IS NOT NULL THEN
      UTL_FILE.PUT_LINE(queue_file, '  Database Version    : ' || 
                                       db_version);      
      UTL_FILE.PUT_LINE(queue_file, '  Instance Number     : ' || 
                                       inst_number);
      UTL_FILE.PUT_LINE(queue_file, '  Queue ID            : ' || 
                                       conf_line.QUEUE_ID);
      UTL_FILE.PUT_LINE(queue_file, '  Queue Name          : ' || 
                                       conf_line.QUEUE_NAME);
      IF db_version >= 11.2 THEN
        -- identify QUEUE_TABLE_ID
        tmpVar := 'SELECT distinct(QUEUE_TABLE_ID) ' || 
                  '  FROM aqmon_pqueues_tab' || 
                  ' WHERE INST_ID  = ' || inst_number ||
                  '   AND QUEUE_ID = ' || queue_iden  ||
                  '   AND QUEUE_TABLE_ID IS NOT NULL';
        OPEN pq_myC1 for tmpVar;
        LOOP
          FETCH pq_myC1 INTO queue_table_iden; 
          EXIT;
        END LOOP;
        CLOSE pq_myC1;

        UTL_FILE.PUT_LINE(queue_file, '  Queue Table ID      : ' ||
                                         queue_table_iden);   
      END IF;

      UTL_FILE.PUT_LINE(queue_file, '  Queue Table Name    : ' || 
                                       conf_line.QUEUE_TABLE); 
      UTL_FILE.PUT_LINE(queue_file, '  Queue Owner         : ' || 
                                       conf_line.QUEUE_OWNER);    
      UTL_FILE.PUT_LINE(queue_file, '  Queue Type          : ' || 
                                       conf_line.QUEUE_TYPE);
      UTL_FILE.PUT_LINE(queue_file, '  Data Type           : ' || 
                                       conf_line.DATA_TYPE);
      UTL_FILE.PUT_LINE(queue_file, '  Object Type         : ' || 
                                       conf_line.OBJECT_TYPE);
      UTL_FILE.PUT_LINE(queue_file, '  Sort Order          : ' || 
                                       conf_line.SORT_ORDER);
      UTL_FILE.PUT_LINE(queue_file, '  Recipients          : ' || 
                                       conf_line.RECIPIENTS);
      UTL_FILE.PUT_LINE(queue_file, '  Message Grouping    : ' || 
                                       conf_line.MESSAGE_GROUPING);
      IF db_version >= 11.1 THEN
        -- identify FIRST_ACTIVITY_TIME     
        tmpVar := 'SELECT distinct(FIRST_ACTIVITY_TIME) ' ||
                  '  FROM aqmon_pqueues_tab' ||
                  ' WHERE INST_ID  = ' || inst_number ||
                  '   AND QUEUE_ID = ' || queue_iden  ||
                  '   AND FIRST_ACTIVITY_TIME IS NOT NULL';
        OPEN pq_myC1 for tmpVar;
        LOOP
          FETCH pq_myC1 INTO first_act_time; 
          EXIT;
        END LOOP;
        CLOSE pq_myC1;

        UTL_FILE.PUT_LINE(queue_file, '  First Activity Time : ' ||
                                         first_act_time);
      END IF; 
    END IF; -- IF conf_line.QUEUE_ID IS NOT NULL

    UTL_FILE.PUT_LINE(queue_file, ' ');    
    UTL_FILE.PUT_LINE(queue_file, ' ');


    -- display frequency info
    freq := prvt_get_param('APT_COLLECT_RUNTIME_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- Runtime snapshots are taken every ' ||
                                     freq || ' second' || plural || '.');
    END IF;


    freq := prvt_get_param('APT_COLLECT_ACCUM_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- Accumulative statistics are ' ||
                 'calculated every ' || freq || ' second' || plural || '.');
      myaccumfreq := freq; 
    END IF;


    freq := prvt_get_param('APT_COLLECT_CONFIG_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      IF freq = INFINITE_TIME THEN
        UTL_FILE.PUT_LINE(queue_file, ' -- Queue configuation statistics are ' ||
                       'taken at the monitor starting time (once).');
      ELSE
        UTL_FILE.PUT_LINE(queue_file, ' -- Queue configuation statistics are ' ||
                         'taken every ' || freq || ' second' || plural || '.');
      END IF;
    END IF;


    freq := prvt_get_param('APT_DUMP_AWR_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- AWR reports are taken every ' || 
                                     freq || ' second' || plural || '.');
    END IF;

  
    -- (accumulative) runtime stats are only available for 11.1 or 11.2
    IF db_version >= 11.1 THEN

      IF conf_line.RECIPIENTS IS NOT NULL AND 
         UPPER(conf_line.RECIPIENTS) = 'MULTIPLE' THEN
        UTL_FILE.PUT_LINE(queue_file, ' ');
        UTL_FILE.PUT_LINE(queue_file, ' -- NOTE: For a multiple-' ||
            'subscriber queue, avg/cpu dequeue time computed at ');
        UTL_FILE.PUT_LINE(queue_file, ' --       the queue level' ||
            ' might not be accurate. Please refer to subscriber');
        UTL_FILE.PUT_LINE(queue_file, ' --       level dequeue' ||
            ' time for accurate computation. ');
      END IF;


      UTL_FILE.PUT_LINE(queue_file, ' ');
      UTL_FILE.PUT_LINE(queue_file, ' ');
      UTL_FILE.PUT_LINE(queue_file, '> Runtime Statistics');
      UTL_FILE.PUT_LINE(queue_file, ' ');
    

      first_row := TRUE;
      -- loop through runtime stats cursor
      tmpVar := 'SELECT ENQUEUED_MSGS, DEQUEUED_MSGS, BROWSED_MSGS, ' ||
        ' ENQUEUED_EXPIRY_MSGS, ENQUEUED_DELAY_MSGS, MSGS_MADE_EXPIRED,' ||
        ' MSGS_MADE_READY, ELAPSED_TRANSFORMATION_TIME, ' ||
        ' ELAPSED_RULE_EVALUATION_TIME, ELAPSED_ENQUEUE_TIME,' ||
        ' ELAPSED_DEQUEUE_TIME, LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
        ' LAST_TM_EXPIRY_TIME, LAST_TM_READY_TIME, MEMSEQ, MONITOR_TIME' ||
        '  FROM aqmon_pqueues_tab' ||
        ' WHERE INST_ID  = ' || inst_number ||
        '   AND QUEUE_ID = ' || queue_iden || 
        ' ORDER BY memseq';
      OPEN pq_myC1 for tmpVar;
      LOOP
        FETCH pq_myC1 INTO curr_enqueued_msgs, curr_dequeued_msgs, 
          curr_browsed_msgs, curr_enqueued_expiry_msgs, 
          curr_enqueued_delay_msgs, curr_msgs_made_expired,
          curr_msgs_made_ready, curr_elapsed_trans_time, 
          curr_elapsed_rule_eval_time, curr_elapsed_enqueue_time,
          curr_elapsed_dequeue_time, my_last_enqtm_accum, 
          my_last_deqtm_accum, my_last_tm_expitm_accum, 
          my_last_tm_readtm_accum, curr_memseq, curr_monitor_time;
        EXIT WHEN pq_myC1%NOTFOUND;
 
        curr_memseq                 := nvl(curr_memseq, -1);
        curr_enqueued_msgs          := nvl(curr_enqueued_msgs, 0);
        curr_dequeued_msgs          := nvl(curr_dequeued_msgs, 0);
        curr_browsed_msgs           := nvl(curr_browsed_msgs, 0);
        curr_enqueued_expiry_msgs   := nvl(curr_enqueued_expiry_msgs, 0);
        curr_enqueued_delay_msgs    := nvl(curr_enqueued_delay_msgs, 0);
        curr_msgs_made_expired      := nvl(curr_msgs_made_expired, 0);
        curr_msgs_made_ready        := nvl(curr_msgs_made_ready, 0);
        curr_elapsed_trans_time     := nvl(curr_elapsed_trans_time, 0);
        curr_elapsed_rule_eval_time := nvl(curr_elapsed_rule_eval_time, 0);
        curr_elapsed_enqueue_time   := nvl(curr_elapsed_enqueue_time, 0);
        curr_elapsed_dequeue_time   := nvl(curr_elapsed_dequeue_time, 0);

  
        IF db_version >= 11.2 THEN
          BEGIN 
            EXECUTE IMMEDIATE 'SELECT ENQUEUE_CPU_TIME, ' || 
              ' DEQUEUE_CPU_TIME, ENQUEUE_TRANSACTIONS,' ||
              ' DEQUEUE_TRANSACTIONS, EXECUTION_COUNT, AVG_MSG_AGE, ' ||
              ' DEQUEUED_MSG_LATENCY ' ||
              '  FROM aqmon_pqueues_tab ' ||
              ' WHERE INST_ID  = ' || inst_number ||
              '   AND QUEUE_ID = ' || queue_iden ||
              '   AND memseq   = ' || curr_memseq 
              INTO curr_enqueue_cpu_time, curr_dequeue_cpu_time, 
                curr_enqueue_trans, curr_dequeue_trans,
                curr_execution_count, my_avg_msg_age_accum, 
                my_deq_msg_latency_accum;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          curr_enqueue_cpu_time    := nvl(curr_enqueue_cpu_time, 0);
          curr_dequeue_cpu_time    := nvl(curr_dequeue_cpu_time, 0);
          curr_enqueue_trans       := nvl(curr_enqueue_trans, 0);
          curr_dequeue_trans       := nvl(curr_dequeue_trans, 0);
          curr_execution_count     := nvl(curr_execution_count, 0);
          my_avg_msg_age_accum     := nvl(my_avg_msg_age_accum, 0);
          my_deq_msg_latency_accum := nvl(my_deq_msg_latency_accum, 0);

        END IF;

        IF first_row THEN 
          first_monitor_time  := curr_monitor_time;
          first_row           := FALSE;

          UTL_FILE.PUT_LINE(queue_file, title_line1);
          UTL_FILE.PUT_LINE(queue_file, title_line2_p1 || title_line2_p2 || 
                                        title_line2_p3 || title_line2_p4);
          UTL_FILE.PUT_LINE(queue_file, title_line3_p1 || title_line3_p2 || 
                                        title_line3_p3 || title_line3_p4);
          UTL_FILE.PUT_LINE(queue_file, ' ');
        ELSE       
          -- unit: second
          my_time_duration    := (CAST(curr_monitor_time as DATE)-
                                  CAST(pre_monitor_time as DATE))*86400.0; 

          my_enqueued_msgs_td := curr_enqueued_msgs - pre_enqueued_msgs;
          my_dequeued_msgs_td := curr_dequeued_msgs - pre_dequeued_msgs;
          my_browsed_msgs_td  := curr_browsed_msgs  - pre_browsed_msgs;
          my_enqueued_expiry_msgs_td := curr_enqueued_expiry_msgs 
                                        - pre_enqueued_expiry_msgs; 
          my_enqueued_delay_msgs_td  := curr_enqueued_delay_msgs  
                                        - pre_enqueued_delay_msgs;
          my_msgs_made_expired_td    := curr_msgs_made_expired    
                                        - pre_msgs_made_expired; 
          my_msgs_made_ready_td      := curr_msgs_made_ready 
                                        - pre_msgs_made_ready;
          my_trans_time_td           := curr_elapsed_trans_time
                                        - pre_elapsed_trans_time;
          my_rule_eval_time_td       := curr_elapsed_rule_eval_time
                                        - pre_elapsed_rule_eval_time; 
          
          IF my_time_duration = 0 THEN
            my_enqueue_rate_td := 0;
            my_dequeue_rate_td := 0;
          ELSE 
            my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
            my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
          END IF;
        
          IF my_enqueued_msgs_td = 0 THEN 
            my_avg_time_per_enq_td := 0;
          ELSE
            my_avg_time_per_enq_td := (curr_elapsed_enqueue_time-
                    pre_elapsed_enqueue_time)/my_enqueued_msgs_td;
          END IF;

          IF my_dequeued_msgs_td = 0 THEN 
            my_avg_time_per_deq_td := 0;
          ELSE
            my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                    pre_elapsed_dequeue_time)/my_dequeued_msgs_td;
          END IF;
      
          -- accum metrics
          my_pending_msgs_accum := curr_enqueued_msgs-curr_dequeued_msgs;


          -- metrics for 11.2 or above
          IF db_version >= 11.2 THEN
            my_enqueue_trans_td   := curr_enqueue_trans-pre_enqueue_trans;
            my_dequeue_trans_td   := curr_dequeue_trans-pre_dequeue_trans;
            my_execution_count_td := curr_execution_count-pre_execution_count;

            IF my_enqueued_msgs_td = 0 THEN 
              my_cpu_time_per_enq_td := 0;
            ELSE
              my_cpu_time_per_enq_td := (curr_enqueue_cpu_time-
                      pre_enqueue_cpu_time)/my_enqueued_msgs_td;
            END IF;

            IF my_dequeued_msgs_td = 0 THEN 
              my_cpu_time_per_deq_td := 0;
            ELSE
              my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                      pre_dequeue_cpu_time)/my_dequeued_msgs_td;
            END IF;

          ELSE
             my_enqueue_trans_td      := 0;
             my_dequeue_trans_td      := 0;
             my_execution_count_td    := 0;
             my_cpu_time_per_enq_td   := 0;
             my_cpu_time_per_deq_td   := 0;
             my_avg_msg_age_accum     := 0;
             my_deq_msg_latency_accum := 0;
          END IF;
          -- end metrics for 11.2 or above

          IF db_version >= 11.2 THEN 
            tmpVar := lpad(my_enqueued_msgs_td, 10)               || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                       || ' ' ||
              lpad(my_browsed_msgs_td,9)                          || ' ' ||
              lpad(my_enqueued_expiry_msgs_td, 13)                || ' ' ||
              lpad(my_enqueued_delay_msgs_td,12)                  || ' ' ||
              lpad(my_msgs_made_expired_td, 11)                   || ' ' ||
              lpad(my_msgs_made_ready_td, 11)                     || ' ' ||
              lpad(to_char(my_enqueue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'),
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'), 
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'), 
                    17) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'), 
                    17) || ' ' ||    -- from centi-second to micro-second
              lpad(my_enqueue_trans_td, 9)                        || ' ' ||
              lpad(my_dequeue_trans_td, 9)                        || ' ' ||
              lpad(my_execution_count_td, 11)                     || ' ' ||
              lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'), 16) 
                        || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_rule_eval_time_td*10000,'FM99999999.00'),11) 
                        || '  |' ||  -- from centi-second to micro-second
              lpad(my_pending_msgs_accum, 9)                      || ' ' ||
              lpad(my_avg_msg_age_accum, 9)                       || ' ' ||
              lpad(my_deq_msg_latency_accum, 13)                  || ' ' ||
              lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)    || ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)    || ' ' ||
              lpad(nvl(to_char(my_last_tm_expitm_accum),' '),30)  || ' ' ||
              lpad(nvl(to_char(my_last_tm_readtm_accum),' '),30)||'  |  '||
              curr_monitor_time;
          ELSIF db_version = 11.1 THEN
            tmpVar := lpad(my_enqueued_msgs_td, 10)               || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                       || ' ' ||
              lpad(my_browsed_msgs_td,9)                          || ' ' ||
              lpad(my_enqueued_expiry_msgs_td, 13)                || ' ' ||
              lpad(my_enqueued_delay_msgs_td,12)                  || ' ' ||
              lpad(my_msgs_made_expired_td, 11)                   || ' ' ||
              lpad(my_msgs_made_ready_td, 11)                     || ' ' ||
              lpad(to_char(my_enqueue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'),
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'), 
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'), 16) 
                        || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_rule_eval_time_td*10000,'FM99999999.00'),11) 
                        || '  |' ||  -- from centi-second to micro-second
              lpad(my_pending_msgs_accum, 9)                      || ' ' ||
              lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)    || ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)    || ' ' ||
              lpad(nvl(to_char(my_last_tm_expitm_accum),' '),30)  || ' ' ||
              lpad(nvl(to_char(my_last_tm_readtm_accum),' '),30)||'  |  '||
              curr_monitor_time;
          ELSE
            tmpVar := NULL;
          END IF;
        
          output_line := lpad(curr_memseq, 5) || ' ' ||   
            lpad(to_char(my_time_duration, 'FM999999'), 6) || ' ' ||
            tmpVar;

          UTL_FILE.PUT_LINE(queue_file, output_line);          


          -- dump stats for gnuplot. Each queue has a specific file 
          IF dump_plottingfile IS NOT NULL AND dump_plottingfile = 1 THEN
   
            -- record accumulative time duration (unit: second)
            tmpTimeDuration := (CAST(curr_monitor_time as DATE)-
                                CAST(first_monitor_time as DATE))*86400.0;
              
            tmpVar := lpad(my_enqueued_msgs_td, 10)               || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                       || ' ' ||
              lpad(my_browsed_msgs_td,9)                          || ' ' ||
              lpad(my_enqueued_expiry_msgs_td, 13)                || ' ' ||
              lpad(my_enqueued_delay_msgs_td,12)                  || ' ' ||
              lpad(my_msgs_made_expired_td, 11)                   || ' ' ||
              lpad(my_msgs_made_ready_td, 11)                     || ' ' ||
              lpad(to_char(my_enqueue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td,'FM9999999.00'),13) || ' ' ||
              lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'), 
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'), 
                    14) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'), 
                    17) || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'), 
                    17) || ' ' ||    -- from centi-second to micro-second
              lpad(my_enqueue_trans_td, 9)                        || ' ' ||
              lpad(my_dequeue_trans_td, 9)                        || ' ' ||
              lpad(my_execution_count_td, 11)                     || ' ' ||
              lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'), 16) 
                       || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_rule_eval_time_td*10000,'FM99999999.00'),11) 
                       || '  ' ||  -- from centi-second to micro-second
              lpad(my_pending_msgs_accum, 9)                      || ' ' ||
              lpad(my_avg_msg_age_accum, 9)                       || ' ' ||
              lpad(my_deq_msg_latency_accum, 13)                || '   ' ||
              curr_monitor_time;

              UTL_FILE.PUT_LINE(plot_file, lpad(to_char(tmpTimeDuration, 
                                   'FM999999'), 6) || '   ' || tmpVar);
            END IF;

        END IF; -- end if first_row

        pre_monitor_time           := curr_monitor_time;
        pre_enqueued_msgs          := curr_enqueued_msgs;
        pre_dequeued_msgs          := curr_dequeued_msgs;
        pre_browsed_msgs           := curr_browsed_msgs;
        pre_enqueued_expiry_msgs   := curr_enqueued_expiry_msgs;
        pre_enqueued_delay_msgs    := curr_enqueued_delay_msgs;
        pre_msgs_made_expired      := curr_msgs_made_expired;
        pre_msgs_made_ready        := curr_msgs_made_ready;
        pre_elapsed_enqueue_time   := curr_elapsed_enqueue_time;
        pre_elapsed_dequeue_time   := curr_elapsed_dequeue_time;
        pre_elapsed_trans_time     := curr_elapsed_trans_time;
        pre_elapsed_rule_eval_time := curr_elapsed_rule_eval_time;   
 
        IF db_version >= 11.2 THEN 
          pre_enqueue_cpu_time  := curr_enqueue_cpu_time;
          pre_dequeue_cpu_time  := curr_dequeue_cpu_time;
          pre_enqueue_trans     := curr_enqueue_trans;
          pre_dequeue_trans     := curr_dequeue_trans;
          pre_execution_count   := curr_execution_count;
        END IF;

      END LOOP; -- runtime cursor
      CLOSE pq_myC1;

      UTL_FILE.PUT_LINE(queue_file, ' ');
      UTL_FILE.PUT_LINE(queue_file, ' ');
      UTL_FILE.PUT_LINE(queue_file, ' ');



      -- dump accumulative statistis at certain time
      EXECUTE IMMEDIATE 'SELECT max(memseq) from aqmon_pqueues_tab' 
        INTO max_memseq;
      IF max_memseq IS NULL THEN
        max_memseq := -1;
      END IF;

      UTL_FILE.PUT_LINE(queue_file, '> Accumulative Runtime Statistics');
      UTL_FILE.PUT_LINE(queue_file, ' ');

      first_row := TRUE;
      -- loop through accumulative stats cursor
      tmpVar := 'SELECT ENQUEUED_MSGS, DEQUEUED_MSGS, BROWSED_MSGS, ' ||
        ' ENQUEUED_EXPIRY_MSGS, ENQUEUED_DELAY_MSGS, MSGS_MADE_EXPIRED,' ||
        ' MSGS_MADE_READY, ELAPSED_TRANSFORMATION_TIME, ' ||
        ' ELAPSED_RULE_EVALUATION_TIME, ELAPSED_ENQUEUE_TIME,' ||
        ' ELAPSED_DEQUEUE_TIME, LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
        ' LAST_TM_EXPIRY_TIME, LAST_TM_READY_TIME, MEMSEQ, MONITOR_TIME' ||
        '  FROM aqmon_pqueues_tab' ||
        ' WHERE INST_ID  = ' || inst_number ||
        '   AND QUEUE_ID = ' || queue_iden || 
        ' ORDER BY memseq';
      OPEN pq_myC1 for tmpVar;
      LOOP
        FETCH pq_myC1 INTO curr_enqueued_msgs, curr_dequeued_msgs, 
          curr_browsed_msgs, curr_enqueued_expiry_msgs, 
          curr_enqueued_delay_msgs, curr_msgs_made_expired,
          curr_msgs_made_ready, curr_elapsed_trans_time, 
          curr_elapsed_rule_eval_time, curr_elapsed_enqueue_time,
          curr_elapsed_dequeue_time, my_last_enqtm_accum, 
          my_last_deqtm_accum, my_last_tm_expitm_accum, 
          my_last_tm_readtm_accum, curr_memseq, curr_monitor_time;
        EXIT WHEN pq_myC1%NOTFOUND;
 
        curr_memseq                 := nvl(curr_memseq, -1);
        curr_enqueued_msgs          := nvl(curr_enqueued_msgs, 0);
        curr_dequeued_msgs          := nvl(curr_dequeued_msgs, 0);
        curr_browsed_msgs           := nvl(curr_browsed_msgs, 0);
        curr_enqueued_expiry_msgs   := nvl(curr_enqueued_expiry_msgs, 0);
        curr_enqueued_delay_msgs    := nvl(curr_enqueued_delay_msgs, 0);
        curr_msgs_made_expired      := nvl(curr_msgs_made_expired, 0);
        curr_msgs_made_ready        := nvl(curr_msgs_made_ready, 0);
        curr_elapsed_trans_time     := nvl(curr_elapsed_trans_time, 0);
        curr_elapsed_rule_eval_time := nvl(curr_elapsed_rule_eval_time, 0);
        curr_elapsed_enqueue_time   := nvl(curr_elapsed_enqueue_time, 0);
        curr_elapsed_dequeue_time   := nvl(curr_elapsed_dequeue_time, 0);


        IF db_version >= 11.2 THEN
          BEGIN 
            EXECUTE IMMEDIATE 'SELECT ENQUEUE_CPU_TIME, ' || 
              ' DEQUEUE_CPU_TIME, ENQUEUE_TRANSACTIONS, ' ||
              ' DEQUEUE_TRANSACTIONS, EXECUTION_COUNT, AVG_MSG_AGE, ' ||
              ' DEQUEUED_MSG_LATENCY ' ||
              '  FROM aqmon_pqueues_tab ' ||
              ' WHERE INST_ID  = ' || inst_number ||
              '   AND QUEUE_ID = ' || queue_iden ||
              '   AND memseq   = ' || curr_memseq 
              INTO curr_enqueue_cpu_time, curr_dequeue_cpu_time, 
                curr_enqueue_trans, curr_dequeue_trans,
                curr_execution_count, my_avg_msg_age_accum, 
                my_deq_msg_latency_accum;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          curr_enqueue_cpu_time    := nvl(curr_enqueue_cpu_time, 0);
          curr_dequeue_cpu_time    := nvl(curr_dequeue_cpu_time, 0);
          curr_enqueue_trans       := nvl(curr_enqueue_trans, 0);
          curr_dequeue_trans       := nvl(curr_dequeue_trans, 0);
          curr_execution_count     := nvl(curr_execution_count, 0);
          my_avg_msg_age_accum     := nvl(my_avg_msg_age_accum, 0);
          my_deq_msg_latency_accum := nvl(my_deq_msg_latency_accum, 0);

        END IF;

        IF first_row THEN
          first_monitor_time           := curr_monitor_time;
          first_enqueued_msgs          := curr_enqueued_msgs;
          first_dequeued_msgs          := curr_dequeued_msgs;
          first_browsed_msgs           := curr_browsed_msgs;
          first_enqueued_expiry_msgs   := curr_enqueued_expiry_msgs;
          first_enqueued_delay_msgs    := curr_enqueued_delay_msgs;
          first_msgs_made_expired      := curr_msgs_made_expired;
          first_msgs_made_ready        := curr_msgs_made_ready;
          first_elapsed_enqueue_time   := curr_elapsed_enqueue_time;
          first_elapsed_dequeue_time   := curr_elapsed_dequeue_time;
          first_elapsed_trans_time     := curr_elapsed_trans_time;
          first_elapsed_rule_eval_time := curr_elapsed_rule_eval_time;

          IF db_version >= 11.2 THEN
            first_enqueue_cpu_time   := curr_enqueue_cpu_time;
            first_dequeue_cpu_time   := curr_dequeue_cpu_time;
            first_enqueue_trans      := curr_enqueue_trans;
            first_dequeue_trans      := curr_dequeue_trans;
            first_execution_count    := curr_execution_count;
          END IF;
 
          first_row := FALSE;     
          tmpInterval := myaccumfreq;

          UTL_FILE.PUT_LINE(queue_file, title_line2_p1 || title_line2_p2 || 
                                        title_line2_p3 || title_line2_p4);
          UTL_FILE.PUT_LINE(queue_file, title_line3_p1 || title_line3_p2 || 
                                        title_line3_p3 || title_line3_p4); 
          UTL_FILE.PUT_LINE(queue_file, ' ');                   
        ELSE
          -- unit: second
          my_time_duration := (CAST(curr_monitor_time as DATE)-
                               CAST(first_monitor_time as DATE))*86400.0;

          -- decide whether to dump the line          
          IF my_time_duration > tmpInterval OR 
             curr_memseq = max_memseq THEN 
            my_enqueued_msgs_td := curr_enqueued_msgs-first_enqueued_msgs;
            my_dequeued_msgs_td := curr_dequeued_msgs-first_dequeued_msgs;
            my_browsed_msgs_td  := curr_browsed_msgs -first_browsed_msgs;
            my_enqueued_expiry_msgs_td := curr_enqueued_expiry_msgs 
                                        - first_enqueued_expiry_msgs; 
            my_enqueued_delay_msgs_td  := curr_enqueued_delay_msgs  
                                        - first_enqueued_delay_msgs;
            my_msgs_made_expired_td    := curr_msgs_made_expired    
                                        - first_msgs_made_expired; 
            my_msgs_made_ready_td      := curr_msgs_made_ready      
                                        - first_msgs_made_ready;
            my_trans_time_td           := curr_elapsed_trans_time  
                                        - first_elapsed_trans_time;
            my_rule_eval_time_td       := curr_elapsed_rule_eval_time 
                                        - first_elapsed_rule_eval_time;

            IF my_time_duration = 0 THEN
              my_enqueue_rate_td := 0;
              my_dequeue_rate_td := 0;
            ELSE 
              my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
              my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
            END IF;
        
            IF my_enqueued_msgs_td = 0 THEN 
              my_avg_time_per_enq_td := 0;
            ELSE
              my_avg_time_per_enq_td := (curr_elapsed_enqueue_time-
                      first_elapsed_enqueue_time)/my_enqueued_msgs_td;
            END IF;

            IF my_dequeued_msgs_td = 0 THEN 
              my_avg_time_per_deq_td := 0;
            ELSE
              my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                      first_elapsed_dequeue_time)/my_dequeued_msgs_td;
            END IF;

            -- accum metrics
            my_pending_msgs_accum := curr_enqueued_msgs-curr_dequeued_msgs;


            -- metrics for 11.2 or above
            IF db_version >= 11.2 THEN
              my_enqueue_trans_td:=curr_enqueue_trans-first_enqueue_trans;
              my_dequeue_trans_td:=curr_dequeue_trans-first_dequeue_trans;
              my_execution_count_td := curr_execution_count
                                      -first_execution_count;

              IF my_enqueued_msgs_td = 0 THEN 
                my_cpu_time_per_enq_td := 0;
              ELSE
                my_cpu_time_per_enq_td := (curr_enqueue_cpu_time-
                        first_enqueue_cpu_time)/my_enqueued_msgs_td;
              END IF;

              IF my_dequeued_msgs_td = 0 THEN 
                my_cpu_time_per_deq_td := 0;
              ELSE
                my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                        first_dequeue_cpu_time)/my_dequeued_msgs_td;
              END IF;
            END IF;
            -- end metrics for 11.2 or above

            IF db_version >= 11.2 THEN 
              tmpVar := lpad(my_enqueued_msgs_td, 10)             || ' ' ||
                lpad(my_dequeued_msgs_td, 10)                     || ' ' ||
                lpad(my_browsed_msgs_td,9)                        || ' ' ||
                lpad(my_enqueued_expiry_msgs_td, 13)              || ' ' ||
                lpad(my_enqueued_delay_msgs_td,12)                || ' ' ||
                lpad(my_msgs_made_expired_td, 11)                 || ' ' ||
                lpad(my_msgs_made_ready_td, 11)                   || ' ' ||
                lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 
                      13) || ' ' ||
                lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 
                      13) || ' ' ||
                lpad(to_char(my_avg_time_per_enq_td*10000,'FM99999999.00'),
                      14) || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_avg_time_per_deq_td*10000,'FM99999999.00'),
                      14) || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'),
                      17) || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'),
                      17) || ' ' ||  -- from centi-second to micro-second
                lpad(my_enqueue_trans_td, 9)                      || ' ' ||
                lpad(my_dequeue_trans_td, 9)                      || ' ' ||
                lpad(my_execution_count_td, 11)                   || ' ' ||
                lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'),16)
                          || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_rule_eval_time_td*10000,'FM99999999.00'),11)
                          || '  |' ||-- from centi-second to micro-second
                lpad(my_pending_msgs_accum, 9)                    || ' ' ||
                lpad(my_avg_msg_age_accum, 9)                     || ' ' ||
                lpad(my_deq_msg_latency_accum, 13)                || ' ' ||
                lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)  || ' ' ||
                lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || ' ' ||
                lpad(nvl(to_char(my_last_tm_expitm_accum),' '),30)|| ' ' ||
                lpad(nvl(to_char(my_last_tm_readtm_accum),' '),30)
                                                              || '  |  ' ||
                curr_monitor_time;
            ELSIF db_version = 11.1 THEN
              tmpVar := lpad(my_enqueued_msgs_td, 10)             || ' ' ||
                lpad(my_dequeued_msgs_td, 10)                     || ' ' ||
                lpad(my_browsed_msgs_td,9)                        || ' ' ||
                lpad(my_enqueued_expiry_msgs_td, 13)              || ' ' ||
                lpad(my_enqueued_delay_msgs_td,12)                || ' ' ||
                lpad(my_msgs_made_expired_td, 11)                 || ' ' ||
                lpad(my_msgs_made_ready_td, 11)                   || ' ' ||
                lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 
                      13) || ' ' ||
                lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 
                      13) || ' ' ||
                lpad(to_char(my_avg_time_per_enq_td*10000,'FM99999999.00'),
                      14) || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_avg_time_per_deq_td*10000,'FM99999999.00'),
                      14) || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'),16)
                          || ' ' ||  -- from centi-second to micro-second
                lpad(to_char(my_rule_eval_time_td*10000,'FM99999999.00'),11)
                          || '  |' ||-- from centi-second to micro-second
                lpad(my_pending_msgs_accum, 9)                    || ' ' ||
                lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)  || ' ' ||
                lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || ' ' ||
                lpad(nvl(to_char(my_last_tm_expitm_accum),' '),30)|| ' ' ||
                lpad(nvl(to_char(my_last_tm_readtm_accum),' '),30)
                                                              || '  |  ' ||
                curr_monitor_time;
            ELSE
              tmpVar := NULL;
            END IF;
 
            output_line := lpad(curr_memseq, 5) || ' ' ||   
               lpad(to_char(my_time_duration, 'FM999999'), 6) || ' ' ||
               tmpVar;
   
            UTL_FILE.PUT_LINE(queue_file, output_line);          

            --modify tmpInterval for next dump 
            tmpInterval := tmpInterval + myaccumfreq;

       -- dump stats for gnuplot. Each queue has a specific file 
       --   IF AQMON_DUMP_PER_QUEUE_STAT THEN
       --     tmpFilename := 'aqmon_accum_' || j.QUEUE_ID || '.log';  
       --     ACCRT_PLOT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, 
       --                          tmpFilename, 'a', 5000);
       --     UTL_FILE.PUT_LINE(ACCRT_PLOT_OUTPUT, to_char(my_time_duration, 
       --                                   '999999.0') || '   ' || tmpVar);
       --     UTL_FILE.FCLOSE(ACCRT_PLOT_OUTPUT);
       --   END IF;

          END IF;  -- if my_time_duration > tmpInterval OR 
                   --    curr_memseq = max_memseq

        END IF; -- if first_row

      END LOOP; -- accumulative stats cursor
      CLOSE pq_myC1;    
      
      UTL_FILE.PUT_LINE(queue_file, ' ');
    
      -- get QMON cache info
      prvt_get_pqmncache(db_version, inst_number, queue_table_iden, 
                         queue_file);

      -- get subscribers info
      prvt_get_psubscribers(db_version, inst_number, queue_iden, queue_file);

    END IF; -- IF db_version >= 11.1

    -- get SQL/Segment info
    prvt_get_sql_seg(inst_number, conf_line.QUEUE_TABLE, queue_file);

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, lpad('%', 50, '%') || '   Report End   '
                               || lpad('%', 50, '%'));  
    UTL_FILE.PUT_LINE(queue_file, ' ');

    UTL_FILE.FCLOSE(queue_file);

    IF UTL_FILE.IS_OPEN(plot_file) THEN
      UTL_FILE.FCLOSE(plot_file);
    END IF;
   
    prvt_echo_line('Report for persistent queue ' || queue_iden ||
                   ' on instance ' || inst_number || ' is generated.');
  END prvt_get_pqueue_report;


  -- Get subscribers info for a buffered queue.
  -- (valid for 10.2, 11.1, 11.2)
  PROCEDURE prvt_get_bsubscribers(db_version  IN NUMBER,
                                  inst_number IN NUMBER,
                                  queue_iden  IN NUMBER,
                                  output_file IN UTL_FILE.FILE_TYPE) AS
    first_row BOOLEAN := TRUE;

    pre_monitor_time             TIMESTAMP;
    pre_cnum_msgs                NUMBER;
    pre_total_dequeued_msg       NUMBER;
    pre_total_spilled_msg        NUMBER;
    pre_expired_msgs             NUMBER;

    my_time_duration             NUMBER;
    my_enqueued_msgs_td          NUMBER;
    my_dequeued_msgs_td          NUMBER;
    my_spilled_msgs_td           NUMBER;
    my_expired_msgs_td           NUMBER;
    my_enqueue_rate_td           NUMBER; 
    my_dequeue_rate_td           NUMBER; 

    -- the following metrics show accumulative stats, not in a certain 
    -- time duration
    my_pending_msgs_accum        NUMBER; 

    -- for 11.2 or above  
    pre_elapsed_dequeue_time     NUMBER;
    pre_dequeue_cpu_time         NUMBER;

    curr_elapsed_dequeue_time    NUMBER;
    curr_dequeue_cpu_time        NUMBER;

    my_avg_time_per_deq_td       NUMBER;
    my_cpu_time_per_deq_td       NUMBER;

    my_oldest_msgid_accum        RAW(16);
    my_oldest_msg_enqtm_accum    TIMESTAMP; 
    my_last_deqtm_accum          TIMESTAMP;
    -- end for 11.2 or above 

    
    title_line1        VARCHAR2(1000); 

    title_line2_p1     VARCHAR2(1000);
    title_line3_p1     VARCHAR2(1000);

    title_line2_p2     VARCHAR2(1000);
    title_line3_p2     VARCHAR2(1000);

    tmpVar             VARCHAR2(1000);
    tmpTime            TIMESTAMP;
    output_line        VARCHAR2(1000);
    sub_num            NUMBER; -- number of subscribers

    -- cursor for looping through different subscribers
    CURSOR bs_subCursor(vINST_ID NUMBER, vQUEUE_ID NUMBER) IS
      SELECT SUBSCRIBER_ID, SUBSCRIBER_NAME, SUBSCRIBER_ADDRESS, 
             PROTOCOL, SUBSCRIBER_TYPE
        FROM aqmon_bsubscribers_tab 
       WHERE INST_ID  = bs_subCursor.vINST_ID
         AND QUEUE_ID = bs_subCursor.vQUEUE_ID  
       GROUP BY SUBSCRIBER_ID, SUBSCRIBER_NAME, SUBSCRIBER_ADDRESS, 
                PROTOCOL, SUBSCRIBER_TYPE
       ORDER BY SUBSCRIBER_ID, SUBSCRIBER_NAME;
    
    -- cursor for per-subscriber stats
    CURSOR bs_statsCursor(vINST_ID NUMBER,
                          vQUEUE_ID NUMBER, 
                          vSUBSCRIBER_ID NUMBER) IS
      SELECT * FROM aqmon_bsubscribers_tab 
       WHERE INST_ID       = bs_statsCursor.vINST_ID
         AND QUEUE_ID      = bs_statsCursor.vQUEUE_ID  
         AND SUBSCRIBER_ID = bs_statsCursor.vSUBSCRIBER_ID 
       ORDER BY memseq;

  BEGIN
    -- check input parameters
    IF db_version IS NULL OR inst_number IS NULL OR queue_iden IS NULL THEN
      RETURN;
    END IF;   

    IF NOT UTL_FILE.IS_OPEN(output_file) THEN
      prvt_echo_error('No valid log file.');
      RETURN;
    END IF;


    IF db_version >= 11.2 THEN 

      title_line1 := lpad(' Per-Period Metrics ', 66, '-') || 
                     lpad(' | ', 53, '-') ||
                     lpad(' Accumulative Metrics ', 56, '-') || 
                     lpad(' | ', 37, '-') ||
                     lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue   Avg Time per';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)    Dequeue(us)';

      title_line2_p2 := '      Avg CPU Time  |  Pending             Oldest        Oldest Msg Enqueue Time              Last Dequeue Time  |  Timestamp';
      title_line3_p2 := '   per Dequeue(us)  |     Msgs              MsgID                 (UTC timezone)                 (UTC timezone)  |  ';

    ELSIF db_version = 10.2 OR db_version = 11.1 THEN

      title_line1 := lpad(' Per-Period Metrics ', 50, '-') || 
                     lpad(' | ', 36, '-') ||
                     '-- Accum Metrics -- |' ||
                     lpad('-', 30, '-') ; 

      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)';

      title_line2_p2 := '  |            Pending  |  Timestamp';
      title_line3_p2 := '  |               Msgs  |  ';

    END IF;

    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    UTL_FILE.PUT_LINE(output_file, ' ');
    
    EXECUTE IMMEDIATE 'SELECT count(distinct SUBSCRIBER_ID)' ||
                      '  FROM aqmon_bsubscribers_tab ' ||
                      ' WHERE INST_ID   = ' || inst_number ||
                      '   AND QUEUE_ID  = ' || queue_iden ||
                      '   AND SUBSCRIBER_ID IS NOT NULL' 
      INTO sub_num;

    IF sub_num = 0 THEN
      tmpVar := 'No subscriber found.';
    ELSIF sub_num = 1 THEN
      tmpVar := '1 subscriber found.';
    ELSE
      tmpVar := sub_num || ' subscribers found.';
    END IF;

    UTL_FILE.PUT_LINE(output_file, 'Subscriber Statistics : ' || tmpVar);
    UTL_FILE.PUT_LINE(output_file, ' ');

    -- loop through different subscribers
    FOR i IN bs_subCursor(inst_number, queue_iden) LOOP
      IF bs_subCursor%ROWCOUNT > 1 THEN 
        UTL_FILE.PUT_LINE(output_file, ' ');
        UTL_FILE.PUT_LINE(output_file, ' ');
        UTL_FILE.PUT_LINE(output_file, ' ');
      END IF;

      UTL_FILE.PUT_LINE(output_file, '  Subscriber ID       : ' || 
                                      i.SUBSCRIBER_ID);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Name     : ' || 
                                      i.SUBSCRIBER_NAME);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Address  : ' || 
                                      i.SUBSCRIBER_ADDRESS);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Protocol : ' || 
                                      i.PROTOCOL);
      UTL_FILE.PUT_LINE(output_file, '  Subscriber Type     : ' || 
                                      i.SUBSCRIBER_TYPE);

      -- get STARTUP_TIME
      SELECT min(STARTUP_TIME) INTO tmpTime
        FROM aqmon_bsubscribers_tab
       WHERE INST_ID         = inst_number
         AND QUEUE_ID        = queue_iden
         AND SUBSCRIBER_ID   = i.SUBSCRIBER_ID
         AND SUBSCRIBER_NAME = i.SUBSCRIBER_NAME
         AND STARTUP_TIME IS NOT NULL;

      UTL_FILE.PUT_LINE(output_file, '  Startup Time        : ' || tmpTime); 
      UTL_FILE.PUT_LINE(output_file, ' ');    
     
      first_row := TRUE;
      -- loop through runtime stats cursor
      FOR j in bs_statsCursor(inst_number, queue_iden, i.SUBSCRIBER_ID) LOOP

        -- fetch metrics for 11.2 or above
        IF db_version >= 11.2 THEN   
          BEGIN
            EXECUTE IMMEDIATE 'SELECT ELAPSED_DEQUEUE_TIME, ' ||
                      '          DEQUEUE_CPU_TIME, LAST_DEQUEUE_TIME, ' ||
                      '          OLDEST_MSGID, OLDEST_MSG_ENQTM' ||
                      '  FROM aqmon_bsubscribers_tab ' ||
                      ' WHERE INST_ID       = ' || inst_number ||
                      '   AND QUEUE_ID      = ' || queue_iden ||
                      '   AND SUBSCRIBER_ID = ' || j.SUBSCRIBER_ID ||
                      '   AND memseq        = ' || j.memseq
              INTO curr_elapsed_dequeue_time, 
                curr_dequeue_cpu_time, my_last_deqtm_accum,
                my_oldest_msgid_accum, my_oldest_msg_enqtm_accum;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          curr_elapsed_dequeue_time := nvl(curr_elapsed_dequeue_time, 0); 
          curr_dequeue_cpu_time     := nvl(curr_dequeue_cpu_time, 0);

        END IF;


        IF first_row THEN 
          first_row             := FALSE;

          UTL_FILE.PUT_LINE(output_file, title_line1);
          UTL_FILE.PUT_LINE(output_file, title_line2_p1 || title_line2_p2 );
          UTL_FILE.PUT_LINE(output_file, title_line3_p1 || title_line3_p2 );
          UTL_FILE.PUT_LINE(output_file, ' ');
        ELSE       
          -- unit: second
          my_time_duration      := (CAST(j.MONITOR_TIME as DATE)-
                                    CAST(pre_monitor_time as DATE))*86400.0; 

          my_enqueued_msgs_td   := nvl(j.CNUM_MSGS,0) - pre_cnum_msgs;
          my_dequeued_msgs_td   := nvl(j.TOTAL_DEQUEUED_MSG,0) 
                                   - pre_total_dequeued_msg;
          my_spilled_msgs_td    := nvl(j.TOTAL_SPILLED_MSG,0)
                                   - pre_total_spilled_msg;
          my_expired_msgs_td    := nvl(j.EXPIRED_MSGS,0) - pre_expired_msgs; 

       
          IF my_time_duration = 0 THEN
            my_enqueue_rate_td := 0;
            my_dequeue_rate_td := 0;
          ELSE 
            my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
            my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
          END IF;
        
          -- accum metrics
          my_pending_msgs_accum := nvl(j.CNUM_MSGS, 0) - 
                                   nvl(j.TOTAL_DEQUEUED_MSG, 0);

          -- metrics for 11.2 or above
          IF db_version >= 11.2 THEN   
            IF my_dequeued_msgs_td = 0 THEN 
              my_avg_time_per_deq_td := 0;
              my_cpu_time_per_deq_td := 0;
            ELSE
              my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                      pre_elapsed_dequeue_time)/my_dequeued_msgs_td;
              my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                      pre_dequeue_cpu_time)/my_dequeued_msgs_td;
            END IF;  
          END IF;
          -- end metrics for 11.2 or above


          IF db_version >= 11.2 THEN 
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_spilled_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'),14)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'),17)
                           || '  |' ||  -- from centi-second to micro-second
              lpad(my_pending_msgs_accum, 9)                        || ' ' ||
              lpad(nvl(to_char(my_oldest_msgid_accum), ' '), 18)    || ' ' ||
              lpad(nvl(to_char(my_oldest_msg_enqtm_accum), ' '), 30)|| ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || '  |  ' ||
              j.MONITOR_TIME;
          ELSIF db_version=10.2 OR db_version=11.1 THEN
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_spilled_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'),13)  || ' ' ||
              lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'),13)|| '  |' || 
              lpad(my_pending_msgs_accum, 19)                   || '  |  ' ||
              j.MONITOR_TIME;
          END IF;

          output_line := lpad(j.memseq, 5) || ' ' ||   
                         lpad(to_char(my_time_duration, 'FM999999'), 6) || 
                         ' ' || tmpVar;

          UTL_FILE.PUT_LINE(output_file, output_line);          
        END IF; -- end if first_row

        pre_monitor_time           := j.MONITOR_TIME;
        pre_cnum_msgs              := nvl(j.CNUM_MSGS, 0);
        pre_total_dequeued_msg     := nvl(j.TOTAL_DEQUEUED_MSG, 0);
        pre_total_spilled_msg      := nvl(j.TOTAL_SPILLED_MSG, 0);
        pre_expired_msgs           := nvl(j.EXPIRED_MSGS, 0);

        IF db_version >= 11.2 THEN  
          pre_elapsed_dequeue_time := curr_elapsed_dequeue_time;
          pre_dequeue_cpu_time     := curr_dequeue_cpu_time;
        END IF;  

      END LOOP; -- FOR j in bs_statsCursor LOOP
    END LOOP; -- FOR i IN bs_subCursor LOOP
   
    UTL_FILE.PUT_LINE(output_file, ' ');
  END prvt_get_bsubscribers;



  -- Generate detailed statistics report for a buffered queue.
  -- (valid for 10.2, 11.1, 11.2)
  PROCEDURE prvt_get_bqueue_report(db_version     IN NUMBER,
                                   inst_number    IN NUMBER,
                                   queue_iden     IN NUMBER,
                                   single_logfile IN BOOLEAN) AS
    first_row BOOLEAN := TRUE;

    -- for 10.2/11.1
    -- statistics from the view    
    first_monitor_time           TIMESTAMP;
    first_num_msgs               NUMBER;
    first_cnum_msgs              NUMBER;
    first_cspill_msgs            NUMBER;
    first_expired_msgs           NUMBER;
        
    pre_monitor_time             TIMESTAMP;
    pre_num_msgs                 NUMBER; 
    pre_cnum_msgs                NUMBER;
    pre_cspill_msgs              NUMBER; 
    pre_expired_msgs             NUMBER; 

    -- statistics computed (in a certain time duration)
    my_time_duration             NUMBER;
    my_enqueued_msgs_td          NUMBER;
    my_dequeued_msgs_td          NUMBER;
    my_spilled_msgs_td           NUMBER;
    my_expired_msgs_td           NUMBER;
    my_enqueue_rate_td           NUMBER; 
    my_dequeue_rate_td           NUMBER; 

    -- the following metrics show accumulative stats, not in a certain 
    -- time duration
    my_pending_msgs_accum        NUMBER; 
    -- end for 10.2/11.1

    -- for 11.2 or above
    first_elapsed_enqueue_time   NUMBER;
    first_elapsed_dequeue_time   NUMBER;
    first_elapsed_trans_time     NUMBER;
    first_elapsed_rule_eval_time NUMBER;
    first_enqueue_cpu_time       NUMBER;
    first_dequeue_cpu_time       NUMBER;

    pre_elapsed_enqueue_time     NUMBER;
    pre_elapsed_dequeue_time     NUMBER;
    pre_elapsed_trans_time       NUMBER;
    pre_elapsed_rule_eval_time   NUMBER;
    pre_enqueue_cpu_time         NUMBER;
    pre_dequeue_cpu_time         NUMBER;

    curr_elapsed_enqueue_time    NUMBER;
    curr_elapsed_dequeue_time    NUMBER;
    curr_elapsed_trans_time      NUMBER;
    curr_elapsed_rule_eval_time  NUMBER;
    curr_enqueue_cpu_time        NUMBER;
    curr_dequeue_cpu_time        NUMBER;
  
    my_avg_time_per_enq_td       NUMBER;
    my_avg_time_per_deq_td       NUMBER;
    my_cpu_time_per_enq_td       NUMBER;
    my_cpu_time_per_deq_td       NUMBER;
    my_trans_time_td             NUMBER;
    my_rule_eval_time_td         NUMBER;

    my_oldest_msgid_accum        RAW(16);
    my_oldest_msg_enqtm_accum    TIMESTAMP;
    my_last_enqtm_accum          TIMESTAMP;
    my_last_deqtm_accum          TIMESTAMP;
    -- end for 11.2 or above 

    title_line1        VARCHAR2(1000); 
    title_line2_p1     VARCHAR2(1000);
    title_line3_p1     VARCHAR2(1000);

    title_line2_p2     VARCHAR2(1000);
    title_line3_p2     VARCHAR2(1000);

    title_line2_p3     VARCHAR2(1000);
    title_line3_p3     VARCHAR2(1000);

    output_line        VARCHAR2(1000);
    tmpVar             VARCHAR2(1000);
    tmpInterval        NUMBER;        
    queue_filename     VARCHAR2(200);
    queue_file         UTL_FILE.FILE_TYPE;      
    plot_filename      VARCHAR2(200);
    plot_file          UTL_FILE.FILE_TYPE;      
    conf_line          aqmon_queue_config_tab%ROWTYPE;
    max_memseq         NUMBER;    
    my_startup_time    TIMESTAMP;
    my_queue_state     VARCHAR2(25);
    freq               NUMBER;
    plural             VARCHAR2(2);
    myaccumfreq        NUMBER;
    dump_plottingfile  NUMBER;
    tmpTimeDuration    NUMBER;

    -- cursor for runtime stats
    CURSOR bq_runtmCursor(vINST_ID NUMBER, vQUEUE_ID NUMBER) IS
      SELECT * FROM aqmon_bqueues_tab 
       WHERE INST_ID  = bq_runtmCursor.vINST_ID
         AND QUEUE_ID = bq_runtmCursor.vQUEUE_ID  
       ORDER BY memseq;

    -- cursor for accumulative stats
    CURSOR bq_accumCursor(vINST_ID NUMBER, vQUEUE_ID NUMBER) IS
      SELECT * FROM aqmon_bqueues_tab 
       WHERE INST_ID  = bq_accumCursor.vINST_ID
         AND QUEUE_ID = bq_accumCursor.vQUEUE_ID
       ORDER BY memseq;            
 
    TYPE bq_refcur IS REF CURSOR;
    bq_myC1 bq_refcur;
    
  BEGIN
    IF db_version IS NULL OR inst_number IS NULL OR queue_iden IS NULL 
       OR single_logfile IS NULL THEN
      RETURN;
    END IF;

    -- adjust collected data (e.g., first snapshot) before analysis
    prvt_postrun_adjust(db_version, inst_number, queue_iden, FALSE);


    -- open log file 1
    IF single_logfile THEN 
      BEGIN
        queue_file := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_ALLQUEUES_REP_FNAME,
                                     'a', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;   
    ELSE
      queue_filename := 'aqmon_queue_' || inst_number || '_' || 
                        queue_iden || '.log';
      BEGIN
        queue_file := UTL_FILE.FOPEN(AQMON_LOGDIR, queue_filename, 'w', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;
    END IF;

    IF NOT UTL_FILE.IS_OPEN(queue_file) THEN
      prvt_echo_error('Cannot create log file in the log directory.');  
      RETURN;
    END IF;


    -- open log file 2 if necessary
    dump_plottingfile := prvt_get_param('APT_DUMP_PLOTTING_FILE');
    IF dump_plottingfile IS NOT NULL AND dump_plottingfile = 1 THEN
      plot_filename := 'aqmon_queue_' || inst_number || '_' || 
                       queue_iden || '.' || db_version || 
                       '.buffered.gnuplot';
      BEGIN
        plot_file := UTL_FILE.FOPEN(AQMON_LOGDIR, plot_filename, 'w', 5000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create log file in the log directory.');  
      END;

      IF NOT UTL_FILE.IS_OPEN(plot_file) THEN
        prvt_echo_error('Cannot create log file in the log directory.');  
        RETURN;
      END IF;
    END IF;


    IF db_version >= 11.2 THEN 

      title_line1 := lpad(' Per-Period Metrics ', 96, '-') || 
                     lpad(' | ', 85, '-') ||
                     lpad(' Accumulative Metrics ', 71, '-') || 
                     lpad(' | ', 53, '-') ||
                     lpad('-', 30, '-') ; 
      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue   Avg Time per';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)    Enqueue(us)';

      title_line2_p2 := '   Avg Time per      Avg CPU Time      Avg CPU Time   Transformation   Rule Eval  |  Pending             Oldest';
      title_line3_p2 := '    Dequeue(us)   per Enqueue(us)   per Dequeue(us)         Time(us)    Time(us)  |     Msgs              MsgID';

      title_line2_p3 := '        Oldest Msg Enqueue Time              Last Enqueue Time              Last Dequeue Time  |  Timestamp';
      title_line3_p3 := '                 (UTC timezone)                 (UTC timezone)                 (UTC timezone)  |  ';
    
    ELSIF db_version=10.2 OR db_version=11.1 THEN
    
      title_line1 := lpad(' Per-Period Metrics ', 54, '-') || 
                     lpad(' | ', 32, '-') ||
                     '-- Accum Metrics -- |' ||
                     lpad('-', 30, '-') ; 
      title_line2_p1 := ' Snap   Time   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue';
      title_line3_p1 := '  Seq    (s)       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)';

      title_line2_p2 := '  |            Pending';
      title_line3_p2 := '  |               Msgs';

      title_line2_p3 := '  |  Timestamp';
      title_line3_p3 := '  |  ';

    END IF;

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, lpad('%', 50, '%') || '  Report Start  '
                               || lpad('%', 50, '%'));  
    UTL_FILE.PUT_LINE(queue_file, ' ');

    UTL_FILE.PUT_LINE(queue_file, 'Buffered Queue Statistics');
    UTL_FILE.PUT_LINE(queue_file, ' ');

    BEGIN
      SELECT * INTO conf_line FROM aqmon_queue_config_tab
       WHERE QUEUE_ID = queue_iden;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        conf_line := NULL;
      WHEN TOO_MANY_ROWS THEN 
        -- this should not happen, unless the table 
        -- is manually manipulated.
        conf_line := NULL; 
    END;
   
    -- identify STARTUP_TIME
    tmpVar := 'SELECT STARTUP_TIME FROM aqmon_bqueues_tab ' ||
              ' WHERE INST_ID  = ' || inst_number ||
              '   AND QUEUE_ID = ' || queue_iden ||
              '   AND STARTUP_TIME IS NOT NULL';
    OPEN bq_myC1 for tmpVar;
    LOOP 
      FETCH bq_myC1 INTO my_startup_time;
      EXIT;
    END LOOP;
    CLOSE bq_myC1;


    IF conf_line.QUEUE_ID IS NOT NULL THEN
      UTL_FILE.PUT_LINE(queue_file, '  Database Version    : ' || 
                                       db_version);      
      UTL_FILE.PUT_LINE(queue_file, '  Instance Number     : ' || 
                                       inst_number);
      UTL_FILE.PUT_LINE(queue_file, '  Queue ID            : ' || 
                                       conf_line.QUEUE_ID);
      UTL_FILE.PUT_LINE(queue_file, '  Queue Name          : ' || 
                                       conf_line.QUEUE_NAME);
      UTL_FILE.PUT_LINE(queue_file, '  Queue Table Name    : ' || 
                                       conf_line.QUEUE_TABLE); 
      UTL_FILE.PUT_LINE(queue_file, '  Queue Owner         : ' || 
                                       conf_line.QUEUE_OWNER);    
      UTL_FILE.PUT_LINE(queue_file, '  Queue Type          : ' || 
                                       conf_line.QUEUE_TYPE);
      IF db_version >= 11.1 THEN
        tmpVar := 'SELECT QUEUE_STATE FROM aqmon_bqueues_tab ' ||
                  ' WHERE INST_ID  = ' || inst_number ||
                  '   AND QUEUE_ID = ' || queue_iden;
        OPEN bq_myC1 for tmpVar;
        LOOP 
          FETCH bq_myC1 INTO my_queue_state;
          EXIT;
        END LOOP;
        CLOSE bq_myC1;

        UTL_FILE.PUT_LINE(queue_file, '  Queue State         : ' || 
                                         my_queue_state);  
      END IF;

      UTL_FILE.PUT_LINE(queue_file, '  Data Type           : ' || 
                                       conf_line.DATA_TYPE);
      UTL_FILE.PUT_LINE(queue_file, '  Object Type         : ' || 
                                       conf_line.OBJECT_TYPE);
      UTL_FILE.PUT_LINE(queue_file, '  Sort Order          : ' || 
                                       conf_line.SORT_ORDER);
      UTL_FILE.PUT_LINE(queue_file, '  Recipients          : ' || 
                                       conf_line.RECIPIENTS);
      UTL_FILE.PUT_LINE(queue_file, '  Message Grouping    : ' || 
                                       conf_line.MESSAGE_GROUPING);
      UTL_FILE.PUT_LINE(queue_file, '  Startup Time        : ' ||
                                       my_startup_time);
    END IF;

    UTL_FILE.PUT_LINE(queue_file, ' ');    
    UTL_FILE.PUT_LINE(queue_file, ' ');


    -- display frequency info
    freq := prvt_get_param('APT_COLLECT_RUNTIME_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- Runtime snapshots are taken every ' ||
                                     freq || ' second' || plural || '.');
    END IF;


    freq := prvt_get_param('APT_COLLECT_ACCUM_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- Accumulative statistics are ' ||
                 'calculated every ' || freq || ' second' || plural || '.');
      myaccumfreq := freq; 
    END IF;


    freq := prvt_get_param('APT_COLLECT_CONFIG_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      IF freq = INFINITE_TIME THEN
        UTL_FILE.PUT_LINE(queue_file, ' -- Queue configuation statistics are ' ||
                       'taken at the monitor starting time (once).');
      ELSE
        UTL_FILE.PUT_LINE(queue_file, ' -- Queue configuation statistics are ' ||
                       'taken every ' || freq || ' second' || plural || '.');
      END IF;
    END IF;

    freq := prvt_get_param('APT_DUMP_AWR_FREQ');
    IF freq IS NOT NULL THEN   
      IF freq > 1 THEN plural := 's';
      ELSE             plural := NULL;    
      END IF;
      UTL_FILE.PUT_LINE(queue_file, ' -- AWR reports are taken every ' || 
                                     freq || ' second' || plural || '.');
    END IF;


    IF conf_line.RECIPIENTS IS NOT NULL AND 
       UPPER(conf_line.RECIPIENTS) = 'MULTIPLE' THEN
      UTL_FILE.PUT_LINE(queue_file, ' ');
      UTL_FILE.PUT_LINE(queue_file, ' -- NOTE: For a multiple-subscriber ' ||
                                  'queue, avg/cpu dequeue time computed at ');
      UTL_FILE.PUT_LINE(queue_file, ' --       the queue level might not ' ||
                                  'be accurate. Please refer to subscriber');
      UTL_FILE.PUT_LINE(queue_file, ' --       level dequeue time for ' ||
                                  'accurate computation. ');
    END IF;

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, '> Runtime Statistics');
    UTL_FILE.PUT_LINE(queue_file, ' ');

    first_row := TRUE;

    -- loop through runtime stats cursor
    FOR j IN bq_runtmCursor(inst_number, queue_iden) LOOP

      IF db_version >= 11.2 THEN
        BEGIN 
          tmpVar := 'SELECT ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, '||
                    '       ELAPSED_TRANSFORMATION_TIME, ' ||
                    '       ELAPSED_RULE_EVALUATION_TIME, ' ||
                    '       ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME, ' || 
                    '       LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
                    '       OLDEST_MSGID, OLDEST_MSG_ENQTM' ||
                    '  FROM aqmon_bqueues_tab ' ||
                    ' WHERE INST_ID  = ' || inst_number ||
                    '   AND QUEUE_ID = ' || queue_iden ||
                    '   AND memseq   = ' || j.memseq; 
          EXECUTE IMMEDIATE tmpVar INTO curr_elapsed_enqueue_time,
            curr_elapsed_dequeue_time, curr_elapsed_trans_time, 
            curr_elapsed_rule_eval_time, curr_enqueue_cpu_time,
            curr_dequeue_cpu_time, my_last_enqtm_accum, my_last_deqtm_accum,
            my_oldest_msgid_accum, my_oldest_msg_enqtm_accum;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        curr_elapsed_enqueue_time   := nvl(curr_elapsed_enqueue_time, 0);
        curr_elapsed_dequeue_time   := nvl(curr_elapsed_dequeue_time, 0);
        curr_elapsed_trans_time     := nvl(curr_elapsed_trans_time, 0); 
        curr_elapsed_rule_eval_time := nvl(curr_elapsed_rule_eval_time, 0); 
        curr_enqueue_cpu_time       := nvl(curr_enqueue_cpu_time, 0); 
        curr_dequeue_cpu_time       := nvl(curr_dequeue_cpu_time, 0);

      END IF;

      IF first_row THEN 
        first_monitor_time  := j.MONITOR_TIME;
        first_row           := FALSE;

        UTL_FILE.PUT_LINE(queue_file, title_line1);
        UTL_FILE.PUT_LINE(queue_file, title_line2_p1 || title_line2_p2 || 
                                      title_line2_p3);
        UTL_FILE.PUT_LINE(queue_file, title_line3_p1 || title_line3_p2 || 
                                      title_line3_p3);
        UTL_FILE.PUT_LINE(queue_file, ' ');
      ELSE       
        -- unit: second
        my_time_duration     := (CAST(j.MONITOR_TIME as DATE)-
                                 CAST(pre_monitor_time as DATE))*86400.0; 

        my_enqueued_msgs_td  := nvl(j.CNUM_MSGS,0)   - pre_cnum_msgs;
        my_spilled_msgs_td   := nvl(j.CSPILL_MSGS,0) - pre_cspill_msgs;
        my_expired_msgs_td   := nvl(j.EXPIRED_MSGS,0)- pre_expired_msgs; 

        -- accum metrics
        my_pending_msgs_accum      := nvl(j.NUM_MSGS, 0);
        -- accum metrics end
       
        my_dequeued_msgs_td := my_enqueued_msgs_td - 
                            (my_pending_msgs_accum - pre_num_msgs);
          
        IF my_time_duration = 0 THEN
          my_enqueue_rate_td := 0;
          my_dequeue_rate_td := 0;
        ELSE 
          my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
          my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
        END IF;
              

        -- metrics for 11.2 or above
        IF db_version >= 11.2 THEN
          my_trans_time_td     := curr_elapsed_trans_time  
                                - pre_elapsed_trans_time;
          my_rule_eval_time_td := curr_elapsed_rule_eval_time 
                                - pre_elapsed_rule_eval_time; 

          IF my_enqueued_msgs_td = 0 THEN 
            my_avg_time_per_enq_td := 0;
            my_cpu_time_per_enq_td := 0;
          ELSE
            my_avg_time_per_enq_td := (curr_elapsed_enqueue_time-
                    pre_elapsed_enqueue_time)/my_enqueued_msgs_td;
            my_cpu_time_per_enq_td := (curr_enqueue_cpu_time-
                    pre_enqueue_cpu_time)/my_enqueued_msgs_td;
          END IF;

          IF my_dequeued_msgs_td = 0 THEN 
            my_avg_time_per_deq_td := 0;
            my_cpu_time_per_deq_td := 0;
          ELSE
            my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                    pre_elapsed_dequeue_time)/my_dequeued_msgs_td;
            my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                    pre_dequeue_cpu_time)/my_dequeued_msgs_td;
          END IF;
        ELSE
          my_trans_time_td       := 0;
          my_rule_eval_time_td   := 0;
          my_avg_time_per_enq_td := 0;
          my_cpu_time_per_enq_td := 0;
          my_avg_time_per_deq_td := 0;
          my_cpu_time_per_deq_td := 0;

          my_oldest_msgid_accum      := NULL;
          my_oldest_msg_enqtm_accum  := NULL;
          my_last_enqtm_accum        := NULL;
          my_last_deqtm_accum        := NULL;
        END IF;
        -- end metrics for 11.2 or above

        IF db_version >= 11.2 THEN
          tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
            lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
            lpad(my_spilled_msgs_td,9)                            || ' ' ||
            lpad(my_expired_msgs_td,9)                            || ' ' ||
            lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'), 14) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'), 14) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'), 17) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'), 17) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'), 16) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_rule_eval_time_td*10000, 'FM99999999.00'), 11) 
                         || '  |' ||  -- from centi-second to micro-second
            lpad(my_pending_msgs_accum, 9)                        || ' ' ||
            lpad(nvl(to_char(my_oldest_msgid_accum), ' '), 18)    || ' ' ||
            lpad(nvl(to_char(my_oldest_msg_enqtm_accum), ' '), 30)|| ' ' ||
            lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)      || ' ' ||
            lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || '  |  ' ||
            j.MONITOR_TIME;
        ELSIF db_version=10.2 OR db_version=11.1 THEN
          tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
            lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
            lpad(my_spilled_msgs_td,9)                            || ' ' ||
            lpad(my_expired_msgs_td,9)                            || ' ' ||
            lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'),13)  || ' ' ||
            lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'),13)|| '  |' ||
            lpad(my_pending_msgs_accum, 19)                    || '  |  ' ||
            j.MONITOR_TIME;
        ELSE
          tmpVar := NULL;
        END IF;
  
        output_line := lpad(j.memseq, 5) || ' ' ||   
                       lpad(to_char(my_time_duration, 'FM999999'), 6)|| ' ' ||
                       tmpVar;

        UTL_FILE.PUT_LINE(queue_file, output_line);     


        -- dump stats for gnuplot. Each queue has a specific file 
        IF dump_plottingfile IS NOT NULL AND dump_plottingfile = 1 THEN
   
          -- record accumulative time duration (unit: second)
          tmpTimeDuration := (CAST(j.MONITOR_TIME as DATE)-
                              CAST(first_monitor_time as DATE))*86400.0;
              
          tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
            lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
            lpad(my_spilled_msgs_td,9)                            || ' ' ||
            lpad(my_expired_msgs_td,9)                            || ' ' ||
            lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'), 14) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'), 14) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'), 17) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'), 17) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'), 16) 
                         || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(my_rule_eval_time_td*10000, 'FM99999999.00'), 11) 
                         || '  ' ||  -- from centi-second to micro-second
            lpad(my_pending_msgs_accum, 9)                      || '   ' ||
            j.MONITOR_TIME;

          UTL_FILE.PUT_LINE(plot_file, lpad(to_char(tmpTimeDuration, 
                                        'FM999999'), 6) || '   ' || tmpVar);
        END IF;
     
      END IF; -- end if first_row

      pre_monitor_time           := j.MONITOR_TIME;
      pre_num_msgs               := nvl(j.NUM_MSGS, 0);
      pre_cnum_msgs              := nvl(j.CNUM_MSGS, 0);  
      pre_cspill_msgs            := nvl(j.CSPILL_MSGS, 0);
      pre_expired_msgs           := nvl(j.EXPIRED_MSGS, 0);

      IF db_version >= 11.2 THEN 
        pre_elapsed_enqueue_time   := curr_elapsed_enqueue_time;
        pre_elapsed_dequeue_time   := curr_elapsed_dequeue_time;
        pre_elapsed_trans_time     := curr_elapsed_trans_time;
        pre_elapsed_rule_eval_time := curr_elapsed_rule_eval_time;   
        pre_enqueue_cpu_time       := curr_enqueue_cpu_time;
        pre_dequeue_cpu_time       := curr_dequeue_cpu_time;
      END IF;

    END LOOP;  -- FOR j IN bq_runtmCursor LOOP

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, ' ');


    -- dump accumulative statistis at certain time
    first_row := TRUE;
    SELECT max(memseq) INTO max_memseq from aqmon_bqueues_tab; 
    IF max_memseq IS NULL THEN
      max_memseq := -1;
    END IF;

    UTL_FILE.PUT_LINE(queue_file, '> Accumulative Runtime Statistics');
    UTL_FILE.PUT_LINE(queue_file, ' ');

    -- loop through accumulative stats cursor
    FOR j IN bq_accumCursor(inst_number, queue_iden) LOOP

      IF db_version >= 11.2 THEN
        BEGIN 
          tmpVar := 'SELECT ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, '||
                    '       ELAPSED_TRANSFORMATION_TIME, ' ||
                    '       ELAPSED_RULE_EVALUATION_TIME, ' ||
                    '       ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME, ' || 
                    '       LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
                    '       OLDEST_MSGID, OLDEST_MSG_ENQTM' ||
                    '  FROM aqmon_bqueues_tab ' ||
                    ' WHERE INST_ID  = ' || inst_number ||
                    '   AND QUEUE_ID = ' || queue_iden ||
                    '   AND memseq   = ' || j.memseq; 
          EXECUTE IMMEDIATE tmpVar INTO curr_elapsed_enqueue_time,
            curr_elapsed_dequeue_time, curr_elapsed_trans_time, 
            curr_elapsed_rule_eval_time, curr_enqueue_cpu_time,
            curr_dequeue_cpu_time, my_last_enqtm_accum, my_last_deqtm_accum,
            my_oldest_msgid_accum, my_oldest_msg_enqtm_accum;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        curr_elapsed_enqueue_time   := nvl(curr_elapsed_enqueue_time, 0);
        curr_elapsed_dequeue_time   := nvl(curr_elapsed_dequeue_time, 0);
        curr_elapsed_trans_time     := nvl(curr_elapsed_trans_time, 0); 
        curr_elapsed_rule_eval_time := nvl(curr_elapsed_rule_eval_time, 0); 
        curr_enqueue_cpu_time       := nvl(curr_enqueue_cpu_time, 0); 
        curr_dequeue_cpu_time       := nvl(curr_dequeue_cpu_time, 0);

      END IF;

      IF first_row THEN
        first_monitor_time           := j.MONITOR_TIME;
        first_num_msgs               := nvl(j.NUM_MSGS, 0);
        first_cnum_msgs              := nvl(j.CNUM_MSGS, 0);  
        first_cspill_msgs            := nvl(j.CSPILL_MSGS, 0);
        first_expired_msgs           := nvl(j.EXPIRED_MSGS, 0);

        IF db_version >= 11.2 THEN
          first_elapsed_enqueue_time   := curr_elapsed_enqueue_time;
          first_elapsed_dequeue_time   := curr_elapsed_dequeue_time;
          first_elapsed_trans_time     := curr_elapsed_trans_time;
          first_elapsed_rule_eval_time := curr_elapsed_rule_eval_time;
          first_enqueue_cpu_time       := curr_enqueue_cpu_time;
          first_dequeue_cpu_time       := curr_dequeue_cpu_time;
        END IF;

        first_row := FALSE;     
        tmpInterval := myaccumfreq;

        UTL_FILE.PUT_LINE(queue_file, title_line2_p1 || title_line2_p2 || 
                                      title_line2_p3);
        UTL_FILE.PUT_LINE(queue_file, title_line3_p1 || title_line3_p2 || 
                                      title_line3_p3); 
        UTL_FILE.PUT_LINE(queue_file, ' ');                   
      ELSE
        -- unit: second
        my_time_duration := (CAST(j.MONITOR_TIME as DATE)-
                             CAST(first_monitor_time as DATE))*86400.0;

        -- decide whether to dump the line          
        IF my_time_duration > tmpInterval OR j.MEMSEQ = max_memseq THEN 

          my_enqueued_msgs_td  := nvl(j.CNUM_MSGS,0)   - first_cnum_msgs;
          my_spilled_msgs_td   := nvl(j.CSPILL_MSGS,0) - first_cspill_msgs;
          my_expired_msgs_td   := nvl(j.EXPIRED_MSGS,0)- first_expired_msgs; 

          -- accum metrics
          my_pending_msgs_accum      := nvl(j.NUM_MSGS, 0);
          -- accum metrics end

          my_dequeued_msgs_td := my_enqueued_msgs_td - 
                              (my_pending_msgs_accum - first_num_msgs);

          IF my_time_duration = 0 THEN
            my_enqueue_rate_td := 0;
            my_dequeue_rate_td := 0;
          ELSE 
            my_enqueue_rate_td  := my_enqueued_msgs_td/my_time_duration; 
            my_dequeue_rate_td  := my_dequeued_msgs_td/my_time_duration; 
          END IF;
        
          -- metrics for 11.2 or above
          IF db_version >= 11.2 THEN
            my_trans_time_td     := curr_elapsed_trans_time  
                                  - first_elapsed_trans_time;
            my_rule_eval_time_td := curr_elapsed_rule_eval_time
                                  - first_elapsed_rule_eval_time;

            IF my_enqueued_msgs_td = 0 THEN 
              my_avg_time_per_enq_td := 0;
              my_cpu_time_per_enq_td := 0;
            ELSE
              my_avg_time_per_enq_td := (curr_elapsed_enqueue_time-
                      first_elapsed_enqueue_time)/my_enqueued_msgs_td;
              my_cpu_time_per_enq_td := (curr_enqueue_cpu_time-
                      first_enqueue_cpu_time)/my_enqueued_msgs_td;
            END IF;

            IF my_dequeued_msgs_td = 0 THEN 
              my_avg_time_per_deq_td := 0;
              my_cpu_time_per_deq_td := 0;
            ELSE
              my_avg_time_per_deq_td := (curr_elapsed_dequeue_time-
                      first_elapsed_dequeue_time)/my_dequeued_msgs_td;
              my_cpu_time_per_deq_td := (curr_dequeue_cpu_time-
                      first_dequeue_cpu_time)/my_dequeued_msgs_td;
            END IF;

          END IF;
          -- end metrics for 11.2 or above

          IF db_version >= 11.2 THEN 
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_spilled_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(my_avg_time_per_enq_td*10000, 'FM99999999.00'),14)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_avg_time_per_deq_td*10000, 'FM99999999.00'),14)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_enq_td*10000, 'FM99999999.00'),17)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_cpu_time_per_deq_td*10000, 'FM99999999.00'),17)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_trans_time_td*10000, 'FM99999999.00'),16)
                           || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(my_rule_eval_time_td*10000, 'FM99999999.00'),11)
                           || '  |' ||  -- from centi-second to micro-second
              lpad(my_pending_msgs_accum, 9)                        || ' ' ||
              lpad(nvl(to_char(my_oldest_msgid_accum), ' '), 18)    || ' ' ||
              lpad(nvl(to_char(my_oldest_msg_enqtm_accum), ' '), 30)|| ' ' ||
              lpad(nvl(to_char(my_last_enqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(my_last_deqtm_accum), ' '), 30)  || '  |  ' ||
              j.MONITOR_TIME;
          ELSIF db_version=10.2 OR db_version=11.1 THEN
            tmpVar := lpad(my_enqueued_msgs_td, 10)                 || ' ' ||
              lpad(my_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(my_spilled_msgs_td,9)                            || ' ' ||
              lpad(my_expired_msgs_td,9)                            || ' ' ||
              lpad(to_char(my_enqueue_rate_td, 'FM9999999.00'),13)  || ' ' ||
              lpad(to_char(my_dequeue_rate_td, 'FM9999999.00'),13)|| '  |' ||
              lpad(my_pending_msgs_accum, 19)                   || '  |  ' ||
              j.MONITOR_TIME;
          ELSE
            tmpVar := NULL;
          END IF;

          output_line := lpad(j.memseq, 5) || ' ' ||   
                       lpad(to_char(my_time_duration, 'FM999999'), 6)|| ' ' ||
                       tmpVar;
   
          UTL_FILE.PUT_LINE(queue_file, output_line);          

          --modify tmpInterval for next dump 
          tmpInterval := tmpInterval + myaccumfreq;

        END IF; -- if my_time_duration > tmpInterval OR j.MEMSEQ = max_memseq

      END IF; -- if first_row then ... else ...

    END LOOP; -- FOR j IN bq_accumCursor LOOP          

    UTL_FILE.PUT_LINE(queue_file, ' ');

    -- get subscribers info
    prvt_get_bsubscribers(db_version, inst_number, queue_iden, queue_file);

    -- get SQL/Segment info
    prvt_get_sql_seg(db_version, conf_line.QUEUE_TABLE, queue_file);

    UTL_FILE.PUT_LINE(queue_file, ' ');
    UTL_FILE.PUT_LINE(queue_file, lpad('%', 50, '%') || '   Report End   '
                               || lpad('%', 50, '%'));  
    UTL_FILE.PUT_LINE(queue_file, ' ');

    UTL_FILE.FCLOSE(queue_file);

    IF UTL_FILE.IS_OPEN(plot_file) THEN
      UTL_FILE.FCLOSE(plot_file);
    END IF;

    prvt_echo_line('Report for buffered queue ' || queue_iden ||
                   ' on instance ' || inst_number || ' is generated.'); 


  END prvt_get_bqueue_report;



  -- EXPORT PROCEDURE: generate a detailed stats report for a specific queue
  --                   on a specific instance.
  --
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_queue_report(queue_id       IN NUMBER DEFAULT NULL,
                             inst_number    IN NUMBER DEFAULT NULL,
                             final_collect  IN BOOLEAN DEFAULT TRUE,
                             single_logfile IN BOOLEAN DEFAULT FALSE) AS
    logdir_path    VARCHAR2(4000);
    cnt            NUMBER;
    queue_cnt      NUMBER;
    mon_status     NUMBER;
    myINST_NUM     NUMBER;
    db_version     NUMBER;   

    stxt           VARCHAR2(1000);
    TYPE gqr_refcur IS REF CURSOR;
    gqr_c1 gqr_refcur;

  BEGIN
    mon_status := prvt_get_param('APT_MONITOR_STATUS');
    IF mon_status IS NULL OR mon_status = AQMON_STATUS_NULL THEN
      prvt_echo_error('Monitor is not started yet.');
      RETURN;
    END IF;

    IF get_queue_report.queue_id IS NULL THEN
      prvt_echo_error('Queue ID cannot be NULL.');
      RETURN;
    ELSIF get_queue_report.queue_id <=0 THEN
      prvt_echo_error('Queue ID must be positive.');
      RETURN;  
    END IF;
   
    IF inst_number < -1 THEN
      prvt_echo_error('Instance number must be positive.');
      RETURN;  
    END IF;

    IF final_collect IS NULL THEN
      prvt_echo_error('Parameter final_collect cannot be NULL.');
      RETURN;  
    END IF;
   
    IF single_logfile IS NULL THEN
      prvt_echo_error('Parameter single_logfile cannot be NULL.');
      RETURN;
    END IF; 


    db_version := prvt_get_param('APT_DATABASE_VERSION');


    cnt := 0;
    queue_cnt := 0;
    IF db_version = 10.2 THEN

      -- for 10.2, aqmon_pqueues_tab does not exist, but aqmon_bqueues_tab
      -- exists.

      IF inst_number = -1 THEN -- search for all instances
        prvt_echo_error('Instance number must be specified or set' ||
                        ' to NULL in Oracle 10.2.');
        RETURN;
      ELSE -- for a specific instance

        myINST_NUM := inst_number;
        IF inst_number IS NULL THEN
          -- get the current instance number
          BEGIN
            SELECT i.instance_number INTO myINST_NUM FROM v$instance i;
          EXCEPTION
            WHEN OTHERS THEN
              myINST_NUM := -100;
          END;
        END IF;

        EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_queue_config_tab' ||
          ' WHERE QUEUE_ID = ' || get_queue_report.queue_id
          INTO cnt;
        IF cnt > 0 THEN

          EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_bqueues_tab' ||
            ' WHERE QUEUE_ID = ' || get_queue_report.queue_id
            INTO cnt;          
          IF cnt > 0 THEN   -- if it is a buffered queue
            EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_bqueues_tab' ||
              ' WHERE QUEUE_ID = ' || get_queue_report.queue_id ||
              '   AND INST_ID  = ' || myINST_NUM
              INTO cnt;          
         
            IF cnt > 0 THEN 
              -- dump one snapshot before generating report
              IF final_collect AND mon_status = AQMON_STATUS_START THEN
                collect_runtime;
                collect_config;
                DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
              END IF;

              prvt_get_bqueue_report(db_version, myINST_NUM, 
                get_queue_report.queue_id, single_logfile);
              queue_cnt := queue_cnt+1;
            END IF;
  
          ELSE   -- if it is not a buffered queue

            -- Nothing to check as aqmon_pqueues_tab does not exist. 
            prvt_get_pqueue_report(db_version, myINST_NUM, 
              get_queue_report.queue_id, single_logfile);  
            queue_cnt := queue_cnt+1;
          END IF;           
        END IF;  
      END IF; -- if inst_number = -1 ... else ...

    ELSIF db_version >= 11.1 THEN

      IF inst_number = -1 THEN -- seach for all instances
        -- check whether it is a persistent queue
        BEGIN 
          EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_pqueues_tab ' ||
                  '  WHERE QUEUE_ID = ' || get_queue_report.queue_id
            INTO cnt;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        IF cnt > 0 THEN -- if it is a persistent_queue
          -- dump one snapshot before generating report
          IF final_collect AND mon_status = AQMON_STATUS_START THEN
            collect_runtime;
            collect_config;
            DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
          END IF;
 
          BEGIN      
            stxt := 'SELECT distinct(INST_ID) from aqmon_pqueues_tab' ||
                    ' WHERE QUEUE_ID = ' || get_queue_report.queue_id;  
            OPEN gqr_c1 for stxt;
            LOOP 
              FETCH gqr_c1 INTO myINST_NUM;
              EXIT WHEN gqr_c1%NOTFOUND;
              prvt_get_pqueue_report(db_version, myINST_NUM, 
                get_queue_report.queue_id, single_logfile); 
              queue_cnt := queue_cnt+1;
            END LOOP;
            CLOSE gqr_c1;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;
        
        ELSE  
          -- otherwise, check whether it is a buffered queue.
          BEGIN
            EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_bqueues_tab ' ||
                    '  WHERE QUEUE_ID = ' || get_queue_report.queue_id  
              INTO cnt;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          IF cnt > 0 THEN 
            -- dump one snapshot before generating report
            IF final_collect AND mon_status = AQMON_STATUS_START THEN
              collect_runtime;
              collect_config;
              DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
            END IF;

            BEGIN      
              stxt := 'SELECT distinct(INST_ID) from aqmon_bqueues_tab' ||
                      ' WHERE QUEUE_ID = ' || get_queue_report.queue_id;  
              OPEN gqr_c1 for stxt;
              LOOP 
                FETCH gqr_c1 INTO myINST_NUM;
                EXIT WHEN gqr_c1%NOTFOUND;
                prvt_get_bqueue_report(db_version, myINST_NUM, 
                  get_queue_report.queue_id, single_logfile); 
                queue_cnt := queue_cnt+1;
              END LOOP;
              CLOSE gqr_c1;
            EXCEPTION WHEN OTHERS THEN NULL;
            END;
          END IF;  -- IF cnt > 0 (buffered)  

        END IF;  -- IF cnt > 0 (persistent) ... ELSE ...

      ELSE -- for a specific instance

        myINST_NUM := inst_number;
        IF inst_number IS NULL THEN
          -- get the current instance number
          BEGIN
            SELECT i.instance_number INTO myINST_NUM FROM v$instance i;
          EXCEPTION
            WHEN OTHERS THEN
              myINST_NUM := -100;
          END;
        END IF;
      
        -- check whether it is a persistent queue
        BEGIN
          EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_pqueues_tab ' ||
                  ' WHERE QUEUE_ID = ' || get_queue_report.queue_id ||
                  '   AND INST_ID  = ' || myINST_NUM
            INTO cnt;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        IF cnt > 0 THEN
          -- dump one snapshot before generating report
          IF final_collect AND mon_status = AQMON_STATUS_START THEN
            collect_runtime;
            collect_config;
            DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
           END IF;

          prvt_get_pqueue_report(db_version, myINST_NUM, 
            get_queue_report.queue_id, single_logfile);
          queue_cnt := queue_cnt+1;
        ELSE  
          -- otherwise, check whether it is a buffered queue.
          BEGIN
            EXECUTE IMMEDIATE 'SELECT count(*) FROM aqmon_bqueues_tab ' ||
                    ' WHERE QUEUE_ID = ' || get_queue_report.queue_id ||
                    '   AND INST_ID  = ' || myINST_NUM
              INTO cnt;
          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          IF cnt > 0 THEN 
            -- dump one snapshot before generating report
            IF final_collect AND mon_status = AQMON_STATUS_START THEN
              collect_runtime;
              collect_config;
              DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
            END IF;

            prvt_get_bqueue_report(db_version, myINST_NUM, 
              get_queue_report.queue_id, single_logfile);
            queue_cnt := queue_cnt+1;
          END IF;  -- IF cnt > 0 (buffered) 
        END IF;  -- IF cnt > 0 ... ELSE ... (persistent)
      END IF; --  IF inst_number = -1 ... ELSE ...

    END IF; -- IF db_version ...

    -- display logdir info   
    IF queue_cnt > 0 THEN
      BEGIN
        SELECT DIRECTORY_PATH INTO logdir_path 
          FROM sys.dba_directories
         WHERE DIRECTORY_NAME = AQMON_LOGDIR;
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          logdir_path := NULL;
      END;
      prvt_echo_line('--');
      prvt_echo_line('Report in directory ' || logdir_path || '.');    
    ElSE
      IF inst_number = -1 THEN
        prvt_echo_line('Queue ' || get_queue_report.queue_id  
         || ' does not exist or is not monitored on any instance.');
      ELSE
        prvt_echo_line('Queue ' || get_queue_report.queue_id || 
         ' on instance ' || myINST_NUM || ' does not exist or is ' || 
         'not monitored.');
      END IF;
    END IF;

  END get_queue_report;



  -- EXPORT PROCEDURE: generate a detailed report for each of the 
  --                   monitored queues on the specified instance.
  --
  -- PARAM  : inst_number - If inst_number is not specified (i.e., 
  --                        inst_number IS NULL), then automatically fetch 
  --                        the current instance number.
  --                        If inst_number = -1, then generate reports for 
  --                        the queue on all instances. 
  --                        For other cases, inst_number should be >= 0.
  --                        In Oracle database 10.2, inst_number cannot be -1.
  --          final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  --          single_logfile - If set to true, then all output will be written
  --                           to a single logfile (file name is specified by 
  --                           AQMON_ALLQUEUES_REP_FNAME). Otherwise, each 
  --                           queue will have its own log file.
  PROCEDURE get_all_queues_reports(inst_number    IN NUMBER DEFAULT NULL,
                                   final_collect  IN BOOLEAN DEFAULT TRUE,
                                   single_logfile IN BOOLEAN DEFAULT FALSE) AS 
    queue_cnt    NUMBER;  
    tmp          NUMBER;
    mon_status   NUMBER;
    myINST_NUM   NUMBER;
    myQUEUE_ID   NUMBER;
    db_version   NUMBER;
    pl           VARCHAR2(2); 
    pl2          VARCHAR2(5);
    logdir_path  VARCHAR2(4000);
  
    stxt         VARCHAR2(1000);
    TYPE gaqr_refcur IS REF CURSOR;
    gaqr_c1 gaqr_refcur;

  BEGIN
    mon_status := prvt_get_param('APT_MONITOR_STATUS');
    IF mon_status IS NULL OR mon_status = AQMON_STATUS_NULL THEN
      prvt_echo_error('Monitor is not started yet.');
      RETURN;
    END IF;

    IF inst_number < -1 THEN
      prvt_echo_error('Instance number must be positive.');
      RETURN;  
    END IF;

    IF final_collect IS NULL THEN
      prvt_echo_error('Parameter final_collect cannot be NULL.');
      RETURN;  
    END IF;

    IF single_logfile IS NULL THEN
      prvt_echo_error('Parameter single_logfile cannot be NULL.');
      RETURN;
    END IF; 

    -- dump one snapshot before generating report
    IF final_collect AND mon_status = AQMON_STATUS_START THEN
      collect_runtime;
      collect_config;
      DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
    END IF;

    -- get logdir info
    BEGIN
      SELECT DIRECTORY_PATH INTO logdir_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        logdir_path := NULL;
    END;

    db_version := prvt_get_param('APT_DATABASE_VERSION');  

    IF db_version = 10.2 THEN
 
      -- for 10.2, aqmon_pqueues_tab does not exist, but aqmon_bqueues_tab
      -- exists.

      IF inst_number = -1 THEN -- search for all instances
        prvt_echo_error('Instance number must be specified or set' ||
                        ' to NULL in Oracle 10.2.');
        RETURN;
      ELSE -- for a specific instance

        myINST_NUM := inst_number;
        IF inst_number IS NULL THEN
          -- get the current instance number
          BEGIN
            SELECT i.instance_number INTO myINST_NUM FROM v$instance i;
          EXCEPTION
            WHEN OTHERS THEN
              myINST_NUM := -100;
          END;
        END IF;

        queue_cnt := 0;

        -- check buffered queues
        stxt := 'SELECT distinct(QUEUE_ID) from aqmon_bqueues_tab' ||
                ' WHERE INST_ID = ' || myINST_NUM ||
                '   AND QUEUE_ID IS NOT NULL' ||  
                ' ORDER BY QUEUE_ID';
        OPEN gaqr_c1 for stxt;
        LOOP
          FETCH gaqr_c1 INTO myQUEUE_ID;
          EXIT WHEN gaqr_c1%NOTFOUND;
      
          prvt_get_bqueue_report(db_version, myINST_NUM, 
            myQUEUE_ID, single_logfile);
          queue_cnt := queue_cnt+1;
        END LOOP;
        CLOSE gaqr_c1;


        -- check remaining queues
        stxt := 'SELECT distinct(QUEUE_ID) FROM aqmon_queue_config_tab' ||
                ' WHERE QUEUE_ID IS NOT NULL' ||
                '   AND NOT QUEUE_ID IN (SELECT QUEUE_ID from ' ||
                '                        aqmon_bqueues_tab)' ||
                '   AND QUEUE_TYPE = ''NORMAL_QUEUE''' ||
                ' ORDER BY QUEUE_ID';
        OPEN gaqr_c1 for stxt;
        LOOP
          FETCH gaqr_c1 INTO myQUEUE_ID;
          EXIT WHEN gaqr_c1%NOTFOUND;
      
          prvt_get_pqueue_report(db_version, myINST_NUM, 
            myQUEUE_ID, single_logfile);
          queue_cnt := queue_cnt+1;
        END LOOP;
        CLOSE gaqr_c1;

      END IF;

      IF queue_cnt = 0 THEN
        prvt_echo_line('> > > No queue found on instance ' || 
                             myINST_NUM || '.');
      ELSE
        IF queue_cnt > 1 THEN
          pl  := 's';
          pl2 := ' are ';
        ELSE
          pl  := NULL;
          pl2 := ' is ';
        END IF;

        prvt_echo_line('> > > Report' || pl || ' for ' || queue_cnt || 
                       ' queue' || pl || ' on instance ' ||
                       myINST_NUM || pl2 || 'generated in directory ' || 
                       logdir_path || '.');
      END IF;

    ELSIF db_version >= 11.1 THEN

      IF inst_number = -1 THEN  -- search for all instances 

        -- check all persistent queues
        prvt_echo_line('--------------------');
        queue_cnt := 0;

        BEGIN
          stxt := 'SELECT INST_ID, QUEUE_ID FROM aqmon_pqueues_tab' ||
                  ' WHERE INST_ID IS NOT NULL' ||
                  '   AND QUEUE_ID IS NOT NULL' ||
                  ' GROUP BY INST_ID, QUEUE_ID' ||
                  ' ORDER BY INST_ID, QUEUE_ID';
          OPEN gaqr_c1 for stxt;
          LOOP
            FETCH gaqr_c1 INTO myINST_NUM, myQUEUE_ID;
            EXIT WHEN gaqr_c1%NOTFOUND;
      
            prvt_get_pqueue_report(db_version, myINST_NUM, 
              myQUEUE_ID, single_logfile); 
            queue_cnt := queue_cnt+1;
          END LOOP;
          CLOSE gaqr_c1;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;
      
        IF queue_cnt = 0 THEN
          prvt_echo_line('> > > No persistent queue found on any instance.');
        ELSE
          IF queue_cnt > 1 THEN
            pl  := 's';
            pl2 := ' are ';
          ELSE
            pl  := NULL;
            pl2 := ' is ';
          END IF;

          prvt_echo_line('> > > Report' || pl || ' for ' || queue_cnt || 
                           ' persistent queue' || pl || pl2 || 
                           'generated in directory ' || logdir_path || '.');
        END IF;


        -- check all buffered queues
        prvt_echo_line('--------------------');
        queue_cnt := 0;

        BEGIN
          stxt := 'SELECT INST_ID, QUEUE_ID FROM aqmon_bqueues_tab' ||
                  ' WHERE INST_ID IS NOT NULL' ||
                  '   AND QUEUE_ID IS NOT NULL' ||
                  ' GROUP BY INST_ID, QUEUE_ID' ||
                  ' ORDER BY INST_ID, QUEUE_ID';
          OPEN gaqr_c1 for stxt;
          LOOP
            FETCH gaqr_c1 INTO myINST_NUM, myQUEUE_ID;
            EXIT WHEN gaqr_c1%NOTFOUND;
      
            prvt_get_bqueue_report(db_version, myINST_NUM, 
              myQUEUE_ID, single_logfile); 
            queue_cnt := queue_cnt+1;
          END LOOP;
          CLOSE gaqr_c1;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        IF queue_cnt = 0 THEN
          prvt_echo_line('> > > No buffered queue found on any instance.');
        ELSE
          IF queue_cnt > 1 THEN
            pl  := 's';
            pl2 := ' are ';
          ELSE
            pl  := NULL;
            pl2 := ' is ';
          END IF;

          prvt_echo_line('> > > Report' || pl || ' for ' || queue_cnt || 
                         ' buffered queue' || pl || pl2 || 
                         'generated in directory ' || logdir_path || '.');
        END IF;

      ELSE -- for a specific instance
 
        myINST_NUM := inst_number;
        IF inst_number IS NULL THEN
          -- get the current instance number
          BEGIN
            SELECT i.instance_number INTO myINST_NUM FROM v$instance i;
          EXCEPTION
            WHEN OTHERS THEN
              myINST_NUM := -100;
          END;
        END IF;

        -- check all persistent queues
        prvt_echo_line('--------------------');
        queue_cnt := 0;

        BEGIN
          stxt := 'SELECT distinct(QUEUE_ID) FROM aqmon_pqueues_tab' ||
                  ' WHERE INST_ID = ' || myINST_NUM ||
                  '   AND QUEUE_ID IS NOT NULL' ||
                  ' ORDER BY QUEUE_ID';
          OPEN gaqr_c1 for stxt;
          LOOP
            FETCH gaqr_c1 INTO myQUEUE_ID;

            EXIT WHEN gaqr_c1%NOTFOUND;

            prvt_get_pqueue_report(db_version, myINST_NUM, 
              myQUEUE_ID, single_logfile); 

            queue_cnt := queue_cnt+1;

          END LOOP;
          CLOSE gaqr_c1;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;
 
        IF queue_cnt = 0 THEN
          prvt_echo_line('> > > No persistent queue found on instance ' || 
                               myINST_NUM || '.');
        ELSE
          IF queue_cnt > 1 THEN
            pl  := 's';
            pl2 := ' are ';
          ELSE
            pl  := NULL;
            pl2 := ' is ';
          END IF;

          prvt_echo_line('> > > Report' || pl || ' for ' || queue_cnt || 
                         ' persistent queue' || pl || ' on instance ' ||
                         myINST_NUM || pl2 || 'generated in directory ' || 
                         logdir_path || '.');

        END IF;

        -- check all buffered queues
        prvt_echo_line('--------------------');
        queue_cnt := 0;

        BEGIN
          stxt := 'SELECT distinct(QUEUE_ID) FROM aqmon_bqueues_tab' ||
                  ' WHERE INST_ID = ' || myINST_NUM ||
                  '   AND QUEUE_ID IS NOT NULL' ||
                  ' ORDER BY QUEUE_ID';
          OPEN gaqr_c1 for stxt;
          LOOP
            FETCH gaqr_c1 INTO myQUEUE_ID;
            EXIT WHEN gaqr_c1%NOTFOUND;
      
            prvt_get_bqueue_report(db_version, myINST_NUM, 
              myQUEUE_ID, single_logfile); 
            queue_cnt := queue_cnt+1;
          END LOOP;
          CLOSE gaqr_c1;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;
     
        IF queue_cnt = 0 THEN
          prvt_echo_line('> > > No buffered queue found on instance ' || 
                               myINST_NUM || '.');
        ELSE
          IF queue_cnt > 1 THEN
            pl  := 's';
            pl2 := ' are ';
          ELSE
            pl  := NULL;
            pl2 := ' is ';
          END IF;

          prvt_echo_line('> > > Report' || pl || ' for ' || queue_cnt || 
                         ' buffered queue' || pl || ' on instance ' ||
                         myINST_NUM || pl2 || 'generated in directory ' || 
                         logdir_path || '.');

        END IF;

      END IF; -- IF inst_number = -1 ... ELSE ...

    END IF; -- IF db_version

  END get_all_queues_reports;



  -- Generate a report for subscr-registration stats.
  -- (valid for 11.1/11.2)
  -- 
  -- Columns
  --   NUM_NTFNS                      :     accumulative
  --   NUM_GROUPING_NTFNS             :     accumulative (?)
  --   NUM_RETRIES                    :     accumulative (?)
  --   EMON#                          : non-accumulative (?)
  --   NUM_NTFNS_CURRENT_GROUP (11.2) :     accumulative (?)
  --   NUM_PENDING_NTFNS (11.2)       : non-accumulative (?)
  --
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_subreg_report(final_collect IN BOOLEAN DEFAULT TRUE) AS

    first_row      BOOLEAN := TRUE;

    -- for 11.1
    pre_monitor_time           TIMESTAMP;    
    pre_num_ntfns              NUMBER;
    pre_num_grp_ntfns          NUMBER;
    pre_tt_emon_latency        NUMBER;
    pre_tt_pl_bytes_sent       NUMBER;
    pre_num_retries            NUMBER;
    pre_tt_plsql_exec_tm       NUMBER;

    curr_memseq                NUMBER;
    curr_monitor_time          TIMESTAMP;
    curr_num_ntfns             NUMBER;
    curr_num_grp_ntfns         NUMBER;
    curr_tt_emon_latency       NUMBER;
    curr_tt_pl_bytes_sent      NUMBER;
    curr_num_retries           NUMBER;
    curr_tt_plsql_exec_tm      NUMBER;

    my_time_duration           NUMBER;
    my_num_ntfns_td            NUMBER;
    my_num_grp_ntfns_td        NUMBER;
    my_tt_emon_latency_td      NUMBER;
    my_tt_pl_bytes_sent_td     NUMBER;
    my_num_retries_td          NUMBER;
    my_tt_plsql_exec_tm_td     NUMBER;

    my_lt_ntfn_start_tm_accum  TIMESTAMP;
    my_lt_ntfn_sent_tm_accum   TIMESTAMP;
    my_emon_num_accum          NUMBER;
    my_all_emon_srvs_accum     RAW(2000);
    my_lt_err_accum            VARCHAR2(90);
    my_lt_err_tm_accum         TIMESTAMP;
    my_lt_update_tm_accum      TIMESTAMP; 
    -- end for 11.1

    -- for 11.2
    pre_num_ntfns_curr_grp     NUMBER;

    curr_num_ntfns_curr_grp    NUMBER;

    my_num_ntfns_curr_grp_td   NUMBER;
    my_num_pending_ntfns_accum NUMBER;
    -- end for 11.1 
 
    title_line1            VARCHAR2(1000);
    title_line2_p1         VARCHAR2(1000);    
    title_line2_p2         VARCHAR2(1000);    
    title_line3_p1         VARCHAR2(1000);
    title_line3_p2         VARCHAR2(1000);
    mon_status             NUMBER;
    db_version             NUMBER;
    stxt                   VARCHAR2(1000);
    stxt2                  VARCHAR2(1000);
    tmpVar                 VARCHAR2(1000);
    logdir_path            VARCHAR2(4000);
    myINST_ID              NUMBER;
    myREG_ID               NUMBER;
    mySUBSCR_NAME          VARCHAR2(128);
    myLOC_NAME             VARCHAR2(256);
    adj_inst_id            NUMBER;
    adj_reg_id             NUMBER;
    adj_memseq             NUMBER;
    adj_flag               NUMBER;
    afteradj_memseq        NUMBER;
    afteradj_monitor_time  TIMESTAMP;
    min_memseq             NUMBER;
    
    TYPE subreg_refcur IS REF CURSOR;
    subreg_c1 subreg_refcur;
    subreg_c2 subreg_refcur;

  BEGIN
    mon_status := prvt_get_param('APT_MONITOR_STATUS');
    IF mon_status IS NULL OR mon_status = AQMON_STATUS_NULL THEN
      prvt_echo_error('Monitor is not started yet.');
      RETURN;
    END IF;

    IF final_collect IS NULL THEN
      prvt_echo_error('Parameter final_collect cannot be NULL.');
      RETURN;  
    END IF;   

    db_version := prvt_get_param('APT_DATABASE_VERSION');
    IF db_version < 11.1 THEN
      prvt_echo_error('Registration information is only available for ' ||
                      'Oracle database 11.1 or above.');
      RETURN;
    END IF;

    BEGIN
      SELECT DIRECTORY_PATH INTO logdir_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        logdir_path := NULL;
    END;

    BEGIN
      AQMON_SUBREG_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_SUBREG_FNAME,
                                          'w', 5000);
    EXCEPTION
      WHEN OTHERS THEN
        prvt_echo_error('Cannot create log file ' || AQMON_SUBREG_FNAME ||
                        ' in directory ' || logdir_path || '.');
    END;

    IF NOT UTL_FILE.IS_OPEN(AQMON_SUBREG_OUTPUT) THEN
      prvt_echo_error('Cannot create log file ' || AQMON_SUBREG_FNAME ||
                      ' in directory ' || logdir_path || '.');
      RETURN;
    END IF;

    -- dump one snapshot before generating report
    IF final_collect AND mon_status = AQMON_STATUS_START THEN
      collect_runtime;
      collect_config;
      DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
    END IF;
    
    -------------------------------------------------------------------------- 
    --  start to collect subscr-registration info
    -------------------------------------------------------------------------- 

    IF db_version >= 11.2 THEN 

      title_line1 := lpad(' Per-Period Metrics ', 62, '-') || 
                     lpad(' | ', 40, '-') ||
                     lpad(' Accumulative Metrics ', 130, '-') || 
                     lpad(' | ', 103, '-') ||
                     lpad('-', 54, '-') ; 


      title_line2_p1 := ' Snap   Time     Num              Num       Num       Num Ntfns      Emon      Payload       PLSQL  |            Num                      Last Ntfn                      Last Ntfn';
      title_line3_p1 := '  Seq    (s)   Ntfns   Grouping Ntfns   Retries   Current Group   Latency   Bytes Sent   Exec Time  |  Pending Ntfns                     Start Time                      Sent Time';

      
      title_line2_p2 := '                                                                                 Last Error                     Last Error                    Last Update  |  Timestamp                     |  Active     All Emon';
      title_line3_p2 := '                                                                                    Message                           Time                           Time  |                                |  Emon Num   Servers';

    ELSIF db_version = 11.1 THEN

      title_line1 := lpad(' Per-Period Metrics ', 51, '-') || 
                     lpad(' | ', 35, '-') ||
                     lpad(' Accumulative Metrics ', 120, '-') || 
                     lpad(' | ', 97, '-') ||
                     lpad('-', 54, '-') ; 


      title_line2_p1 := ' Snap   Time     Num              Num       Num      Emon      Payload       PLSQL  |                     Last Ntfn                      Last Ntfn';
      title_line3_p1 := '  Seq    (s)   Ntfns   Grouping Ntfns   Retries   Latency   Bytes Sent   Exec Time  |                    Start Time                      Sent Time';

      
      title_line2_p2 := '                                                                                 Last Error                     Last Error                    Last Update  |  Timestamp                     |  Active     All Emon';
      title_line3_p2 := '                                                                                    Message                           Time                           Time  |                                |  Emon Num   Servers';

    END IF;


    UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, 'Registration Statistics');
    UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');


    -- loop through different inst_id and reg_id
    stxt := 'SELECT INST_ID, REG_ID FROM aqmon_subreg_tab' ||
            ' WHERE INST_ID IS NOT NULL AND REG_ID IS NOT NULL' ||
            ' GROUP BY INST_ID, REG_ID' ||
            ' ORDER BY INST_ID, REG_ID';  
    OPEN subreg_c1 FOR stxt;
    LOOP
      FETCH subreg_c1 INTO myINST_ID, myREG_ID;
      EXIT WHEN subreg_c1%NOTFOUND;


      --  adjust aqmon_subreg_tab as in prvt_postrun_adjust
      stxt2 := 'SELECT INST_ID, REG_ID, memseq, FLAG' ||
               '  FROM aqmon_subreg_tab' ||
               ' WHERE INST_ID = ' || myINST_ID ||
               '   AND REG_ID  = ' || myREG_ID ||
               '   AND memseq = (SELECT min(memseq)' ||
               '                   FROM aqmon_subreg_tab' ||
               '                  WHERE INST_ID = ' || myINST_ID ||
               '                    AND REG_ID  = ' || myREG_ID || ')';        
      OPEN subreg_c2 for stxt2;
      LOOP
        FETCH subreg_c2 INTO adj_inst_id, adj_reg_id, adj_memseq, adj_flag;
        EXIT;
      END LOOP;
      CLOSE subreg_c2;
 
      EXECUTE IMMEDIATE 'SELECT min(memseq) FROM aqmon_subreg_tab'
        INTO min_memseq;
      IF min_memseq IS NULL THEN
        min_memseq := -1;
      END IF;

      -- If this reg_id appeared in gv$subscr_registration_stats after 
      -- monitoring starts, we generate a 'first-snapshot' with flag=1
      IF (adj_reg_id IS NOT NULL) AND (min_memseq > 0) AND 
         (adj_memseq > min_memseq) AND (adj_flag = 0) THEN

        -- get the immediately previous memseq before adj_memseq
        stxt2 := 'SELECT memseq, MONITOR_TIME FROM aqmon_subreg_tab ' ||
          ' WHERE memseq = (SELECT max(memseq) FROM aqmon_subreg_tab ' ||
          ' WHERE memseq < ' || adj_memseq || ')';
        OPEN subreg_c2 for stxt2;
        LOOP 
          FETCH subreg_c2 INTO afteradj_memseq, afteradj_monitor_time;
          EXIT;
        END LOOP;
        CLOSE subreg_c2;

        stxt2 := 'INSERT INTO aqmon_subreg_tab (INST_ID, REG_ID, ' ||
                 ' MEMSEQ, MONITOR_TIME, FLAG)' ||
                 ' VALUES (:1, :2, :3, :4, 1)';
        EXECUTE IMMEDIATE stxt2 USING myINST_ID, myREG_ID, 
          afteradj_memseq, afteradj_monitor_time;

        EXECUTE IMMEDIATE 'COMMIT';

      END IF; 

      -- end for adjust


      stxt2 := 'SELECT SUBSCRIPTION_NAME, LOCATION_NAME' ||
               '  FROM dba_subscr_registrations' ||
               ' WHERE REG_ID = ' || myREG_ID;
      OPEN subreg_c2 for stxt2;
      LOOP
        FETCH subreg_c2 INTO mySUBSCR_NAME, myLOC_NAME;
        EXIT;
      END LOOP;
      CLOSE subreg_c2;

      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, 'Instance ID       : ' || 
                                              myINST_ID);
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, 'Registration ID   : ' || 
                                              myREG_ID);
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, 'Subscription Name : ' ||
                                              mySUBSCR_NAME);
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, 'Location Name     : ' ||
                                              myLOC_NAME);
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
     
      -- loop through different memseq
      first_row := TRUE;
      stxt2 := 'SELECT NUM_NTFNS, NUM_GROUPING_NTFNS, LAST_NTFN_START_TIME,'||
        ' LAST_NTFN_SENT_TIME, TOTAL_EMON_LATENCY, EMON#, ALL_EMON_SERVERS,'||
        ' TOTAL_PAYLOAD_BYTES_SENT, NUM_RETRIES, TOTAL_PLSQL_EXEC_TIME, '   ||
        ' LAST_ERR, LAST_ERR_TIME, LAST_UPDATE_TIME, MEMSEQ, MONITOR_TIME'  ||
        ' FROM aqmon_subreg_tab' ||
        ' WHERE INST_ID = ' || myINST_ID ||
        '   AND REG_ID  = ' || myREG_ID  ||
        ' ORDER BY memseq';
 
      OPEN subreg_c2 for stxt2;
      LOOP
        FETCH subreg_c2 INTO curr_num_ntfns, curr_num_grp_ntfns,
          my_lt_ntfn_start_tm_accum, my_lt_ntfn_sent_tm_accum,
          curr_tt_emon_latency, my_emon_num_accum, my_all_emon_srvs_accum,
          curr_tt_pl_bytes_sent, curr_num_retries, curr_tt_plsql_exec_tm,
          my_lt_err_accum, my_lt_err_tm_accum, my_lt_update_tm_accum,
          curr_memseq, curr_monitor_time;

        EXIT WHEN subreg_c2%NOTFOUND;
 
        curr_memseq            := nvl(curr_memseq, -1);
        curr_num_ntfns         := nvl(curr_num_ntfns, 0); 
        curr_num_grp_ntfns     := nvl(curr_num_grp_ntfns, 0);
        curr_tt_emon_latency   := nvl(curr_tt_emon_latency, 0);
        my_emon_num_accum      := nvl(my_emon_num_accum, 0);
        curr_tt_pl_bytes_sent  := nvl(curr_tt_pl_bytes_sent, 0);
        curr_num_retries       := nvl(curr_num_retries, 0);
        curr_tt_plsql_exec_tm  := nvl(curr_tt_plsql_exec_tm, 0);

        IF db_version >= 11.2 THEN
          EXECUTE IMMEDIATE 'SELECT NUM_NTFNS_CURRENT_GROUP, ' ||
            ' NUM_PENDING_NTFNS FROM aqmon_subreg_tab ' ||
            ' WHERE INST_ID = ' || myINST_ID ||
            '   AND REG_ID  = ' || myREG_ID  ||
            '   AND MEMSEQ  = ' || curr_memseq
            INTO curr_num_ntfns_curr_grp, my_num_pending_ntfns_accum;
          
          curr_num_ntfns_curr_grp    := nvl(curr_num_ntfns_curr_grp, 0);
          my_num_pending_ntfns_accum := nvl(my_num_pending_ntfns_accum, 0);
           
        END IF;

        IF first_row THEN 

          UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, title_line1);          
          UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, title_line2_p1 || 
                                                 title_line2_p2 );
          UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, title_line3_p1 || 
                                                 title_line3_p2 );
          UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
          first_row := FALSE;

        ELSE    
          -- unit: second
          my_time_duration := (CAST(curr_monitor_time as DATE)-
                               CAST(pre_monitor_time as DATE))*86400.0; 

          my_num_ntfns_td       := curr_num_ntfns       - pre_num_ntfns;
          my_num_grp_ntfns_td   := curr_num_grp_ntfns   - pre_num_grp_ntfns;
          my_num_retries_td     := curr_num_retries     - pre_num_retries;
          my_tt_emon_latency_td := curr_tt_emon_latency - pre_tt_emon_latency;
          my_tt_pl_bytes_sent_td:= curr_tt_pl_bytes_sent- pre_tt_pl_bytes_sent;
          my_tt_plsql_exec_tm_td:= curr_tt_plsql_exec_tm- pre_tt_plsql_exec_tm;

          IF db_version >= 11.2 THEN
            my_num_ntfns_curr_grp_td := curr_num_ntfns_curr_grp - 
                                        pre_num_ntfns_curr_grp;
          END IF;

          IF db_version >= 11.2 THEN
            tmpVar := lpad(curr_memseq, 5)                           || ' ' ||
              lpad(to_char(my_time_duration, 'FM999999'), 6)         || ' ' ||
              lpad(my_num_ntfns_td, 7)                               || ' ' ||
              lpad(my_num_grp_ntfns_td, 16)                          || ' ' ||
              lpad(my_num_retries_td, 9)                             || ' ' ||
              lpad(my_num_ntfns_curr_grp_td, 15 )                    || ' ' ||
              lpad(my_tt_emon_latency_td, 9)                         || ' ' ||
              lpad(my_tt_pl_bytes_sent_td, 12)                       || ' ' ||
              lpad(to_char(my_tt_plsql_exec_tm_td,'FM9999999.000'),11)||'  |'||
              lpad(my_num_pending_ntfns_accum, 15)                   || ' ' ||
              lpad(nvl(to_char(my_lt_ntfn_start_tm_accum), ' '), 30) || ' ' ||
              lpad(nvl(to_char(my_lt_ntfn_sent_tm_accum), ' '), 30)  || ' ' ||
              lpad(nvl(to_char(my_lt_err_accum), ' '), 90)           || ' ' ||
              lpad(nvl(to_char(my_lt_err_tm_accum), ' '),30)         || ' ' ||
              lpad(nvl(to_char(my_lt_update_tm_accum), ' '), 30)     ||'  |'||
              lpad(nvl(to_char(curr_monitor_time), ' '),30)        ||'  |  '||
              rpad(my_emon_num_accum, 8)                           || '   ' ||
              to_char(my_all_emon_srvs_accum);
          ELSIF db_version = 11.1 THEN
            tmpVar := lpad(curr_memseq, 5)                           || ' ' ||
              lpad(to_char(my_time_duration, 'FM999999'), 6)         || ' ' ||
              lpad(my_num_ntfns_td, 7)                               || ' ' ||
              lpad(my_num_grp_ntfns_td, 16)                          || ' ' ||
              lpad(my_num_retries_td, 9)                             || ' ' ||
              lpad(my_tt_emon_latency_td, 9)                         || ' ' ||
              lpad(my_tt_pl_bytes_sent_td, 12)                       || ' ' ||
              lpad(to_char(my_tt_plsql_exec_tm_td,'FM9999999.000'),11)||'  |'||
              lpad(nvl(to_char(my_lt_ntfn_start_tm_accum), ' '), 30) || ' ' ||
              lpad(nvl(to_char(my_lt_ntfn_sent_tm_accum), ' '), 30)  || ' ' ||
              lpad(nvl(to_char(my_lt_err_accum), ' '), 90)           || ' ' ||
              lpad(nvl(to_char(my_lt_err_tm_accum), ' '),30)         || ' ' ||
              lpad(nvl(to_char(my_lt_update_tm_accum), ' '), 30)     ||'  |'||
              lpad(nvl(to_char(curr_monitor_time), ' '),30)        ||'  |  '||
              rpad(my_emon_num_accum, 8)                           || '   ' ||
              to_char(my_all_emon_srvs_accum);
          END IF;

          UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, tmpVar);          
        END IF; -- end if first_row

        pre_monitor_time     := curr_monitor_time;   
        pre_num_ntfns        := curr_num_ntfns;
        pre_num_grp_ntfns    := curr_num_grp_ntfns;
        pre_tt_emon_latency  := curr_tt_emon_latency;
        pre_tt_pl_bytes_sent := curr_tt_pl_bytes_sent;
        pre_num_retries      := curr_num_retries;
        pre_tt_plsql_exec_tm := curr_tt_plsql_exec_tm;
     
        IF db_version >= 11.2 THEN
          pre_num_ntfns_curr_grp := curr_num_ntfns_curr_grp;
        END IF;

      END LOOP; -- loop through different memseq
      CLOSE subreg_c2;

      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_SUBREG_OUTPUT, ' ');

    END LOOP; -- loop through different inst_id and reg_id
    CLOSE subreg_c1;


    UTL_FILE.FCLOSE(AQMON_SUBREG_OUTPUT);
 
    -- display log file info
    IF logdir_path IS NOT NULL THEN
      logdir_path := TRIM(logdir_path);
      IF SUBSTR(logdir_path, LENGTH(logdir_path), 1) != '/' THEN
        logdir_path := CONCAT(logdir_path, '/');
      END IF;
      prvt_echo_line('Registration report is generated: ' || logdir_path || 
                     AQMON_SUBREG_FNAME);
    END IF;


  END get_subreg_report;


   

  -- EXPORT PROCEDURE: generate a summary report for all monitored queues.
  -- 
  -- PARAM  : final_collect - whether collect one snapshot before generating
  --                          report (only effective when the monitor is 
  --                          still running)
  PROCEDURE get_report(final_collect IN BOOLEAN DEFAULT TRUE) AS

    breakline          VARCHAR2(200) := rpad('-', 86, '-');
    title_line1_p1     VARCHAR2(1000);
    title_line1_p2     VARCHAR2(1000);
    title_line1_p3     VARCHAR2(1000);
    title_line1_p4     VARCHAR2(1000);
    title_line2_p1     VARCHAR2(1000);
    title_line2_p2     VARCHAR2(1000);
    title_line2_p3     VARCHAR2(1000);
    title_line2_p4     VARCHAR2(1000);
    title_line3        VARCHAR2(1000);

    var                VARCHAR2(1000);
    sqlstxt            VARCHAR2(1000);
    cnt                NUMBER;
    logdir_path        VARCHAR2(4000);
    plural             VARCHAR2(2);
    mon_status         NUMBER;
    db_version         NUMBER;

    ntfn_loc_name      VARCHAR2(256);
    ntfn_queue_name    VARCHAR2(100);
    ntfn_first_act_tm  TIMESTAMP;
    ntfn_enq_msgs      NUMBER;
    ntfn_deq_msgs      NUMBER;
    ntfn_ela_enq_tm    NUMBER;
    ntfn_ela_deq_tm    NUMBER;      
    ntfn_ls_enq_tm     TIMESTAMP;
    ntfn_ls_deq_tm     TIMESTAMP;

    myDBID             NUMBER;
    my_inst_id         NUMBER;
    my_queue_id        NUMBER;
    my_queue_schema    VARCHAR2(30);
    my_queue_name      VARCHAR2(30);

    wrt_flag               BOOLEAN;
    begin_monitor_tm       TIMESTAMP;
    end_monitor_tm         TIMESTAMP;
    mon_beginsnp           NUMBER;
    mon_endsnp             NUMBER;
    streams_pool_sz_begin  VARCHAR2(100);
    streams_pool_sz_end    VARCHAR2(100);

    -- statistics for persistent queues
    pq_monitor_time_1             TIMESTAMP;
    pq_queue_id_1                 NUMBER;
    pq_enqueued_msgs_1            NUMBER;
    pq_dequeued_msgs_1            NUMBER;
    pq_browsed_msgs_1             NUMBER;
    pq_enqueued_expiry_msgs_1     NUMBER;
    pq_enqueued_delay_msgs_1      NUMBER;
    pq_msgs_made_expired_1        NUMBER;
    pq_msgs_made_ready_1          NUMBER;
    pq_trans_time_1               NUMBER;
    pq_rule_eval_time_1           NUMBER;
    pq_ela_enqueue_time_1         NUMBER;
    pq_ela_dequeue_time_1         NUMBER;

    pq_monitor_time_2             TIMESTAMP;
    pq_queue_id_2                 NUMBER;
    pq_enqueued_msgs_2            NUMBER;
    pq_dequeued_msgs_2            NUMBER;
    pq_browsed_msgs_2             NUMBER;
    pq_enqueued_expiry_msgs_2     NUMBER;
    pq_enqueued_delay_msgs_2      NUMBER;
    pq_msgs_made_expired_2        NUMBER;
    pq_msgs_made_ready_2          NUMBER;
    pq_trans_time_2               NUMBER;
    pq_rule_eval_time_2           NUMBER;
    pq_ela_enqueue_time_2         NUMBER;
    pq_ela_dequeue_time_2         NUMBER;

    pq_time_duration              NUMBER;
    pq_enqueued_msgs_td           NUMBER;
    pq_dequeued_msgs_td           NUMBER;
    pq_browsed_msgs_td            NUMBER;
    pq_enqueued_expiry_msgs_td    NUMBER;
    pq_enqueued_delay_msgs_td     NUMBER;
    pq_msgs_made_expired_td       NUMBER;
    pq_msgs_made_ready_td         NUMBER;
    pq_enqueue_rate_td            NUMBER;
    pq_dequeue_rate_td            NUMBER;
    pq_avg_time_per_enq_td        NUMBER;
    pq_avg_time_per_deq_td        NUMBER;
    pq_trans_time_td              NUMBER;
    pq_rule_eval_time_td          NUMBER;

    pq_pending_msgs_accum         NUMBER; 
    pq_last_enqtm_accum           TIMESTAMP;
    pq_last_deqtm_accum           TIMESTAMP;
    pq_last_tm_expitm_accum       TIMESTAMP;
    pq_last_tm_readtm_accum       TIMESTAMP;

    -- for 11.2 or above 
    pq_enqueue_cpu_time_1         NUMBER;
    pq_dequeue_cpu_time_1         NUMBER;
    pq_enqueue_trans_1            NUMBER;
    pq_dequeue_trans_1            NUMBER;
    pq_execution_count_1          NUMBER;

    pq_enqueue_cpu_time_2         NUMBER;
    pq_dequeue_cpu_time_2         NUMBER;
    pq_enqueue_trans_2            NUMBER;
    pq_dequeue_trans_2            NUMBER;
    pq_execution_count_2          NUMBER;

    pq_cpu_time_per_enq_td        NUMBER;
    pq_cpu_time_per_deq_td        NUMBER;
    pq_enqueue_trans_td           NUMBER;
    pq_dequeue_trans_td           NUMBER;
    pq_execution_count_td         NUMBER;       
    pq_avg_msg_age_accum          NUMBER;
    pq_deq_msg_latency_accum      NUMBER;
    -- end for 11.2 or above
    

    -- statistics for buffered queues
    bq_time_duration              NUMBER;
    bq_enqueued_msgs_td           NUMBER;
    bq_dequeued_msgs_td           NUMBER;
    bq_spilled_msgs_td            NUMBER;
    bq_expired_msgs_td            NUMBER;
    bq_enqueue_rate_td            NUMBER; 
    bq_dequeue_rate_td            NUMBER; 

    bq_pending_msgs_accum         NUMBER; 
    

    -- for 11.2 or above
    bq_enqueue_cpu_time_1         NUMBER;
    bq_dequeue_cpu_time_1         NUMBER;
    bq_enqueue_ela_time_1         NUMBER;
    bq_dequeue_ela_time_1         NUMBER;
    bq_trans_time_1               NUMBER;
    bq_rule_eval_time_1           NUMBER; 

    bq_enqueue_cpu_time_2         NUMBER;
    bq_dequeue_cpu_time_2         NUMBER;
    bq_enqueue_ela_time_2         NUMBER;
    bq_dequeue_ela_time_2         NUMBER;
    bq_trans_time_2               NUMBER;
    bq_rule_eval_time_2           NUMBER; 
 
    bq_avg_time_per_enq_td        NUMBER;
    bq_avg_time_per_deq_td        NUMBER;
    bq_cpu_time_per_enq_td        NUMBER;
    bq_cpu_time_per_deq_td        NUMBER;
    bq_trans_time_td              NUMBER;
    bq_rule_eval_time_td          NUMBER;

    bq_oldest_msgid_accum         RAW(16);
    bq_oldest_msg_enqtm_accum     TIMESTAMP;
    bq_last_enqtm_accum           TIMESTAMP;
    bq_last_deqtm_accum           TIMESTAMP;
    -- end for 11.2 or above

    bq_line_start  aqmon_bqueues_tab%ROWTYPE;
    bq_line_end    aqmon_bqueues_tab%ROWTYPE;


    TYPE TABLE_RECIPENTS IS TABLE OF VARCHAR2(20);
    tab_rec TABLE_RECIPENTS;
    
    TYPE TABLE_GROUPING IS TABLE OF VARCHAR2(20);
    tab_grp TABLE_GROUPING;

    TYPE gr_refcur IS REF CURSOR;
    gr_c1 gr_refcur;

    cursor gr_ownerCursor IS 
      SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab ORDER BY QUEUE_OWNER;

    CURSOR gr_bqueueCursor IS 
      SELECT INST_ID, QUEUE_SCHEMA, QUEUE_ID, QUEUE_NAME 
        FROM aqmon_bqueues_tab
       WHERE INST_ID IS NOT NULL
         AND QUEUE_SCHEMA IS NOT NULL 
         AND QUEUE_ID IS NOT NULL
       GROUP BY INST_ID, QUEUE_SCHEMA, QUEUE_NAME, QUEUE_ID
       ORDER BY INST_ID, QUEUE_SCHEMA, QUEUE_NAME, QUEUE_ID;

    CURSOR gr_instCursor IS
      SELECT INST_ID FROM aqmon_pqueues_tab
       WHERE INST_ID IS NOT NULL
       UNION
      SELECT INST_ID FROM aqmon_bqueues_tab
       WHERE INST_ID IS NOT NULL;

    CURSOR gr_conf0a(inst_id NUMBER) IS
      SELECT * FROM gv$instance
       WHERE INST_ID = gr_conf0a.inst_id;

    CURSOR gr_conf0b(dbid NUMBER, inst_num NUMBER, snap_id NUMBER) IS
      SELECT * FROM dba_hist_parameter
       WHERE DBID            = gr_conf0b.dbid
         AND INSTANCE_NUMBER = gr_conf0b.inst_num
         AND SNAP_ID         = gr_conf0b.snap_id;

    -- gr_conf1.recipents = 'SINGLE' or 'MULTIPLE'
    CURSOR gr_conf1(recipients VARCHAR2) IS 
      SELECT  rpad(decode(QUEUE_NAME,  null, ' ', QUEUE_NAME),  30) || ' ' || 
              rpad(decode(QUEUE_ID,    null, ' ', QUEUE_ID),    10) || ' ' || 
              rpad(decode(QUEUE_TABLE, null, ' ', QUEUE_TABLE), 30) || ' ' ||
              rpad(decode(QUEUE_OWNER, null, ' ', QUEUE_OWNER), 30) || ' ' || 
              rpad(decode(DATA_TYPE,   null, ' ', DATA_TYPE),   10) || ' ' ||
              rpad(decode(OBJECT_TYPE, null, ' ', OBJECT_TYPE), 62) || ' ' || 
              rpad(decode(SORT_ORDER,  null, ' ', SORT_ORDER),  22) aa 
        FROM  aqmon_queue_config_tab
       WHERE  QUEUE_TYPE  = 'NORMAL_QUEUE' 
         AND  RECIPIENTS  = gr_conf1.recipients
       ORDER  BY QUEUE_OWNER, QUEUE_NAME, QUEUE_ID;

    -- gr_conf2.grouping = 'NONE' or 'TRANSACTIONAL'
    CURSOR gr_conf2(grouping VARCHAR2) IS 
      SELECT  rpad(decode(QUEUE_NAME,  null, ' ', QUEUE_NAME),  30) || ' ' || 
              rpad(decode(QUEUE_ID,    null, ' ', QUEUE_ID),    10) || ' ' || 
              rpad(decode(QUEUE_TABLE, null, ' ', QUEUE_TABLE), 30) || ' ' || 
              rpad(decode(QUEUE_OWNER, null, ' ', QUEUE_OWNER), 30) || ' ' || 
              rpad(decode(DATA_TYPE,   null, ' ', DATA_TYPE),   10) || ' ' ||
              rpad(decode(OBJECT_TYPE, null, ' ', OBJECT_TYPE), 62) || ' ' || 
              rpad(decode(SORT_ORDER,  null, ' ', SORT_ORDER),  22) aa 
        FROM  aqmon_queue_config_tab
       WHERE  QUEUE_TYPE  = 'NORMAL_QUEUE' 
         AND  MESSAGE_GROUPING  = gr_conf2.grouping
       ORDER  BY QUEUE_OWNER, QUEUE_NAME, QUEUE_ID;
    
    CURSOR gr_conf3 IS 
      SELECT  rpad(decode(QUEUE_NAME,    null, ' ', QUEUE_NAME), 30)    ||' '||
              rpad(decode(QUEUE_TABLE,   null, ' ', QUEUE_TABLE), 30)   ||' '||
              rpad(decode(CONSUMER_NAME, null, ' ', CONSUMER_NAME), 30) ||' '||
              rpad(decode(OWNER,         null, ' ', OWNER), 30)         ||' '||
              rpad(decode(TRANSFORMATION,null, ' ', TRANSFORMATION), 61)||' '||
              rpad(decode(DELIVERY_MODE, null, ' ', DELIVERY_MODE), 22) ||' '||
              rpad(decode(QUEUE_TO_QUEUE,null, ' ', QUEUE_TO_QUEUE), 17)||' '||
              rpad(decode(ADDRESS,       null, ' ', ADDRESS), 200)  aa
        FROM  aqmon_subscriber_config_tab
       ORDER  BY OWNER, QUEUE_NAME, CONSUMER_NAME;

    CURSOR gr_db0(duration_start TIMESTAMP,
                  duration_end   TIMESTAMP) IS
      SELECT sum(TM_DELTA_CPU_TIME)  sum_cpu_tm,
             rpad(decode(ash.SESSION_ID,null,' ',ash.SESSION_ID),30)  || ' ' ||
             rpad(decode(ash.SESSION_TYPE, null, ' ', ash.SESSION_TYPE), 13) 
                                                                      || ' ' ||
             rpad(decode(p.PROGRAM, null, ' ', p.PROGRAM), 50)        || ' ' ||
             rpad(sum(TM_DELTA_CPU_TIME), 30)                         || ' ' ||
             rpad(decode(sum(TM_DELTA_TIME), 0, ' ', null, ' ',
                  to_char(sum(TM_DELTA_CPU_TIME)*100/sum(TM_DELTA_TIME),  
                           'FM999.0000')), 30)                        || ' ' ||
             rpad(sum(TM_DELTA_DB_TIME), 30)                          || ' ' ||
             rpad(decode(sum(TM_DELTA_TIME), 0, ' ', null, ' ',
                  case when sum(TM_DELTA_DB_TIME)*100/sum(TM_DELTA_TIME) >= 100
                       then '100.0000'
                       else to_char(sum(TM_DELTA_DB_TIME)*100/sum(TM_DELTA_TIME),
                           'FM999.0000')
                  end), 30)            aa
        FROM V$ACTIVE_SESSION_HISTORY ash, V$SQL sql, V$PROCESS p, 
             V$SESSION sess 
       WHERE p.ADDR     = sess.PADDR 
         AND sess.SID   = ash.SESSION_ID 
         AND sql.SQL_ID = ash.SQL_ID 
         AND p.PROGRAM NOT LIKE '%SMCO%' 
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(sql.SQL_FULLTEXT, QUEUE_TABLE) >= 1) 
         AND ash.SAMPLE_TIME >= duration_start
         AND ash.SAMPLE_TIME <= duration_end
       GROUP BY ash.SESSION_ID, ash.SESSION_TYPE, p.PROGRAM 
       ORDER BY sum_cpu_tm desc;
 
    CURSOR gr_db1 IS
      SELECT rpad(decode(a.OWNER,  null, ' ', a.OWNER), 30)           || ' ' ||
             rpad(decode(a.TABLESPACE_NAME, null, ' ', 
                         a.TABLESPACE_NAME), 30)                      || ' ' ||
             rpad(decode(a.TABLE_NAME, null, ' ', a.TABLE_NAME), 30)  || ' ' ||
             rpad(decode(b.BLOCK_SIZE, null, ' ', b.BLOCK_SIZE), 12)  || ' ' ||
             rpad(decode(b.EXTENT_MANAGEMENT, null, ' ', 
                         b.EXTENT_MANAGEMENT), 20)                    || ' ' ||
             rpad(decode(b.SEGMENT_SPACE_MANAGEMENT, null, ' ', 
                         b.SEGMENT_SPACE_MANAGEMENT), 30)  aa
        FROM DBA_TABLES a, DBA_TABLESPACES b
       WHERE a.OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND a.TABLESPACE_NAME = b.TABLESPACE_NAME
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(a.TABLE_NAME, QUEUE_TABLE) >= 1) 
       ORDER BY a.OWNER, a.TABLESPACE_NAME, a.TABLE_NAME;     
  

    CURSOR gr_db2a IS 
      SELECT rpad(decode(SEGMENT_TYPE, null, ' ', SEGMENT_TYPE), 18) || ' ' ||
             rpad(decode(OWNER, null, ' ', OWNER), 30)               || ' ' ||
             rpad(decode(TABLESPACE_NAME, null, ' ', 
                         TABLESPACE_NAME), 30)                       || ' ' ||
             rpad(decode(SEGMENT_NAME, null, ' ', SEGMENT_NAME), 81) || ' ' ||
             rpad(decode(SEGMENT_NAME, null, ' ',
                        (case when length(SEGMENT_NAME) <= 30
                              then SEGMENT_NAME
                              else substr(SEGMENT_NAME, 1, 30)
                         end)), 30)                                  || ' ' ||
             rpad(decode(BYTES, null, ' ', BYTES), 15)               || ' ' ||
             rpad(decode(BLOCKS, null, ' ', BLOCKS), 10)             || ' ' ||
             rpad(decode(EXTENTS, null, ' ', EXTENTS), 10)  aa
        FROM dba_segments
       WHERE OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND SEGMENT_TYPE = 'TABLE'
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(SEGMENT_NAME, QUEUE_TABLE) >= 1) 
       ORDER BY OWNER, TABLESPACE_NAME, SEGMENT_NAME;


    CURSOR gr_db2b IS 
      SELECT rpad(decode(a.SEGMENT_TYPE, null, ' ', 
                         a.SEGMENT_TYPE), 18)                        || ' ' ||
             rpad(decode(a.OWNER, null, ' ', a.OWNER), 30)           || ' ' ||
             rpad(decode(a.TABLESPACE_NAME, null, ' ', 
                         a.TABLESPACE_NAME), 30)                     || ' ' ||
             rpad(decode(a.SEGMENT_NAME, null, ' ', 
                         a.SEGMENT_NAME), 81)                        || ' ' ||
             rpad(decode(b.TABLE_NAME, null, ' ', b.TABLE_NAME), 30) || ' ' ||
             rpad(decode(a.BYTES, null, ' ', a.BYTES), 15)           || ' ' ||
             rpad(decode(a.BLOCKS, null, ' ', a.BLOCKS), 10)         || ' ' ||
             rpad(decode(a.EXTENTS, null, ' ', a.EXTENTS), 10)  aa
        FROM dba_segments a, dba_indexes b
       WHERE a.OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND a.SEGMENT_TYPE = 'INDEX'
         AND a.OWNER        = b.OWNER
         AND a.SEGMENT_NAME = b.INDEX_NAME
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(b.TABLE_NAME, QUEUE_TABLE) >= 1) 
       ORDER BY a.OWNER, a.TABLESPACE_NAME, a.SEGMENT_NAME;


    CURSOR gr_db2c IS 
      SELECT rpad(decode(a.SEGMENT_TYPE, null, ' ', 
                         a.SEGMENT_TYPE), 18)                        || ' ' ||
             rpad(decode(a.OWNER, null, ' ', a.OWNER), 30)           || ' ' ||
             rpad(decode(a.TABLESPACE_NAME, null, ' ', 
                         a.TABLESPACE_NAME), 30)                     || ' ' ||
             rpad(decode(a.SEGMENT_NAME, null, ' ', 
                         a.SEGMENT_NAME), 81)                        || ' ' ||
             rpad(decode(b.TABLE_NAME, null, ' ', b.TABLE_NAME), 30) || ' ' ||
             rpad(decode(a.BYTES, null, ' ', a.BYTES), 15)           || ' ' ||
             rpad(decode(a.BLOCKS, null, ' ', a.BLOCKS), 10)         || ' ' ||
             rpad(decode(a.EXTENTS, null, ' ', a.EXTENTS), 10)  aa
        FROM dba_segments a, dba_lobs b
       WHERE a.OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND a.SEGMENT_TYPE = 'LOBSEGMENT'
         AND a.OWNER        = b.OWNER
         AND a.SEGMENT_NAME = b.SEGMENT_NAME
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(b.TABLE_NAME, QUEUE_TABLE) >= 1) 
       ORDER BY a.OWNER, a.TABLESPACE_NAME, a.SEGMENT_NAME;


    CURSOR gr_db2d IS 
      SELECT rpad(decode(a.SEGMENT_TYPE, null, ' ', 
                         a.SEGMENT_TYPE), 18)                        || ' ' ||
             rpad(decode(a.OWNER, null, ' ', a.OWNER), 30)           || ' ' ||
             rpad(decode(a.TABLESPACE_NAME, null, ' ', 
                         a.TABLESPACE_NAME), 30)                     || ' ' ||
             rpad(decode(a.SEGMENT_NAME, null, ' ', 
                         a.SEGMENT_NAME), 81)                        || ' ' ||
             rpad(decode(b.TABLE_NAME, null, ' ', b.TABLE_NAME), 30) || ' ' ||
             rpad(decode(a.BYTES, null, ' ', a.BYTES), 15)           || ' ' ||
             rpad(decode(a.BLOCKS, null, ' ', a.BLOCKS), 10)         || ' ' ||
             rpad(decode(a.EXTENTS, null, ' ', a.EXTENTS), 10)  aa
        FROM dba_segments a, dba_lobs b
       WHERE a.OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND a.SEGMENT_TYPE = 'LOBINDEX'
         AND a.OWNER        = b.OWNER
         AND a.SEGMENT_NAME = b.INDEX_NAME
         AND EXISTS(SELECT QUEUE_TABLE FROM aqmon_queue_config_tab
                     WHERE instrb(b.TABLE_NAME, QUEUE_TABLE) >= 1) 
       ORDER BY a.OWNER, a.TABLESPACE_NAME, a.SEGMENT_NAME;


    CURSOR gr_db2e IS 
      SELECT rpad(decode(SEGMENT_TYPE, null, ' ', SEGMENT_TYPE), 18) || ' ' ||
             rpad(decode(OWNER, null, ' ', OWNER), 30)               || ' ' ||
             rpad(decode(TABLESPACE_NAME, null, ' ', 
                         TABLESPACE_NAME), 30)                       || ' ' ||
             rpad(decode(SEGMENT_NAME, null, ' ', SEGMENT_NAME), 81) || ' ' ||
             rpad(' ', 30)                                           || ' ' ||
             rpad(decode(BYTES, null, ' ', BYTES), 15)               || ' ' ||
             rpad(decode(BLOCKS, null, ' ', BLOCKS), 10)             || ' ' ||
             rpad(decode(EXTENTS, null, ' ', EXTENTS), 10)  aa
        FROM dba_segments 
       WHERE OWNER IN (SELECT QUEUE_OWNER FROM aqmon_queue_owners_tab)
         AND SEGMENT_TYPE != 'TABLE'
         AND SEGMENT_TYPE != 'INDEX'
         AND SEGMENT_TYPE != 'LOBSEGMENT'
         AND SEGMENT_TYPE != 'LOBINDEX'
       ORDER BY OWNER, TABLESPACE_NAME, SEGMENT_NAME;

   /*
    CURSOR gr_db3 IS 
      SELECT QUEUE_OWNER, QUEUE_NAME, QUEUE_TABLE
        FROM aqmon_queue_config_tab
       WHERE QUEUE_TYPE = 'NORMAL_QUEUE'
       GROUP BY QUEUE_OWNER, QUEUE_NAME, QUEUE_TABLE
       ORDER BY QUEUE_OWNER, QUEUE_NAME, QUEUE_TABLE; 
   */

    CURSOR gr_db4 IS
      SELECT rpad(decode(rs.rule_set_owner, null, ' ', rs.rule_set_owner), 
                  30)                        || ' ' ||
             rpad(decode(rs.rule_set_name, null, ' ', rs.rule_set_name),
                  30)                        || ' ' ||
             rpad(decode(r.rule_name, null, ' ', r.rule_name),
                  30)                        || ' ' ||             
             to_char(nvl(substr(r.rule_condition, 1, 1000), 
                  to_clob(' ** Rule condition not available ** '))) aa 
       FROM dba_rules r, dba_rule_set_rules rs 
      WHERE rs.rule_name = r.rule_name 
        AND rs.rule_owner = r.rule_owner 
        AND rs.rule_set_owner IN 
            (SELECT queue_owner FROM aqmon_queue_owners_tab);


    PROCEDURE prvt_dump_sorted_queues(sort_metric      IN VARCHAR2,
                                      persistent_queue IN BOOLEAN) AS
      stxt           VARCHAR2(1000);      
      titleword      VARCHAR2(100);
      value          VARCHAR2(100);
      queue_type     VARCHAR2(30);
      table_name     VARCHAR2(40);

      my_inst_id     NUMBER;
      my_queue_id    NUMBER;
      my_queue_name  VARCHAR2(30);
      my_metric      NUMBER;

      TYPE pdsq_refcur IS REF CURSOR;
      pdsq_c1 pdsq_refcur;
    BEGIN
      IF sort_metric IS NULL OR persistent_queue IS NULL THEN
        RETURN;
      END IF;

      IF NOT UTL_FILE.IS_OPEN(AQMON_REPORT_OUTPUT) THEN
        prvt_echo_error('No valid log file.');
        RETURN;
      END IF;

      CASE UPPER(sort_metric) 
        WHEN 'ENQUEUED_MSGS'        THEN titleword:='Enqueued Msgs';
        WHEN 'DEQUEUED_MSGS'        THEN titleword:='Dequeued Msgs';
        WHEN 'ENQUEUE_RATE'         THEN titleword:='Enqueue Rate(msg/s)';
        WHEN 'DEQUEUE_RATE'         THEN titleword:='Dequeue Rate(msg/s)';
        WHEN 'AVG_TIME_PER_ENQUEUE' THEN titleword:='Avg Time per Enqueue(us)';
        WHEN 'AVG_TIME_PER_DEQUEUE' THEN titleword:='Avg Time per Dequeue(us)';
        WHEN 'PENDING_MSGS'         THEN titleword:='Pending Msgs';
        ELSE                             RETURN;
      END CASE;

      IF persistent_queue THEN
        queue_type := 'Persistent Queues';
        table_name := 'aqmon_pqueues_accum_tab';
      ELSE
        queue_type := 'Buffered Queues';
        table_name := 'aqmon_bqueues_accum_tab';
      END IF;

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, breakline);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '' || queue_type ||
                                             ' Ordered by ' || sort_metric);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

      -- in descending order
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '(descending order)'     ||
                                              lpad('Queue Name', 14)  ||
                                              lpad('Queue ID', 11)    ||
                                              lpad('Inst Number', 14) ||   
                                              lpad(titleword, 27));
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

      stxt := 'SELECT INST_ID, QUEUE_ID, QUEUE_NAME, ' || sort_metric ||
              '  FROM ' || table_name || 
              ' ORDER BY ' || sort_metric || ' DESC';
      OPEN pdsq_c1 for stxt;
      LOOP
        FETCH pdsq_c1 INTO my_inst_id, my_queue_id, my_queue_name, 
                           my_metric;
        EXIT WHEN pdsq_c1%ROWCOUNT >= AQMON_TOP_N_QUEUE OR pdsq_c1%NOTFOUND;
     
        CASE UPPER(sort_metric) 
          WHEN 'ENQUEUED_MSGS' THEN value:=to_char(my_metric);
          WHEN 'DEQUEUED_MSGS' THEN value:=to_char(my_metric);
          WHEN 'ENQUEUE_RATE'  THEN value:=to_char(my_metric, '9999999.00');
          WHEN 'DEQUEUE_RATE'  THEN value:=to_char(my_metric, '9999999.00');
          WHEN 'AVG_TIME_PER_ENQUEUE'  THEN value := 
            to_char(my_metric*10000, '99999999.00');
          WHEN 'AVG_TIME_PER_DEQUEUE'  THEN value := 
            to_char(my_metric*10000, '99999999.00');
          WHEN 'PENDING_MSGS'  THEN value:=to_char(my_metric);
          ELSE                      value := ' ';
        END CASE;

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 
                          lpad(my_queue_name, 32) ||
                          lpad(my_queue_id, 11)   ||
                          lpad(my_inst_id, 14)    ||
                          lpad(value, 27));        
      END LOOP;
      CLOSE pdsq_c1;

      -- in ascending order
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '(ascending order)'      ||
                                              lpad('Queue Name', 15)  ||
                                              lpad('Queue ID', 11)    ||
                                              lpad('Inst Number', 14) ||
                                              lpad(titleword, 27));
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

      stxt := 'SELECT INST_ID, QUEUE_ID, QUEUE_NAME, ' || sort_metric ||
              '  FROM ' || table_name || ' ORDER BY ' || sort_metric;

      OPEN pdsq_c1 for stxt;
      LOOP
        FETCH pdsq_c1 INTO my_inst_id, my_queue_id, my_queue_name, 
                           my_metric;
        EXIT WHEN pdsq_c1%ROWCOUNT >= AQMON_TOP_N_QUEUE OR pdsq_c1%NOTFOUND;
     
        CASE UPPER(sort_metric) 
          WHEN 'ENQUEUED_MSGS' THEN value:=to_char(my_metric);
          WHEN 'DEQUEUED_MSGS' THEN value:=to_char(my_metric);
          WHEN 'ENQUEUE_RATE'  THEN value:=to_char(my_metric, '9999999.00');
          WHEN 'DEQUEUE_RATE'  THEN value:=to_char(my_metric, '9999999.00');
          WHEN 'AVG_TIME_PER_ENQUEUE'  THEN value := 
            to_char(my_metric*10000, '99999999.00');
          WHEN 'AVG_TIME_PER_DEQUEUE'  THEN value := 
            to_char(my_metric*10000, '99999999.00');
          WHEN 'PENDING_MSGS'  THEN value:=to_char(my_metric);
          ELSE                      value := ' ';
        END CASE;
           
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 
                          lpad(my_queue_name, 32) ||
                          lpad(my_queue_id, 11)   ||
                          lpad(my_inst_id, 14)    ||
                          lpad(value, 27));  
      END LOOP;
      CLOSE pdsq_c1;

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

    END prvt_dump_sorted_queues;


    FUNCTION prvt_get_sys_param(mydbid        NUMBER,
                                myinst_num    NUMBER,
                                mybeginsnp    NUMBER,
                                myendsnp      NUMBER,
                                myparam_name  VARCHAR2) 
    RETURN VARCHAR2 IS
      beginvalue VARCHAR2(512);
      endvalue   VARCHAR2(512);
      result     VARCHAR2(1500);
     
      CURSOR getvalue(mysnp NUMBER) IS
        SELECT VALUE FROM dba_hist_parameter
         WHERE DBID            = mydbid
           AND INSTANCE_NUMBER = myinst_num
           AND PARAMETER_NAME  = myparam_name
           AND SNAP_ID         = mysnp;
    BEGIN
      IF mydbid IS NULL OR myinst_num IS NULL OR mybeginsnp IS NULL OR
         myendsnp IS NULL OR myparam_name IS NULL THEN
        return 'null';
      END IF;
     
      FOR i in getvalue(mybeginsnp) LOOP
        beginvalue := i.VALUE;
      END LOOP;
      IF beginvalue IS NULL THEN
        beginvalue := 'null';
      END IF;

      FOR i in getvalue(myendsnp) LOOP
        endvalue := i.VALUE;
      END LOOP;
      IF endvalue IS NULL THEN
        endvalue := 'null';
      END IF;

      IF beginvalue = endvalue THEN
        result := beginvalue;
      ELSE
        result := beginvalue || ' (begin value),  ' || 
                    endvalue || ' (end value)';
      END IF;

      return result;
    END prvt_get_sys_param;

  BEGIN
    mon_status := prvt_get_param('APT_MONITOR_STATUS');
    IF mon_status IS NULL OR mon_status = AQMON_STATUS_NULL THEN
      prvt_echo_error('Monitor is not started yet.');
      RETURN;
    END IF;
 
    IF final_collect IS NULL THEN
      prvt_echo_error('Parameter final_collect cannot be NULL.');
      RETURN;  
    END IF;

    -- Get system info
    BEGIN
      SELECT DIRECTORY_PATH INTO logdir_path 
        FROM sys.dba_directories
       WHERE DIRECTORY_NAME = AQMON_LOGDIR;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        logdir_path := NULL;
    END;

    db_version := prvt_get_param('APT_DATABASE_VERSION');
    
    mon_beginsnp := prvt_get_param('APT_BEGIN_SNAPSHOT_ID');
    IF mon_beginsnp < 0 THEN
      prvt_echo_error('The beginning snapshot is not recorded.');
    END IF;

    begin_monitor_tm := prvt_get_param2('APT_BEGIN_MONITOR_TM'); 
    IF begin_monitor_tm IS NULL THEN
      prvt_echo_error('Monitor start time is not recorded.');
    END IF;

    -- If the monitor is still running, set the latest snap_id to 
    -- APT_END_SNAPSHOT_ID, also set APT_END_MONITOR_TM.
    IF mon_status IS NOT NULL AND mon_status = AQMON_STATUS_START THEN      
      SELECT max(SNAP_ID) INTO mon_endsnp
        FROM sys.dba_hist_snapshot;
      IF mon_endsnp IS NULL THEN
        mon_endsnp := -1;
      END IF;
      prvt_set_param('APT_END_SNAPSHOT_ID', mon_endsnp);

      SELECT systimestamp INTO end_monitor_tm FROM DUAL;
      prvt_set_param2('APT_END_MONITOR_TM', end_monitor_tm);      
    END IF;

    mon_endsnp     := prvt_get_param('APT_END_SNAPSHOT_ID');
    end_monitor_tm := prvt_get_param2('APT_END_MONITOR_TM'); 

    BEGIN
      SELECT dbid INTO myDBID FROM v$database;
    EXCEPTION
      WHEN OTHERS THEN
        myDBID := -1;
    END;

    BEGIN
      AQMON_REPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_REPORT_FNAME,
                                          'w', 5000);
    EXCEPTION
      WHEN OTHERS THEN
        prvt_echo_error('Cannot create log file ' || AQMON_REPORT_FNAME ||
                        ' in directory ' || logdir_path || '.');
    END;

    IF NOT UTL_FILE.IS_OPEN(AQMON_REPORT_OUTPUT) THEN
      prvt_echo_error('Cannot create log file ' || AQMON_REPORT_FNAME ||
                      ' in directory ' || logdir_path || '.');
      RETURN;
    END IF;

    -- dump one snapshot before generating report
    IF final_collect AND mon_status = AQMON_STATUS_START THEN
      collect_runtime;
      collect_config;
      DBMS_WORKLOAD_REPOSITORY.CREATE_SNAPSHOT;
    END IF;

    -------------------------------------------------------------------------- 
    -- sec 1: dump configuration info                                        
    -------------------------------------------------------------------------- 

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Summary Report For All Queues');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'System Configuration');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~~~~~~');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Monitor Start Time : ' || 
                                              begin_monitor_tm);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Monitor End Time   : ' || 
                                              end_monitor_tm);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Database ID        : ' ||
                                              myDBID);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Snapshots ID       : ' || 
                                              mon_beginsnp  || ' - ' ||
                                              mon_endsnp);

    -- dump per instance info
    FOR i IN gr_instCursor LOOP

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
       
      -- info from gv$instnace
      FOR j1 IN gr_conf0a(i.INST_ID) LOOP
        -- there shall be one record per inst_id
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Inst ID      : ' ||
                                                  j1.INST_ID);
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Inst Number  : ' ||
                                                  j1.INSTANCE_NUMBER);
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Inst Name    : ' ||
                                                  j1.INSTANCE_Name);
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Host Name    : ' ||
                                                  j1.HOST_Name);
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Version      : ' ||
                                                  j1.VERSION);
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Startup Date : ' ||
                                                  j1.STARTUP_TIME);
        -- system parameters
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  Parameters    : (Begin ' || 
          'value will be shown below. End value will be shown only if ' || 
          'different.)');  

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - streams_pool_size   : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'streams_pool_size'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - shared_pool_size    : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'shared_pool_size'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - sga_max_size        : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'sga_max_size'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - sga_target          : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'sga_target'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - memory_max_target   : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'memory_max_target'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - memory_target       : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'memory_target'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - compatible          : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'compatible'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - processes           : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'processes'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - sessions            : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'sessions'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - aq_tm_processes     : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'aq_tm_processes'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - job_queue_processes : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'job_queue_processes'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - _job_queue_interval : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, '_job_queue_interval'));

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT,'    - global_names        : '||
          prvt_get_sys_param(myDBID, j1.INSTANCE_NUMBER, mon_beginsnp, 
                             mon_endsnp, 'global_names'));
      END LOOP;  
    END LOOP;

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
            

    -- dump queue owners in the monitoring list
    SELECT count(QUEUE_OWNER) INTO cnt
      FROM aqmon_queue_owners_tab;

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Queue Owners in the ' ||
                                           'Monitoring List');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 35, '~'));

    IF cnt = 0 THEN
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 
                        '  No queue owner found.');
    ELSE
      IF cnt > 1 THEN  plural := 's';
      ELSE             plural := NULL;
      END IF;

      FOR i IN gr_ownerCursor LOOP
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '  ' || i.QUEUE_OWNER);
      END LOOP;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' queue owner' || 
                                             plural || ' found.');

    END IF;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  


    var := rpad('Queue Name', 30)  || ' ' ||
           rpad('Queue ID', 10)    || ' ' ||
           rpad('Queue Table', 30) || ' ' ||
           rpad('Queue Owner', 30) || ' ' ||
           rpad('Data Type', 10)   || ' ' || 
           rpad('Object Type', 62) || ' ' ||
           rpad('Sort Order', 22);

    -- statistics on single/multiple-consumer queues
    tab_rec := TABLE_RECIPENTS('Single', 'Multiple');    
    FOR elem in 1..tab_rec.count LOOP     
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'All ' || tab_rec(elem) || 
                                             '-Consumer Queues');
      IF elem = 1 THEN 
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 26, '~'));
      ELSE
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 28, '~'));
      END IF;
    
      cnt := 0;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 202, '-'));
      FOR i IN gr_conf1(upper(tab_rec(elem))) LOOP
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);
        cnt := gr_conf1%ROWCOUNT;
      END LOOP;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      IF cnt > 1 THEN 
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' queues found.');  
      ELSE
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' queue found.');  
      END IF;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    END LOOP;


    -- statistics on transactional/non-transactional queues
    tab_grp := TABLE_GROUPING('Transactional');   --'None' 
    FOR elem in 1..tab_grp.count LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'All ' || tab_grp(elem) || 
                                             '-Grouping Queues');
      IF elem = 1 THEN
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 33, '~'));
      ELSE
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 24, '~'));
      END IF;    

      cnt := 0;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 202, '-'));
      FOR i IN gr_conf2(upper(tab_grp(elem))) LOOP
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);
        cnt := gr_conf2%ROWCOUNT;
      END LOOP;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      IF cnt > 1 THEN 
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' queues found.');  
      ELSE
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' queue found.');  
      END IF;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    END LOOP;


    -- statistics on subscribers
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'All Subscribers');    
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~');

    cnt := 0;
    var := rpad('Queue Name', 30)      || ' ' ||
           rpad('Queue Table', 30)     || ' ' || 
           rpad('Consumer Name', 30)   || ' ' || 
           rpad('Owner', 30)           || ' ' || 
           rpad('Transformation', 61)  || ' ' ||
           rpad('Delivery Mode', 22)   || ' ' ||
           rpad('Queue to Queue', 17)  || ' ' ||
           rpad('Address', 200);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 275, '-'));
    FOR i IN gr_conf3 LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);
      cnt := gr_conf3%ROWCOUNT;
    END LOOP;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    IF cnt > 1 THEN 
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' subscribers found.');  
    ELSE
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, cnt || ' subscriber found.');  
    END IF;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  


    -------------------------------------------------------------------------- 
    -- sec 2: dump database-level info                 
    -------------------------------------------------------------------------- 

    IF begin_monitor_tm IS NOT NULL AND end_monitor_tm IS NOT NULL THEN
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Queue Related Session Info');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~~~~~~~~~~~~'); 
      var := rpad('Session ID', 30)                  || ' ' ||
             rpad('Session Type', 13)                || ' ' ||
             rpad('Program', 50)                     || ' ' ||
             rpad('CPU Time in Session (us)', 30)    || ' ' ||
             rpad('CPU Time Fraction(%)', 30)        || ' ' ||
             rpad('DB Time in Session (us)', 30)     || ' ' ||
             rpad('DB Time Fraction(%)', 30);      
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 210, '-'));
      FOR i in gr_db0(begin_monitor_tm, end_monitor_tm) LOOP
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
      END LOOP;
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    END IF;

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Rule Set Info');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~');
    var := rpad('Rule Set Owner', 30)         || ' ' ||
           rpad('Rule Set Name', 30)          || ' ' ||
           rpad('Rule Name', 30)              || ' ' ||
           'Rule Condition';   
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 109, '-'));
    FOR i in gr_db4 LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);
    END LOOP; 
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  


    -- display PL/SQL notification info
    IF db_version >= 11.1 THEN

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'PL/SQL Notification Info');  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~~~~~~~~~~');

      cnt := 0;
      sqlstxt := 'SELECT rpad(REG_ID, 8) || ''   '' || ' ||
                 ' rpad(nvl(SUBSCRIPTION_NAME, '' ''), 128) || ''   '' || ' ||
                 ' LOCATION_NAME aa, LOCATION_NAME ' ||
                 '  FROM dba_subscr_registrations' ||
                 ' WHERE UPPER(LOCATION_NAME) LIKE ''PLSQL%''';

      OPEN gr_c1 for sqlstxt;
      LOOP
        FETCH gr_c1 INTO var, ntfn_loc_name;
        EXIT WHEN gr_c1%NOTFOUND;
         
        FOR i in gr_ownerCursor LOOP
          IF instr(UPPER(ntfn_loc_name), UPPER(i.QUEUE_OWNER)) > 0 THEN
            
            IF cnt = 0 THEN 
              UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('Reg ID', 11) ||
                                   rpad('Subscription Name', 131) ||
                                   'Location Name');
              UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 200, '-'));
            END IF;
            UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
 
            cnt := cnt+1;

          END IF; 
        END LOOP;        

      END LOOP;
      CLOSE gr_c1;


      IF cnt > 0 THEN                      
        IF db_version = 11.1 THEN
          ntfn_queue_name := 'AQ_SRVNTFN_TABLE_Q';
        ELSIF db_version = 11.2 THEN
          ntfn_queue_name := 'AQ_SRVNTFN_TABLE_Q_1';
        END IF;

        sqlstxt := 'SELECT FIRST_ACTIVITY_TIME, ENQUEUED_MSGS, ' ||
          'DEQUEUED_MSGS, ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME,' ||
          'LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME' ||
          '  FROM gv$persistent_queues' ||
          ' WHERE QUEUE_SCHEMA = ''SYS''' ||
          '   AND QUEUE_NAME   = ''' || ntfn_queue_name || '''';
         
        BEGIN
          EXECUTE IMMEDIATE sqlstxt 
            INTO ntfn_first_act_tm, ntfn_enq_msgs, ntfn_deq_msgs,
                 ntfn_ela_enq_tm, ntfn_ela_deq_tm, ntfn_ls_enq_tm, 
                 ntfn_ls_deq_tm;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Server Notification Queue: ' ||
                     ntfn_queue_name);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'First Activity Time      : ' ||
                     ntfn_first_act_tm);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Enqueued Msgs            : ' ||
                     ntfn_enq_msgs);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Dequeued Msgs            : ' ||
                     ntfn_deq_msgs);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Elapsed Enqueue time(us) : ' ||
                     ntfn_ela_enq_tm*10000);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Elapsed Dequeue time(us) : ' ||
                     ntfn_ela_deq_tm*10000);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Last Enqueue Time        : ' ||
                     ntfn_ls_enq_tm);  
        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Last Dequeue Time        : ' ||
                     ntfn_ls_deq_tm);  
      END IF;

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '');

    END IF;    


    -------------------------------------------------------------------------- 
    -- sec 3: dump accumulative stats for persistent queues                  
    -------------------------------------------------------------------------- 
    IF db_version >= 11.1 THEN 

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Accumulative Statistics'|| 
                                             ' on Persistent Queues');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 44, '~'));

      -- dump complete accumulative stats
      IF db_version >= 11.2 THEN 
  
        title_line1_p1 := 'Instance                          Queue      Queue   Enqueued   Dequeued   Browsed      Enqueued     Enqueued   Msgs Made   Msgs Made       Enqueue       Dequeue';
        title_line2_p1 := '  Number                           Name         ID       Msgs       Msgs      Msgs   Expiry Msgs   Delay Msgs     Expired       Ready   Rate(msg/s)   Rate(msg/s)';

        title_line1_p2 := '   Avg Time per   Avg Time per      Avg CPU Time      Avg CPU Time   Enqueue   Dequeue   Execution   Transformation   Rule Eval  |  ';
        title_line2_p2 := '    Enqueue(us)    Dequeue(us)   per Enqueue(us)   per Dequeue(us)     Trans     Trans       Count         Time(us)    Time(us)  |  ';

        title_line1_p3 := 'Pending   Avg Msg   Dequeue Msg              Last Enqueue Time              Last Dequeue Time       Last Time Manager Expiry';
        title_line2_p3 := '   Msgs    Age(s)    Latency(s)                 (UTC timezone)                 (UTC timezone)            Time (UTC timezone)';

        title_line1_p4 := '   Last Time Manager Ready Time  |         Time                          First                           Last';
        title_line2_p4 := '                 (UTC timezone)  |  Duration(s)                   Monitor Time                   Monitor Time';

        title_line3    := rpad('-', 528, '-');

      ELSIF db_version=11.1 THEN

        title_line1_p1 := 'Instance                          Queue      Queue   Enqueued   Dequeued   Browsed      Enqueued     Enqueued   Msgs Made   Msgs Made       Enqueue       Dequeue';
        title_line2_p1 := '  Number                           Name         ID       Msgs       Msgs      Msgs   Expiry Msgs   Delay Msgs     Expired       Ready   Rate(msg/s)   Rate(msg/s)';

        title_line1_p2 := '   Avg Time per   Avg Time per   Transformation   Rule Eval  |  ';
        title_line2_p2 := '    Enqueue(us)    Dequeue(us)         Time(us)    Time(us)  |  ';

        title_line1_p3 := 'Pending              Last Enqueue Time              Last Dequeue Time       Last Time Manager Expiry';
        title_line2_p3 := '   Msgs                 (UTC timezone)                 (UTC timezone)            Time (UTC timezone)';

        title_line1_p4 := '   Last Time Manager Ready Time  |         Time                          First                           Last';
        title_line2_p4 := '                 (UTC timezone)  |  Duration(s)                   Monitor Time                   Monitor Time';

        title_line3    := rpad('-', 436, '-');
  
      END IF;
    

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line1_p1 || title_line1_p2
                                          || title_line1_p3 || title_line1_p4);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line2_p1 || title_line2_p2
                                          || title_line2_p3 || title_line2_p4);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line3);


      sqlstxt := 'SELECT INST_ID, QUEUE_SCHEMA, QUEUE_ID, QUEUE_NAME ' ||
         ' FROM aqmon_pqueues_tab WHERE INST_ID IS NOT NULL ' ||
         ' AND QUEUE_SCHEMA IS NOT NULL AND QUEUE_ID IS NOT NULL ' ||
         ' GROUP BY INST_ID, QUEUE_SCHEMA, QUEUE_NAME, QUEUE_ID ' ||
         ' ORDER BY INST_ID, QUEUE_SCHEMA, QUEUE_NAME, QUEUE_ID';
      OPEN gr_c1 for sqlstxt;
      LOOP
        FETCH gr_c1 INTO my_inst_id, my_queue_schema, my_queue_id, 
                         my_queue_name;
        EXIT WHEN gr_c1%NOTFOUND;

        -- adjust collected data (e.g., first snapshot) before analysis
        prvt_postrun_adjust(db_version, my_inst_id, my_queue_id, TRUE);

        -- get the first snapshot
        BEGIN
          EXECUTE IMMEDIATE 'SELECT QUEUE_ID, ENQUEUED_MSGS, DEQUEUED_MSGS,'||
            ' BROWSED_MSGS, ENQUEUED_EXPIRY_MSGS, ENQUEUED_DELAY_MSGS, ' ||
            ' MSGS_MADE_EXPIRED, MSGS_MADE_READY, ' ||
            ' ELAPSED_TRANSFORMATION_TIME, ELAPSED_RULE_EVALUATION_TIME,' ||
            ' ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, MONITOR_TIME' ||  
            '  FROM aqmon_pqueues_tab ' ||
            ' WHERE INST_ID      = ' || my_inst_id ||
            '   AND QUEUE_SCHEMA = ' || prvt_enquote_str(my_queue_schema) || 
            '   AND QUEUE_ID     = ' || my_queue_id ||
            '   AND QUEUE_NAME   = ' || prvt_enquote_str(my_queue_name) || 
            '   AND memseq = (SELECT min(memseq)  FROM aqmon_pqueues_tab ' ||
            '  WHERE INST_ID     = ' || my_inst_id ||
            '    AND QUEUE_SCHEMA= ' || prvt_enquote_str(my_queue_schema) ||
            '    AND QUEUE_ID    = ' || my_queue_id  ||
            '    AND QUEUE_NAME  = ' || prvt_enquote_str(my_queue_name)||')'
            INTO pq_queue_id_1,
                 pq_enqueued_msgs_1, pq_dequeued_msgs_1, pq_browsed_msgs_1,
                 pq_enqueued_expiry_msgs_1, pq_enqueued_delay_msgs_1,
                 pq_msgs_made_expired_1, pq_msgs_made_ready_1,
                 pq_trans_time_1, pq_rule_eval_time_1, pq_ela_enqueue_time_1,
                 pq_ela_dequeue_time_1, pq_monitor_time_1; 
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

        -- get the last snapshot
        BEGIN
          EXECUTE IMMEDIATE 'SELECT QUEUE_ID, ENQUEUED_MSGS, DEQUEUED_MSGS,'||
            ' BROWSED_MSGS, ENQUEUED_EXPIRY_MSGS, ENQUEUED_DELAY_MSGS, ' ||
            ' MSGS_MADE_EXPIRED, MSGS_MADE_READY, ' ||
            ' ELAPSED_TRANSFORMATION_TIME, ELAPSED_RULE_EVALUATION_TIME,' ||
            ' ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, ' ||
            ' LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME, LAST_TM_EXPIRY_TIME,' ||
            ' LAST_TM_READY_TIME, MONITOR_TIME' ||  
            '  FROM aqmon_pqueues_tab ' ||
            ' WHERE INST_ID      = ' || my_inst_id ||
            '   AND QUEUE_SCHEMA = ' || prvt_enquote_str(my_queue_schema) || 
            '   AND QUEUE_ID     = ' || my_queue_id ||
            '   AND QUEUE_NAME   = ' || prvt_enquote_str(my_queue_name) || 
            '   AND memseq = (SELECT max(memseq)  FROM aqmon_pqueues_tab ' ||
            '  WHERE INST_ID     = ' || my_inst_id ||
            '    AND QUEUE_SCHEMA= ' || prvt_enquote_str(my_queue_schema) ||
            '    AND QUEUE_ID    = ' || my_queue_id  ||
            '    AND QUEUE_NAME  = ' || prvt_enquote_str(my_queue_name)||')'
            INTO pq_queue_id_2,
                 pq_enqueued_msgs_2, pq_dequeued_msgs_2, pq_browsed_msgs_2,
                 pq_enqueued_expiry_msgs_2, pq_enqueued_delay_msgs_2,
                 pq_msgs_made_expired_2, pq_msgs_made_ready_2,
                 pq_trans_time_2, pq_rule_eval_time_2, pq_ela_enqueue_time_2,
                 pq_ela_dequeue_time_2, pq_last_enqtm_accum, 
                 pq_last_deqtm_accum, pq_last_tm_expitm_accum,
                 pq_last_tm_readtm_accum, pq_monitor_time_2;
        EXCEPTION WHEN OTHERS THEN NULL;
        END;

    
        --calculate accumulative statistics
        IF (pq_queue_id_1 IS NOT NULL) AND (pq_queue_id_2 IS NOT NULL) THEN
    
          -- unit: second          
          pq_time_duration:=(CAST(pq_monitor_time_2 as DATE)-
                             CAST(pq_monitor_time_1 as DATE))*86400.0;

          pq_enqueued_msgs_td       := nvl(pq_enqueued_msgs_2, 0)  
                                     - nvl(pq_enqueued_msgs_1, 0);
          pq_dequeued_msgs_td       := nvl(pq_dequeued_msgs_2, 0)  
                                     - nvl(pq_dequeued_msgs_1, 0);
          pq_browsed_msgs_td        := nvl(pq_browsed_msgs_2, 0)   
                                     - nvl(pq_browsed_msgs_1, 0);
          pq_enqueued_expiry_msgs_td:= nvl(pq_enqueued_expiry_msgs_2,0) 
                                     - nvl(pq_enqueued_expiry_msgs_1,0);
          pq_enqueued_delay_msgs_td := nvl(pq_enqueued_delay_msgs_2, 0)  
                                     - nvl(pq_enqueued_delay_msgs_1, 0);
          pq_msgs_made_expired_td   := nvl(pq_msgs_made_expired_2, 0)    
                                     - nvl(pq_msgs_made_expired_1, 0); 
          pq_msgs_made_ready_td     := nvl(pq_msgs_made_ready_2, 0)      
                                     - nvl(pq_msgs_made_ready_1, 0);
          pq_trans_time_td          := nvl(pq_trans_time_2, 0)
                                     - nvl(pq_trans_time_1, 0);
          pq_rule_eval_time_td      := nvl(pq_rule_eval_time_2, 0)
                                     - nvl(pq_rule_eval_time_1, 0);

          IF pq_time_duration = 0 THEN
            pq_enqueue_rate_td := 0;
            pq_dequeue_rate_td := 0;
          ELSE 
            pq_enqueue_rate_td  := pq_enqueued_msgs_td/pq_time_duration; 
            pq_dequeue_rate_td  := pq_dequeued_msgs_td/pq_time_duration; 
          END IF;
        
          IF pq_enqueued_msgs_td = 0 THEN 
            pq_avg_time_per_enq_td := 0;
          ELSE
            pq_avg_time_per_enq_td :=(nvl(pq_ela_enqueue_time_2, 0)
              - nvl(pq_ela_enqueue_time_1, 0))/pq_enqueued_msgs_td;
          END IF;

          IF pq_dequeued_msgs_td = 0 THEN 
            pq_avg_time_per_deq_td := 0;
          ELSE
            pq_avg_time_per_deq_td :=(nvl(pq_ela_dequeue_time_2, 0)
               - nvl(pq_ela_dequeue_time_1, 0))/pq_dequeued_msgs_td;
          END IF;

          pq_pending_msgs_accum := nvl(pq_enqueued_msgs_2, 0)
                                 - nvl(pq_dequeued_msgs_2, 0);         


          IF db_version >= 11.2 THEN
            -- metrics for 11.2 or above
            BEGIN 
              var := 'SELECT ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME, ' || 
                '            ENQUEUE_TRANSACTIONS, DEQUEUE_TRANSACTIONS, ' ||
                '            EXECUTION_COUNT' ||
                '  FROM aqmon_pqueues_tab ' ||
                ' WHERE INST_ID      =' || my_inst_id ||
                '   AND QUEUE_SCHEMA =' || prvt_enquote_str(my_queue_schema) ||
                '   AND QUEUE_ID     =' || my_queue_id ||
                '   AND QUEUE_NAME   =' || prvt_enquote_str(my_queue_name) ||
                '   AND memseq = (SELECT min(memseq) FROM aqmon_pqueues_tab' ||
                '   WHERE INST_ID    =' ||my_inst_id ||
                '     AND QUEUE_SCHEMA='||prvt_enquote_str(my_queue_schema)|| 
                '     AND QUEUE_ID   =' ||my_queue_id ||
                '     AND QUEUE_NAME =' ||prvt_enquote_str(my_queue_name)||')';
              EXECUTE IMMEDIATE var INTO pq_enqueue_cpu_time_1, 
                pq_dequeue_cpu_time_1, pq_enqueue_trans_1, pq_dequeue_trans_1,
                pq_execution_count_1;
            EXCEPTION WHEN OTHERS THEN NULL;
            END;

            BEGIN 
              var := 'SELECT ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME, ' || 
               '       ENQUEUE_TRANSACTIONS, DEQUEUE_TRANSACTIONS, ' ||
               '       EXECUTION_COUNT, AVG_MSG_AGE, ' ||
               '       DEQUEUED_MSG_LATENCY ' ||
               '  FROM aqmon_pqueues_tab ' ||
               ' WHERE INST_ID      = ' || my_inst_id ||
               '   AND QUEUE_SCHEMA = ' || prvt_enquote_str(my_queue_schema) ||
               '   AND QUEUE_ID     = ' || my_queue_id ||
               '   AND QUEUE_NAME   = ' || prvt_enquote_str(my_queue_name)||
               '   AND memseq = (SELECT max(memseq) FROM aqmon_pqueues_tab' ||
               '    WHERE INST_ID     ='||my_inst_id ||
               '      AND QUEUE_SCHEMA='||prvt_enquote_str(my_queue_schema) ||
               '      AND QUEUE_ID    ='||my_queue_id ||
               '      AND QUEUE_NAME  ='||prvt_enquote_str(my_queue_name)||')';
              EXECUTE IMMEDIATE var INTO pq_enqueue_cpu_time_2, 
                pq_dequeue_cpu_time_2, pq_enqueue_trans_2, pq_dequeue_trans_2,
                pq_execution_count_2, pq_avg_msg_age_accum, 
                pq_deq_msg_latency_accum;
            EXCEPTION WHEN OTHERS THEN NULL;
            END;

            pq_enqueue_trans_td     := nvl(pq_enqueue_trans_2, 0) 
                                     - nvl(pq_enqueue_trans_1,0);
            pq_dequeue_trans_td     := nvl(pq_dequeue_trans_2, 0) 
                                     - nvl(pq_dequeue_trans_1,0);
            pq_execution_count_td   := nvl(pq_execution_count_2, 0)      
                                     - nvl(pq_execution_count_1, 0);

            IF pq_enqueued_msgs_td = 0 THEN 
              pq_cpu_time_per_enq_td := 0;
            ELSE
              pq_cpu_time_per_enq_td := (nvl(pq_enqueue_cpu_time_2, 0)
                - nvl(pq_enqueue_cpu_time_1, 0))/pq_enqueued_msgs_td;
            END IF;

            IF pq_dequeued_msgs_td = 0 THEN 
              pq_cpu_time_per_deq_td := 0;
            ELSE
              pq_cpu_time_per_deq_td := (nvl(pq_dequeue_cpu_time_2, 0)
                 - nvl(pq_dequeue_cpu_time_1, 0))/pq_dequeued_msgs_td;
            END IF;

            pq_avg_msg_age_accum     := nvl(pq_avg_msg_age_accum,0);
            pq_deq_msg_latency_accum := nvl(pq_deq_msg_latency_accum,0);
          ELSE
            -- for 11.1
            pq_cpu_time_per_enq_td    := 0;
            pq_cpu_time_per_deq_td    := 0;
            pq_enqueue_trans_td       := 0;
            pq_dequeue_trans_td       := 0;
            pq_execution_count_td     := 0;       
            pq_avg_msg_age_accum      := 0;
            pq_deq_msg_latency_accum  := 0;
          END IF;

          EXECUTE IMMEDIATE 'DELETE FROM aqmon_pqueues_accum_tab WHERE ' ||
                   ' INST_ID=:1 AND QUEUE_ID=:2'
            USING my_inst_id, my_queue_id;

          EXECUTE IMMEDIATE 'INSERT INTO aqmon_pqueues_accum_tab VALUES ' ||
                   ' (:1, :2, :3, :4, :5, :6, :7, :8, :9, :10, :11, :12, ' ||
                   ' :13, :14, :15, :16, :17, :18, :19, :20, :21, :22, ' ||
                   ' :23, :24, :25, :26, :27, :28, :29, :30, :31)'
            USING my_inst_id, my_queue_id, my_queue_name,
              pq_enqueued_msgs_td, pq_dequeued_msgs_td, pq_browsed_msgs_td,
              pq_enqueued_expiry_msgs_td, pq_enqueued_delay_msgs_td,
              pq_msgs_made_expired_td, pq_msgs_made_ready_td, 
              pq_enqueue_rate_td, pq_dequeue_rate_td, pq_avg_time_per_enq_td,
              pq_avg_time_per_deq_td, pq_trans_time_td, 
              pq_rule_eval_time_td, pq_pending_msgs_accum,  
              pq_last_enqtm_accum, pq_last_deqtm_accum, 
              pq_last_tm_expitm_accum, pq_last_tm_readtm_accum, 
              pq_cpu_time_per_enq_td, pq_cpu_time_per_deq_td, 
              pq_enqueue_trans_td, pq_dequeue_trans_td, pq_execution_count_td,
              pq_avg_msg_age_accum, pq_deq_msg_latency_accum,
              pq_time_duration, pq_monitor_time_1, pq_monitor_time_2;


          IF db_version >= 11.2 THEN
            var := lpad(my_inst_id, 8)                              || ' ' ||
              lpad(my_queue_name, 30)                               || ' ' || 
              lpad(my_queue_id, 10)                                 || ' ' ||
              lpad(pq_enqueued_msgs_td, 10)                         || ' ' ||
              lpad(pq_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(pq_browsed_msgs_td,9)                            || ' ' ||
              lpad(pq_enqueued_expiry_msgs_td, 13)                  || ' ' ||
              lpad(pq_enqueued_delay_msgs_td,12)                    || ' ' ||
              lpad(pq_msgs_made_expired_td, 11)                     || ' ' ||
              lpad(pq_msgs_made_ready_td, 11)                       || ' ' ||
              lpad(to_char(pq_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(pq_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(pq_avg_time_per_enq_td*10000, 'FM99999999.00'), 14) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_avg_time_per_deq_td*10000, 'FM99999999.00'), 14) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_cpu_time_per_enq_td*10000, 'FM99999999.00'), 17) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_cpu_time_per_deq_td*10000, 'FM99999999.00'), 17) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(pq_enqueue_trans_td, 9)                          || ' ' ||
              lpad(pq_dequeue_trans_td, 9)                          || ' ' ||
              lpad(pq_execution_count_td, 11)                       || ' ' ||
              lpad(to_char(pq_trans_time_td*10000, 'FM99999999.00'), 16)  
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_rule_eval_time_td*10000, 'FM99999999.00'), 11) 
                          || '  |' ||  -- from centi-second to micro-second
              lpad(pq_pending_msgs_accum, 9)                        || ' ' ||
              lpad(pq_avg_msg_age_accum, 9)                         || ' ' ||
              lpad(pq_deq_msg_latency_accum, 13)                    || ' ' ||
              lpad(nvl(to_char(pq_last_enqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(pq_last_deqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(pq_last_tm_expitm_accum), ' '),30)   || ' ' ||
              lpad(nvl(to_char(pq_last_tm_readtm_accum), ' '),30)|| '  | ' ||
              lpad(to_char(pq_time_duration, '99999999.00'), 12)    || ' ' ||
              lpad(pq_monitor_time_1, 30)                           || ' ' ||
              lpad(pq_monitor_time_2, 30); 
          ELSIF db_version = 11.1 THEN
            var := lpad(my_inst_id, 8)                              || ' ' ||
              lpad(my_queue_name, 30)                               || ' ' || 
              lpad(my_queue_id, 10)                                 || ' ' ||
              lpad(pq_enqueued_msgs_td, 10)                         || ' ' ||
              lpad(pq_dequeued_msgs_td, 10)                         || ' ' ||
              lpad(pq_browsed_msgs_td,9)                            || ' ' ||
              lpad(pq_enqueued_expiry_msgs_td, 13)                  || ' ' ||
              lpad(pq_enqueued_delay_msgs_td,12)                    || ' ' ||
              lpad(pq_msgs_made_expired_td, 11)                     || ' ' ||
              lpad(pq_msgs_made_ready_td, 11)                       || ' ' ||
              lpad(to_char(pq_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(pq_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
              lpad(to_char(pq_avg_time_per_enq_td*10000, 'FM99999999.00'), 14) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_avg_time_per_deq_td*10000, 'FM99999999.00'), 14) 
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_trans_time_td*10000, 'FM99999999.00'), 16)  
                          || ' ' ||    -- from centi-second to micro-second
              lpad(to_char(pq_rule_eval_time_td*10000, 'FM99999999.00'), 11) 
                          || '  |' ||  -- from centi-second to micro-second
              lpad(pq_pending_msgs_accum, 9)                        || ' ' ||
              lpad(nvl(to_char(pq_last_enqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(pq_last_deqtm_accum), ' '), 30)      || ' ' ||
              lpad(nvl(to_char(pq_last_tm_expitm_accum), ' '),30)   || ' ' ||
              lpad(nvl(to_char(pq_last_tm_readtm_accum), ' '),30)|| '  | ' ||
              lpad(to_char(pq_time_duration, '99999999.00'), 12)    || ' ' ||
              lpad(pq_monitor_time_1, 30)                           || ' ' ||
              lpad(pq_monitor_time_2, 30); 
          END IF;

          UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
        END IF; -- IF (pq_queue_id_1 IS NOT NULL) AND 
                --    (pq_queue_id_2 IS NOT NULL)

      END LOOP;   -- loop through different persistent queues;
      CLOSE gr_c1;

      EXECUTE IMMEDIATE 'COMMIT';

      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');


      -- dump sorted queues
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Top Persistent Queues');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~~~~~~~');
      prvt_dump_sorted_queues('ENQUEUED_MSGS',        TRUE);
      prvt_dump_sorted_queues('DEQUEUED_MSGS',        TRUE);
      prvt_dump_sorted_queues('ENQUEUE_RATE',         TRUE);
      prvt_dump_sorted_queues('DEQUEUE_RATE',         TRUE);
      prvt_dump_sorted_queues('AVG_TIME_PER_ENQUEUE', TRUE);
      prvt_dump_sorted_queues('AVG_TIME_PER_DEQUEUE', TRUE);
      prvt_dump_sorted_queues('PENDING_MSGS',         TRUE);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, breakline);
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');

    END IF;  -- db_version >= 11.1

    -------------------------------------------------------------------------- 
    -- sec 4: dump accumulative stats for buffered queues                    
    -------------------------------------------------------------------------- 

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Accumulative Statistics on '|| 
                                           'Buffered Queues');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('~', 42, '~'));

    -- dump complete accumulative stats
    IF db_version >= 11.2 THEN 

      title_line1_p1 := 'Instance                          Queue      Queue   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue   Avg Time per';
      title_line2_p1 := '  Number                           Name         ID       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)    Enqueue(us)';

      title_line1_p2 := '   Avg Time per      Avg CPU Time      Avg CPU Time   Transformation   Rule Eval  |  Pending             Oldest';
      title_line2_p2 := '    Dequeue(us)   per Enqueue(us)   per Dequeue(us)         Time(us)    Time(us)  |     Msgs              MsgID';

      title_line1_p3 := '        Oldest Msg Enqueue Time              Last Enqueue Time              Last Dequeue Time  |         Time                          First                           Last';
      title_line2_p3 := '                 (UTC timezone)                 (UTC timezone)                 (UTC timezone)  |  Duration(s)                   Monitor Time                   Monitor Time';

      title_line3    := rpad('-', 419, '-');

    ELSIF db_version = 10.2 OR db_version=11.1 THEN

      title_line1_p1 := 'Instance                          Queue      Queue   Enqueued   Dequeued   Spilled   Expired       Enqueue       Dequeue';
      title_line2_p1 := '  Number                           Name         ID       Msgs       Msgs      Msgs      Msgs   Rate(msg/s)   Rate(msg/s)';

      title_line1_p2 := '  |  Pending';
      title_line2_p2 := '  |     Msgs';

      title_line1_p3 := '  |         Time                          First                           Last';
      title_line2_p3 := '  |  Duration(s)                   Monitor Time                   Monitor Time';

      title_line3    := rpad('-', 212, '-');

    END IF;
  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line1_p1 || title_line1_p2 || 
                                           title_line1_p3);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line2_p1 || title_line2_p2 || 
                                           title_line2_p3);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, title_line3);


    FOR i IN gr_bqueueCursor LOOP

      -- adjust collected data (e.g., first snapshot) before analysis
      prvt_postrun_adjust(db_version, i.INST_ID, i.QUEUE_ID, FALSE);

      -- get the first snapshot
      BEGIN
        SELECT * into bq_line_start 
          FROM aqmon_bqueues_tab 
         WHERE INST_ID      = i.INST_ID
           AND QUEUE_SCHEMA = i.QUEUE_SCHEMA
           AND QUEUE_ID     = i.QUEUE_ID
           AND QUEUE_NAME   = i.QUEUE_NAME
           AND memseq       = (SELECT min(memseq) 
                                 FROM aqmon_bqueues_tab 
                                WHERE INST_ID      = i.INST_ID
                                  AND QUEUE_SCHEMA = i.QUEUE_SCHEMA
                                  AND QUEUE_ID     = i.QUEUE_ID
                                  AND QUEUE_NAME   = i.QUEUE_NAME);
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          bq_line_start := NULL;
        WHEN TOO_MANY_ROWS THEN 
          -- this should not happen, unless the table 
          -- is manually manipulated.
          bq_line_start := NULL; 
      END;


      -- get the last snapshot
      BEGIN
        SELECT * into bq_line_end 
          FROM aqmon_bqueues_tab 
         WHERE INST_ID      = i.INST_ID
           AND QUEUE_SCHEMA = i.QUEUE_SCHEMA
           AND QUEUE_ID     = i.QUEUE_ID
           AND QUEUE_NAME   = i.QUEUE_NAME
           AND memseq       = (SELECT max(memseq) 
                                 FROM aqmon_bqueues_tab 
                                WHERE INST_ID      = i.INST_ID
                                  AND QUEUE_SCHEMA = i.QUEUE_SCHEMA
                                  AND QUEUE_ID     = i.QUEUE_ID
                                  AND QUEUE_NAME   = i.QUEUE_NAME);
      EXCEPTION
        WHEN NO_DATA_FOUND THEN
          bq_line_end := NULL;
        WHEN TOO_MANY_ROWS THEN 
          -- this should not happen, unless the table 
          -- is manually manipulated.
          bq_line_end := NULL; 
      END;

     
      IF (bq_line_start.QUEUE_ID IS NOT NULL) AND 
         (bq_line_end.QUEUE_ID IS NOT NULL) THEN
        --calculate accumulative statistics
    
        -- unit: second          
        bq_time_duration:=(CAST(bq_line_end.MONITOR_TIME as DATE)-
                           CAST(bq_line_start.MONITOR_TIME as DATE))*86400.0;

        bq_enqueued_msgs_td    := nvl(bq_line_end.CNUM_MSGS, 0)  
                                - nvl(bq_line_start.CNUM_MSGS, 0);
        bq_spilled_msgs_td     := nvl(bq_line_end.CSPILL_MSGS, 0)   
                                - nvl(bq_line_start.CSPILL_MSGS, 0);
        bq_expired_msgs_td     := nvl(bq_line_end.EXPIRED_MSGS, 0)   
                                - nvl(bq_line_start.EXPIRED_MSGS, 0);

        -- accum metrics
        bq_pending_msgs_accum  := nvl(bq_line_end.NUM_MSGS, 0);
        -- accum metrics end

        bq_dequeued_msgs_td := bq_enqueued_msgs_td -
             (bq_pending_msgs_accum - nvl(bq_line_start.NUM_MSGS,0));


        IF bq_time_duration = 0 THEN
          bq_enqueue_rate_td := 0;
          bq_dequeue_rate_td := 0;
        ELSE 
          bq_enqueue_rate_td  := bq_enqueued_msgs_td/bq_time_duration; 
          bq_dequeue_rate_td  := bq_dequeued_msgs_td/bq_time_duration; 
        END IF;

        IF db_version >= 11.2 THEN
          -- metrics for 11.2 or above
          BEGIN 
            Var := 'SELECT ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, '||
              '            ELAPSED_TRANSFORMATION_TIME, ' ||
              '            ELAPSED_RULE_EVALUATION_TIME, ' ||
              '            ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME' || 
              '  FROM aqmon_bqueues_tab ' ||
              ' WHERE INST_ID      = ' || i.INST_ID ||
              '   AND QUEUE_SCHEMA = ' || prvt_enquote_str(i.QUEUE_SCHEMA) ||
              '   AND QUEUE_ID     = ' || i.QUEUE_ID ||
              '   AND QUEUE_NAME   = ' || prvt_enquote_str(i.QUEUE_NAME) ||
              '   AND memseq = (SELECT min(memseq)' ||
              '     FROM aqmon_bqueues_tab' ||
              '    WHERE INST_ID     ='|| i.INST_ID ||
              '      AND QUEUE_SCHEMA='|| prvt_enquote_str(i.QUEUE_SCHEMA) ||
              '      AND QUEUE_ID    ='|| i.QUEUE_ID ||
              '      AND QUEUE_NAME  ='|| prvt_enquote_str(i.QUEUE_NAME)||')';

            EXECUTE IMMEDIATE var INTO bq_enqueue_ela_time_1, 
              bq_dequeue_ela_time_1, bq_trans_time_1, bq_rule_eval_time_1,
              bq_enqueue_cpu_time_1, bq_dequeue_cpu_time_1;


          EXCEPTION WHEN OTHERS THEN NULL;
          END;

          BEGIN 
            var := 'SELECT ELAPSED_ENQUEUE_TIME, ELAPSED_DEQUEUE_TIME, '||
              '            ELAPSED_TRANSFORMATION_TIME, ' ||
              '            ELAPSED_RULE_EVALUATION_TIME, ' ||
              '            ENQUEUE_CPU_TIME, DEQUEUE_CPU_TIME, ' || 
              '            LAST_ENQUEUE_TIME, LAST_DEQUEUE_TIME,' ||
              '            OLDEST_MSGID, OLDEST_MSG_ENQTM' ||
              '  FROM aqmon_bqueues_tab ' ||
              ' WHERE INST_ID      = ' || i.INST_ID ||
              '   AND QUEUE_SCHEMA = ' || prvt_enquote_str(i.QUEUE_SCHEMA) ||
              '   AND QUEUE_ID     = ' || i.QUEUE_ID ||
              '   AND QUEUE_NAME   = ' || prvt_enquote_str(i.QUEUE_NAME) ||
              '   AND memseq = (SELECT max(memseq)' ||
              '     FROM aqmon_bqueues_tab' ||
              '    WHERE INST_ID     ='|| i.INST_ID ||
              '      AND QUEUE_SCHEMA='|| prvt_enquote_str(i.QUEUE_SCHEMA) ||
              '      AND QUEUE_ID    ='|| i.QUEUE_ID ||
              '      AND QUEUE_NAME  ='|| prvt_enquote_str(i.QUEUE_NAME)||')';

            EXECUTE IMMEDIATE var INTO bq_enqueue_ela_time_2, 
              bq_dequeue_ela_time_2, bq_trans_time_2, bq_rule_eval_time_2,
              bq_enqueue_cpu_time_2, bq_dequeue_cpu_time_2, 
              bq_last_enqtm_accum, bq_last_deqtm_accum, 
              bq_oldest_msgid_accum,  bq_oldest_msg_enqtm_accum;

          EXCEPTION WHEN OTHERS THEN NULL;
          END;


          bq_trans_time_td     := nvl(bq_trans_time_2,0) 
                                - nvl(bq_trans_time_1,0);
          bq_rule_eval_time_td := nvl(bq_rule_eval_time_2,0)
                                - nvl(bq_rule_eval_time_1,0);

          IF bq_enqueued_msgs_td = 0 THEN 
            bq_avg_time_per_enq_td := 0;
            bq_cpu_time_per_enq_td := 0;
          ELSE
            bq_avg_time_per_enq_td :=(nvl(bq_enqueue_ela_time_2,0)
              - nvl(bq_enqueue_ela_time_1, 0))/bq_enqueued_msgs_td;
            bq_cpu_time_per_enq_td :=(nvl(bq_enqueue_cpu_time_2, 0)
              - nvl(bq_enqueue_cpu_time_1, 0))/bq_enqueued_msgs_td;
          END IF;

          IF bq_dequeued_msgs_td = 0 THEN 
            bq_avg_time_per_deq_td := 0;
            bq_cpu_time_per_deq_td := 0;
          ELSE
            bq_avg_time_per_deq_td :=(nvl(bq_dequeue_ela_time_2,0)
               - nvl(bq_dequeue_ela_time_1, 0))/bq_dequeued_msgs_td;
            bq_cpu_time_per_deq_td :=(nvl(bq_dequeue_cpu_time_2, 0)
               - nvl(bq_dequeue_cpu_time_1, 0))/bq_dequeued_msgs_td;
          END IF;

        ELSE   
          -- for 11.1 or below 
          bq_avg_time_per_enq_td  := 0;
          bq_avg_time_per_deq_td  := 0;
          bq_cpu_time_per_enq_td  := 0;
          bq_cpu_time_per_deq_td  := 0;
          bq_trans_time_td        := 0;
          bq_rule_eval_time_td    := 0;

          bq_oldest_msgid_accum      := NULL;
          bq_oldest_msg_enqtm_accum  := NULL;
          bq_last_enqtm_accum        := NULL;
          bq_last_deqtm_accum        := NULL;
        END IF;


        sqlstxt := 'DELETE FROM aqmon_bqueues_accum_tab WHERE ' ||
                   ' INST_ID=:1 AND QUEUE_ID=:2'; 
        EXECUTE IMMEDIATE sqlstxt USING i.INST_ID, i.QUEUE_ID;
 
        sqlstxt := 'INSERT INTO aqmon_bqueues_accum_tab VALUES ' ||
                   ' (:1, :2, :3, :4, :5, :6, :7, :8, :9, :10, :11, :12, ' ||
                   '  :13, :14, :15, :16, :17, :18, :19, :20, :21, :22, :23)';
        EXECUTE IMMEDIATE sqlstxt USING i.INST_ID, i.QUEUE_ID, i.QUEUE_NAME,
          bq_enqueued_msgs_td, bq_dequeued_msgs_td, bq_spilled_msgs_td,
          bq_expired_msgs_td, bq_enqueue_rate_td, bq_dequeue_rate_td,
          bq_pending_msgs_accum,
          bq_avg_time_per_enq_td, bq_avg_time_per_deq_td,
          bq_cpu_time_per_enq_td, bq_cpu_time_per_deq_td,
          bq_trans_time_td, bq_rule_eval_time_td,
          bq_oldest_msgid_accum, bq_oldest_msg_enqtm_accum, 
          bq_last_enqtm_accum, bq_last_deqtm_accum, bq_time_duration, 
          bq_line_start.MONITOR_TIME, bq_line_end.MONITOR_TIME;

        IF db_version >= 11.2 THEN
          var := lpad(i.INST_ID, 8)                               || ' ' ||
            lpad(i.QUEUE_NAME, 30)                                || ' ' || 
            lpad(i.QUEUE_ID, 10)                                  || ' ' ||
            lpad(bq_enqueued_msgs_td, 10)                         || ' ' ||
            lpad(bq_dequeued_msgs_td, 10)                         || ' ' ||
            lpad(bq_spilled_msgs_td,9)                            || ' ' ||
            lpad(bq_expired_msgs_td,9)                            || ' ' ||
            lpad(to_char(bq_enqueue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(bq_dequeue_rate_td, 'FM9999999.00'), 13) || ' ' ||
            lpad(to_char(bq_avg_time_per_enq_td*10000, 'FM99999999.00'), 14) 
                        || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(bq_avg_time_per_deq_td*10000, 'FM99999999.00'), 14) 
                        || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(bq_cpu_time_per_enq_td*10000, 'FM99999999.00'), 17) 
                        || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(bq_cpu_time_per_deq_td*10000, 'FM99999999.00'), 17) 
                        || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(bq_trans_time_td*10000, 'FM99999999.00'), 16)  
                        || ' ' ||    -- from centi-second to micro-second
            lpad(to_char(bq_rule_eval_time_td*10000, 'FM99999999.00'), 11) 
                        || '  |' ||  -- from centi-second to micro-second
            lpad(bq_pending_msgs_accum, 9)                        || ' ' ||
            lpad(nvl(to_char(bq_oldest_msgid_accum), ' '), 18)    || ' ' ||
            lpad(nvl(to_char(bq_oldest_msg_enqtm_accum), ' '),30) || ' ' ||
            lpad(nvl(to_char(bq_last_enqtm_accum), ' '),30)       || ' ' ||
            lpad(nvl(to_char(bq_last_deqtm_accum), ' '),30)       || '  | ' ||
            lpad(to_char(bq_time_duration, '99999999.00'), 12)    || ' ' ||
            lpad(bq_line_start.MONITOR_TIME, 30)                  || ' ' ||
            lpad(bq_line_end.MONITOR_TIME, 30); 
        ELSE
          var := lpad(i.INST_ID, 8)                               || ' ' ||
            lpad(i.QUEUE_NAME, 30)                                || ' ' || 
            lpad(i.QUEUE_ID, 10)                                  || ' ' ||
            lpad(bq_enqueued_msgs_td, 10)                         || ' ' ||
            lpad(bq_dequeued_msgs_td, 10)                         || ' ' ||
            lpad(bq_spilled_msgs_td,9)                            || ' ' ||
            lpad(bq_expired_msgs_td,9)                            || ' ' ||
            lpad(to_char(bq_enqueue_rate_td, 'FM9999999.00'),13)  || ' ' ||
            lpad(to_char(bq_dequeue_rate_td, 'FM9999999.00'),13)|| '  |' ||
            lpad(bq_pending_msgs_accum, 9)                     || '  | ' ||
            lpad(to_char(bq_time_duration, '99999999.00'), 12)    || ' ' ||
            lpad(bq_line_start.MONITOR_TIME, 30)                  || ' ' ||
            lpad(bq_line_end.MONITOR_TIME, 30); 
        END IF;

        UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
      END IF; -- IF (bq_line_start IS NOT NULL) AND (bq_line_end IS NOT NULL)

    END LOOP;   -- FOR i IN gr_bqueueCursor LOOP

    EXECUTE IMMEDIATE 'COMMIT';

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');


    -- dump sorted queues
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Top Buffered Queues');
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~~~~~');

    prvt_dump_sorted_queues('ENQUEUED_MSGS',  FALSE);
    prvt_dump_sorted_queues('DEQUEUED_MSGS',  FALSE);
    prvt_dump_sorted_queues('ENQUEUE_RATE',   FALSE);
    prvt_dump_sorted_queues('DEQUEUE_RATE',   FALSE);
    prvt_dump_sorted_queues('PENDING_MSGS',   FALSE);

    IF db_version >= 11.2 THEN
      prvt_dump_sorted_queues('AVG_TIME_PER_ENQUEUE', FALSE);
      prvt_dump_sorted_queues('AVG_TIME_PER_DEQUEUE', FALSE);
    END IF;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, breakline);

    -------------------------------------------------------------------------- 
    -- sec 5: Others - lengthy output, so put them at the end of the report
    -------------------------------------------------------------------------- 

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');     
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');     

    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Tablespace Info');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~~~~'); 
    var := rpad('Queue Owner', 30)              || ' ' ||
           rpad('Tablespace Name', 30)          || ' ' ||
           rpad('Queue Table', 30)              || ' ' ||
           rpad('Block Size', 12)               || ' ' ||
           rpad('Extent Management', 20)        || ' ' ||
           rpad('Segment Space Management', 30);   
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 153, '-'));
    FOR i in gr_db1 LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
    END LOOP;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  


    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, 'Segment Info');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, '~~~~~~~~~~~~');
    var := rpad('Segment Type', 18)    || ' ' ||
           rpad('Queue Owner', 30)     || ' ' ||
           rpad('Tablespace Name', 30) || ' ' ||
           rpad('Segment Name', 81)    || ' ' ||
           rpad('Table Name', 30)      || ' ' ||
           rpad('Bytes', 15)           || ' ' ||
           rpad('Blocks', 10)          || ' ' ||   
           rpad('Extents', 10); 
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, var);
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, rpad('-', 230, '-'));
    FOR i in gr_db2a LOOP  
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa); 
    END LOOP;
    FOR i in gr_db2b LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
    END LOOP;
    FOR i in gr_db2c LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
    END LOOP;
    FOR i in gr_db2d LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
    END LOOP;
    FOR i in gr_db2e LOOP
      UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, i.aa);        
    END LOOP;
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');  
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');     
    UTL_FILE.PUT_LINE(AQMON_REPORT_OUTPUT, ' ');     

    -------------------------------------------------------------------------- 
    -- sec 6: end                                                            
    -------------------------------------------------------------------------- 

    UTL_FILE.FCLOSE(AQMON_REPORT_OUTPUT);
 
    -- display log file info
    IF logdir_path IS NOT NULL THEN
      logdir_path := TRIM(logdir_path);
      IF SUBSTR(logdir_path, LENGTH(logdir_path), 1) != '/' THEN
        logdir_path := CONCAT(logdir_path, '/');
      END IF;
      prvt_echo_line('Summary report is generated: ' || logdir_path || 
                     AQMON_REPORT_FNAME);
    END IF;


  END get_report;
  
  FUNCTION get_parameter(
   param_name        IN varchar2,
   param_value       IN OUT varchar2,
   table_name        IN varchar2,
   table_param_name  IN varchar2,
   table_value       IN varchar2
  )  return boolean AS
  statement varchar2(4000);
   begin
   -- construct query 
   statement :=  'select ' || table_value || ' from ' || table_name || ' where ' 
                || table_param_name || '=''' || param_name || '''';

   begin
     execute immediate statement into param_value;
    exception when no_data_found then
     -- data is not found, so return FALSE
     return FALSE;
   end;
   -- data found, so return TRUE
   return TRUE;
  end get_parameter;
  
  PROCEDURE verify_init_parameter( 
   param_name         IN varchar2, 
   expected_value     IN varchar2,
   verbose            IN boolean,
   more_info          IN varchar2 := NULL,
   more_info2         IN varchar2 := NULL,
   at_least           IN boolean := FALSE,
   is_error           IN boolean := FALSE,
   use_like           IN boolean := FALSE,
   -- may not be necessary
   alert_if_not_found IN boolean := TRUE
  )  AS
   current_val_num  NUMBER;
   expected_val_num NUMBER;
   current_value    varchar2(512);
   prefix           varchar2(20);
   matches          boolean := FALSE;
   comparison_str   varchar2(20);
  begin
    -- Set prefix as warning or error
   if is_error then
     prefix := '+  ERROR:  ';
   else
     prefix := '+  WARNING:  ';
   end if;

   -- Set comparison string
   if at_least then
    comparison_str := ' at least ';
   elsif use_like then
    comparison_str := ' like ';
   else
    comparison_str := ' set to ';
   end if;

   -- Get value
   if get_parameter(param_name, current_value, 'v$parameter', 'name', 'value') = FALSE 
     and alert_if_not_found then
    -- Value isn't set, so output alert
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, prefix || 'The parameter ''' || param_name || ''' should be'
                         || comparison_str || '''' || expected_value 
                         || ''', instead it has been left to its default value.'); 
    if verbose and more_info is not null then
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, more_info);
      if more_info2 is not null then
        UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, more_info2);
      end if;
    end if;
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, '+');
    return;
   end if;

   -- See if the expected value is what is actually set
   if use_like then
    -- Compare with 'like'
    if current_value like '%'||expected_value||'%' then
      matches := TRUE;
    end if;
   elsif at_least then
     -- Do at least
     current_val_num := to_number(current_value);
     expected_val_num := to_number(expected_value);
     if current_val_num >= expected_val_num then
      matches := TRUE;
     end if;
   else
    -- Do normal comparison
    if current_value = expected_value then
      matches := TRUE;
    end if;
   end if;
  
   if matches = FALSE then
    -- The values don't match, so alert
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, prefix || 'The parameter ''' || param_name || ''' should be'
                         || comparison_str || '''' || expected_value 
                         || ''', instead it has the value ''' || current_value || '''.'); 
    if verbose and more_info is not null then
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, more_info);
      if more_info2 is not null then
        UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, more_info2);
      end if;
    end if;
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, '+');
   end if;

  end verify_init_parameter;
 
  --This procedure performs a variety of checks on an individual queue table. 
  
  PROCEDURE hcheck_queue_table(
   owner_name IN VARCHAR2 DEFAULT NULL,
   queue_table IN VARCHAR2 DEFAULT NULL) AS
   
    --Checking for invalid queues associated with a particular queue table
	
    cursor invalid_q_objs is 
	     select object_name from dba_objects o, dba_queues q 
	     where o.object_type='QUEUE' and o.status = 'INVALID'
         and o.object_name=q.name and q.queue_table = queue_table 
		 and q.owner=owner_name;
	 
	--Checking for invalid objects associated with a queue table
	
	cursor invalid_qt_objs is
		 select object_name from dba_objects o, dba_queue_tables qt 
	     where o.status = 'INVALID' and o.object_name=qt.queue_table 
		 and qt.queue_table = queue_table
		 and qt.owner = owner_name;
	 
	
	TYPE objnum_t IS TABLE OF dba_objects.object_id%TYPE;
	obj_ids objnum_t;
	
	TYPE objnam_t IS TABLE OF dba_objects.object_name%TYPE;
	obj_nams	objnam_t;
	
	TYPE objtyp_t IS TABLE OF dba_objects.object_type%TYPE;
	obj_typs objtyp_t;
	
	statement varchar2(4000);
	
	log_entries BOOLEAN := TRUE;
	
  BEGIN
   
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;	
   
   --check the objects which exist for a particular queue table to see if anything is missing or is a check
   --for invalid objects enough?
   --check whether any of the objects associated with a queue table are invalid
   
   FOR i in invalid_q_objs LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queues or queue related objects owned by '||owner_name||' are invalid');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;

	  --Check for objects which have been created which have a name based on the queue like rules/rulesets
	  --Cannot use a bind variable for object_name as this involves a like
	  statement := 'select object_id, object_name, object_type from dba_objects where object_name like '||CHR(39)||'%'||i.object_name||'%'||CHR(39)||' AND owner = :1 AND status = '||CHR(39)||'INVALID'||CHR(39)||' order by owner, object_name';	 
	  --UTL_FILE.put_line(statement);
	  
      EXECUTE IMMEDIATE statement  
      BULK COLLECT INTO obj_ids, obj_nams, obj_typs
	  USING owner_name;
	  
	  FOR j IN obj_ids.FIRST..obj_ids.LAST LOOP    
		UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'object_id= '||obj_ids(j)||', object_name = '||obj_nams(j)||' and object_type = '||obj_typs(j) );		
	  END LOOP;
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
    END LOOP;
	
	log_entries := TRUE;
	
	FOR i in invalid_qt_objs LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queue_table or queue_table related objects owned by '||owner_name||' are invalid');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;

	  --Check for objects which have been created which have a name based on the queue like rules/rulesets
	  --Cannot use a bind variable for object_name as this involves a like
	  statement := 'select object_id, object_name, object_type from dba_objects where object_name like '||CHR(39)||'%'||i.object_name||'%'||CHR(39)||' AND owner = :1 AND status = '||CHR(39)||'INVALID'||CHR(39)||' order by owner, object_name';	 
	  --UTL_FILE.put_line(statement);
	  
      EXECUTE IMMEDIATE statement  
      BULK COLLECT INTO obj_ids, obj_nams, obj_typs
	  USING owner_name;
	  
	  FOR j IN obj_ids.FIRST..obj_ids.LAST LOOP    
		UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'object_id= '||obj_ids(j)||', object_name = '||obj_nams(j)||' and object_type = '||obj_typs(j) );		
		--UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Advice on validating the objects ...' );
	  END LOOP;
	  
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
    END LOOP;
   	
    --Check the size of the objects associated with a queue table and if any exceed a certain critera flag this up?
	--This information is listed in the Monitor output but perhaps it should be in the healthcheck component
	--given that we are only really interested in any particularly large segments
	
	--Could also check is IOTS are needing manually coalesced.
      
  END hcheck_queue_table;
 
  --This procedure checks for logically corrupted messages in a particular queue table
  --It does not take into account dequeue_log issues which may exist in 11.2 onwards

  PROCEDURE hcheck_queue_table_messages(
   owner_name IN VARCHAR2 DEFAULT NULL,
   queue_table IN VARCHAR2 DEFAULT NULL) AS
   
   cursor qt_corr_msgs is select queue_name, msgid, subscriber# subscriber, corruption_cause
                          from aqhc_messages_healthcheck mhc 
						  where mhc.owner=owner_name and mhc.queue_table=queue_table;
						  
   log_entries BOOLEAN := TRUE;
						  
  BEGIN
  
    IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
     ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
    END IF;	
   
    FOR i in qt_corr_msgs LOOP
	
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queue_table '||owner_name||'.'||queue_table||' has logically corrupted messages.');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
	 
	 log_entries := FALSE;
	 
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Queue_name: '||i.queue_name||' MSGID: '||i.msgid||' associated with subscriber#: '||i.subscriber||' and corruption cause '||i.corruption_cause);
	 
	END LOOP;
   
  END hcheck_queue_table_messages;
  
  PROCEDURE hcheck_clean_queue_table_msg(
    owner_name IN VARCHAR2 DEFAULT NULL,
    queue_table IN VARCHAR2 DEFAULT NULL) AS
    opt DBMS_AQADM.aq$_purge_options_t;
    CURSOR qt_purge_msgs IS 
    	SELECT msgid, subscriber_id, corruption_type FROM 
      table(aq$_get_corrupted_messages(owner_name, queue_table, 9));
      log_entries BOOLEAN := TRUE;
    qt_purge_msg qt_purge_msgs%ROWTYPE;
  BEGIN
  	opt.block := false;
    opt.delivery_mode := dbms_aq.PERSISTENT_OR_BUFFERED;
    IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
    END IF;
    IF log_entries THEN
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queue_table '||owner_name||'.'||queue_table||' will be cleaned for corrupted message.');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
      log_entries := false;
    END IF;
    OPEN qt_purge_msgs;
    LOOP
    	FETCH qt_purge_msgs INTO qt_purge_msg;
    	EXIT WHEN qt_purge_msgs%NOTFOUND;
    -- FOR qt_purge_msg IN qt_purge_msgs LOOP
      sys.DBMS_AQADM_SYS.purge_queue_table(queue_table => owner_name||'.'||queue_table, 
      purge_condition => 'PURGE_BY_MSGID msgid='''||qt_purge_msg.msgid||'''',  purge_options => opt);
      DBMS_OUTPUT.put_line('purged: '||qt_purge_msg.msgid);
    END LOOP;
  END hcheck_clean_queue_table_msg;

      
  --This procedure checks relevant AQ init.ora parameters are set correctly 

  PROCEDURE hcheck_parameters AS
   -- Change the variable below to FALSE if you just want the warnings and errors, not the advice
   verbose            boolean := TRUE;
   row_count          number;
   newline            varchar2(1) := ' ';
   mycheck number;
  BEGIN

  
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;	
   
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_parameters');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
   
   -- Then warnings
   /* This is not required at present as far as I am aware but may be in 12g
   verify_init_parameter('compatible', '11.2.0', verbose, 
                        '+    To use the new Streams features introduced in Oracle Database 11g Release 2, '||  
                        'this parameter must be set to a value greater than ''11.2.0''',
                        use_like => TRUE);
    */

   -- explictly check if aq_tm_processes has been manually set to 0.  If so, raise error.
   begin
     select 1 into mycheck from v$parameter where name = 'aq_tm_processes' and value = '0'
      and (ismodified <> 'FALSE' OR isdefault='FALSE');
     if mycheck = 1 then  
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'ERROR 1:');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The parameter ''aq_tm_processes'' should not be explicitly set to 0!');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Queue monitoring is disabled for all queues.');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'To resolve this problem, set the value to 1 using:  ALTER SYSTEM SET AQ_TM_PROCESSES=1 SCOPE=MEMORY;  ');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'and remove the entry from your init.ora or spfile via ALTER SYSTEM RESET AQ_TM_PROCESSES SCOPE=SPFILE SID='||CHR(39)||'*'||CHR(39)||';');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, newline);
     end if;
    exception when no_data_found then null;
   end;
   
      -- explictly check if job_queue_processes has been manually set to 0.  If so, raise error.
   begin
     select 1 into mycheck from v$parameter where name = 'job_queue_processes' and value = '0'
      and (ismodified <> 'FALSE' OR isdefault='FALSE');
     if mycheck = 1 then
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'ERROR 2:');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The parameter ''job_queue_processes'' should not be explicitly set to 0!');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Propagation/AQ PL/SQL notification is disabled for all queues.');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'To resolve this problem, set the value to 10 using:  ALTER SYSTEM SET JOB_QUEUE_PROCESSES=10 SCOPE=MEMORY;');
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'and if this is 11.1 onwards remove the entry from your init.ora or spfile via ALTER SYSTEM RESET JOB_QUEUE_PROCESSES SCOPE=SPFILE SID='||CHR(39)||'*'||CHR(39)||';');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, newline); 
     end if;
    exception when no_data_found then null;
   end;

   -- explictly check if aq_tm_processes has been manually set to 10.  If so, raise error.

   begin
    select 1 into mycheck from v$parameter where name = 'aq_tm_processes' and isdefault = 'FALSE'
     and value = '10';
    if mycheck = 1 then
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'ERROR 3:');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The parameter ''aq_tm_processes'' should not be explicitly set to 10!');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'With this setting, queue monitoring is disabled for buffered queues.');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'To resolve this problem, set the value to 1 using:  ALTER SYSTEM SET AQ_TM_PROCESSES=1; SCOPE=MEMORY');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'and remove the entry from your init.ora or spfile via ALTER SYSTEM RESET AQ_TM_PROCESSES SCOPE=SPFILE SID='||CHR(39)||'*'||CHR(39)||';');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, newline);
	end if;
    exception when no_data_found then null;
   end;
	
   begin
    select 1 into mycheck from v$parameter where name = 'aq_tm_processes' and isdefault = 'FALSE'
     and value = '10';
    if mycheck = 1 then
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'ERROR 3:');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The parameter ''aq_tm_processes'' should not be explicitly set to 10!');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'With this setting, queue monitoring is disabled for buffered queues.');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'To resolve this problem, set the value to 1 using:  ALTER SYSTEM SET AQ_TM_PROCESSES=1; SCOPE=MEMORY');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'and remove the entry from your init.ora or spfile via ALTER SYSTEM RESET AQ_TM_PROCESSES SCOPE=SPFILE SID='||CHR(39)||'*'||CHR(39)||';');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, newline);
	end if;
    exception when no_data_found then null;
   end;

    -- AQ hidden parameters available in 11.2
	--
    -- In kpon.h
    -- _emon_max_active_connections - maximum open connections to clients per emon - static
	-- _emon_outbound_connect_timeout - timeout for completing connection set up to clients - static
	-- _emon_regular_ntfn_slaves - number of EMON slaves doing regular database notifications - static
	-- _client_ntfn_pingtimeout - timeout to connect to unreachable notification clients -static
	-- _client_ntfn_pinginterval - time between pings to unreachable notification clients - static
	-- _client_ntfn_pingretries - number of times to ping unreachable notification clients - static 
	-- _client_ntfn_cleanup_interval - interval after which dead client registration cleanup task repeats - static
	-- _srvntfn_job_deq_timeout - srvntfn job deq timeout - dynamic
    -- _srvntfn_jobsubmit_interval - srvntfn job submit interval - dynamic
    -- _srvntfn_max_concurrent_jobs - srvntfn max concurrent jobs -dynamic
    -- _srvntfn_q_msgcount - srvntfn q msg count for job exit - dynamic
    -- _srvntfn_q_msgcount_inc - srvntfn q msg count increase for job submit - dynamic

	-- In kwqit.h
	-- _orph_cln_interval - qmon periodic interval for removed subscriber messages cleanup - static
	-- _aq_tm_deqcountinterval - dequeue count interval for Time Managers to cleanup DEQ IOT BLOCKS -dynamic
    -- _aq_tm_scanlimit - scan limit for Time Managers to clean up IOT -dynamic
    -- _aq_tm_statistics_duration - number 	statistics collection window duration - static

    -- In knla.h	
    -- _max_aq_persistent_queue_memory - max aq persistent queue memory - dynamic
	
	-- In kwqbm.h
    -- _buffered_message_spill_age - Buffered message spill age - static
    
	-- In kwqb.h
	-- _buffered_publisher_flow_control_threshold - Flow control threshold for buffered publishers except capture
    -- _bufq_stop_flow_control - Stop enforcing flow control for buffered queues 
	
	-- In kjc.h
	-- _lm_postevent_buffer_size - postevent buffer size 	
	
	-- In kwqi.h
	-- _deq_execute_reset_time - deq execute reset time -dynamic
    -- _deq_max_fetch_count - deq max fetch count - dynamic
    -- _deq_maxwait_time - Change wait times between dequeue calls - static
    -- _close_deq_by_cond_curs - Close Dequeue By Condition Cursors - static
	
	-- In kwqdl.h
	-- _deq_ht_child_latches - deq ht child latches -static
    -- _deq_ht_max_elements - deq ht max elements - dynamic
    -- _deq_large_txn_size - deq large txn size - dynamic
    -- _deq_log_array_size - deq log array size - static
   
  verify_init_parameter('streams_pool_size', '0', TRUE, 
                        'If this parameter is 0 and sga_target is non-zero, then autotuning of the streams pool is implied.'||newline||
                        'If the sga_target parameter is set to 0 and streams_pool_size is 0,'|| newline||
                        '10% of the shared pool will be used for Streams.' || newline ||
                        'If sga_target is 0, the minimum recommendation for streams_pool_size is 200M.'|| newline||
                        'Note you must bounce the database if changing the ',
                        'value from zero to a nonzero value.  But if simply increasing this' || newline ||
                        'value from an already nonzero value, the database need not be bounced.',
                        at_least=> TRUE);



   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_parameters');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));

  END hcheck_parameters;
 
  --This procedure checks that the aq metadata in system.aq$_queues and system.aq$queue_tables is consistent with what is in obj$
  --and if not reports the objects which are not present in these tables and any associated components of the queue tables
  --which still exist.
  
  PROCEDURE hcheck_dictionary AS
  
    --Entries missing from dba_queues and therefore from system.aq$_queues
	
    cursor orphan_qs is 
	 select owner, object_name from dba_objects where object_type='QUEUE'
      minus
     select owner, name from dba_queues order by owner;
	 
	--Entries missing from dba_queue_tables and therefore from system.aq$_queue_tables
	
	cursor orphan_qts is
	 select u.name OWNER, o.name OBJECT_NAME from sys.obj$ o, sys.tab$ t, sys.user$ u where u.user#=o.owner#
     and t.obj# = o.obj# and o.type#=2 and bitand(t.property, 131072)=131072 and o.name not like '%AQ$_%'
    union
     select u.name, o.name from sys.obj$ o, sys.tab$ t, sys.user$ u where u.user#=o.owner#
     and t.obj# = o.obj# and o.type#=2 and bitand(t.property, 131072)=131072 and o.name='AQ$_MEM_MC' OR o.name='AQ$_MEM_SC'
    minus
    select owner, queue_table from dba_queue_tables;
	
	TYPE objnum_t IS TABLE OF dba_objects.object_id%TYPE;
	obj_ids objnum_t;
	
	TYPE objnam_t IS TABLE OF dba_objects.object_name%TYPE;
	obj_nams	objnam_t;
	
	TYPE objtyp_t IS TABLE OF dba_objects.object_type%TYPE;
	obj_typs objtyp_t;
	
	statement varchar2(4000);
	
	log_entries BOOLEAN := true;
	
  BEGIN
   
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;
   
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_dictionary checks');
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	
    FOR i in orphan_qs LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queues do not have relevant entries in system.aq$_queues');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'or the reference to the associated queue table has been lost due to it');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'not being present in the system.aq$_queue_tables and are present in obj$.');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Problem queue ' || i.owner ||'.'|| i.object_name);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	 
	  --Check for objects which have been created which have a name based on the queue like rules/rulesets
	  --Cannot use a bind variable for object_name as this involves a like
	  statement := 'select object_id, object_name, object_type from dba_objects where object_name like '||CHR(39)||'%'||i.object_name||'%'||CHR(39)||' AND owner = :1 order by owner, object_name';	 
	  --UTL_FILE.put_line(statement);
	  
      EXECUTE IMMEDIATE statement  
      BULK COLLECT INTO obj_ids, obj_nams, obj_typs
	  USING i.owner;
	  
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Related queue objects are: ');
	  FOR j IN obj_ids.FIRST..obj_ids.LAST LOOP    
		UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'object_id= '||obj_ids(j)||', object_name = '||obj_nams(j)||' and object_type = '||obj_typs(j) );		
	  END LOOP;
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
    END LOOP;
	
	log_entries := true;

	FOR i in orphan_qts LOOP
	
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following queue_tables do not have the relevant entries in system.aq$_queue_tables but are present in obj$');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
	 
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Problem queue_table ' || i.owner ||'.'|| i.object_name);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	 
	  --Check for objects which have been created which have a name based on the queue_table name
	  --Cannot use a bind variable for object_name as this involves a like
	  statement := 'select object_id, object_name, object_type from dba_objects where object_name like '||CHR(39)||'%'||i.object_name||'%'||CHR(39)||' AND owner = :1 order by owner, object_name';	 
	  --UTL_FILE.put_line(statement);
	  
      EXECUTE IMMEDIATE statement  
      BULK COLLECT INTO obj_ids, obj_nams, obj_typs
	  USING i.owner;
	  
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Related queue_table objects are: ');
	  
	  FOR j IN obj_ids.FIRST..obj_ids.LAST LOOP	    
		UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'object_id= '||obj_ids(j)||', object_name = '||obj_nams(j)||' and object_type = '||obj_typs(j) );
	  END LOOP;
	  
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	
    END LOOP;
	
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_dictionary checks');
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	
  END hcheck_dictionary;
 
  --This procedure will gather information relating to propagation schedules
 
  PROCEDURE hcheck_propagation_sender AS
  
   cursor prob_qsched is 
      select * from dba_queue_schedules where schedule_disabled='Y';
  
   cursor not_verified_qsched is 
      select schema_name, queue_name, destination, verified from sys.aq$_message_types where verified !='T';
  
   cursor orphaned_rows_propstat is 
      select count(*) orphan_entries, queue_id, destination FROM sys.aq$_propagation_status props
      WHERE NOT EXISTS (SELECT Null FROM dba_queues qs WHERE  qs.qid = props.queue_id) and props.status !=2
	  GROUP BY queue_id, destination;
	   
   cursor pend_msg_status is select count(*) entry, ps.status from sys.aq$_pending_messages pm, sys.aq$_propagation_status ps 
                         where pm.sequence = ps.sequence group by status order by status;
						 
   cursor prop_pre11 is select p.spid, substr(p.program, instr(p.program,'(')+1,4) program, p.PGA_USED_MEM, p.PGA_ALLOC_MEM, p.PGA_FREEABLE_MEM, p.PGA_MAX_MEM from v$process p, dba_jobs_running jr, v$session s, dba_jobs j 
                       where s.sid=jr.sid and s.paddr=p.addr and jr.job=j.job and j.what like '%sys.dbms_aqadm.aq$_propaq(job)%';
					   
   cursor prop_11on is select substr(p.program, instr(p.program,'(')+1,4) program, s.slave_os_process_id, p.PGA_USED_MEM, p.PGA_ALLOC_MEM, p.PGA_FREEABLE_MEM, p.PGA_MAX_MEM from v$process p, dba_scheduler_running_jobs s 
                      where s.slave_os_process_id = p.spid and job_name like '%AQ_JOB$_%';
								  
   v_statement varchar2(4000);
   v_tablespace_name dba_segments.tablespace_name%TYPE;
   v_bytes           dba_segments.bytes%TYPE;
   v_blocks          dba_segments.blocks%TYPE;
   v_extents         dba_segments.extents%TYPE;
   log_entries BOOLEAN := true;
   db_version   NUMBER;
  
   --rc_value number;
  
  BEGIN
  
   prvt_get_database_version;
   db_version := prvt_get_param('APT_DATABASE_VERSION');
  
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;
   
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_propagation_sender checks');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
    --check for disabled propagation schedules and report these
	
    FOR i in prob_qsched LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following propagation schedules are disabled');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Queue :'|| i.schema ||'.'|| i.qname);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Destination :'|| i.destination);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Type of messages being propagated :'|| i.message_delivery_mode);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Last error date :'|| i.last_error_date ||' at :'||i.last_error_time);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Last error message :'|| i.last_error_msg);	 
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
    END LOOP;
	
    --Check sys.aq$_message_types for schedules which indicate that the target queue has been dropped.
		
	FOR i in not_verified_qsched LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'dbms_aq_monitor.hcheck_propagation_sender');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following propagation schedules have a verified status of D or F');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'which means that we believe the target queue has been dropped');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Queue :'|| i.schema_name ||'.'|| i.queue_name);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Destination :'|| i.destination);
	 IF i.verified = 'D' THEN
       UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The verified column is indicating that the target queue has been dropped');
	 ELSE
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The verified column is indicating that the target queue type does not match');
	 END IF;
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	 
	 /* Be good if we could execute verify_queue_types to see if the status is correct.
	  How can we easily determine what the target queue should be in the case of queue to dblink?
      In the case of queue_to_queue it should auto verify.	
	 BEGIN
       DBMS_AQADM.VERIFY_QUEUE_TYPES(src_queue_name => i.schema_name ||'.'|| i.queue_name,
                                     dest_queue_name => ?,
                                     destination => i.destination,
                                     rc => rc_value);
       dbms_output.put_line('rc_value code is '||rc_value);
	 END;
	 */
	  
    END LOOP;	
	
	--Check for orphaned rows in sys.aq$_propagation_status which can cause sys.aq$_pending_messages
	--to grow. 
	
	FOR i in orphaned_rows_propstat LOOP
		 
	 IF log_entries THEN
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The following orphan entries exist in sys.aq_propagation_status');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'These entries can cause sys.aq$_pending_messages to remain undeleted');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  log_entries := false;
	 END IF;
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Number of Orphan entries :'|| i.orphan_entries);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Associated with this non-existant queue id :'|| i.queue_id);
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'for the following Destination :'|| i.destination); 
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
    END LOOP;
	
	--Check the physical size of the sys.aq$_pending_messages table. The criterion chosen is 100MB. Is this reasonable?
	
	v_statement := 'SELECT TABLESPACE_NAME, BYTES, BLOCKS, EXTENTS FROM dba_segments WHERE OWNER = '||CHR(39)||'SYS'||CHR(39)||' AND SEGMENT_NAME = '||CHR(39)||'AQ$_PENDING_MESSAGES'||CHR(39);
	
	EXECUTE IMMEDIATE v_statement INTO v_tablespace_name, v_bytes, v_blocks, v_extents;
	
	IF (v_bytes/(1024*1024)) > 100 THEN
	
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'sys.aq$_pending_messages which resides in the '||v_tablespace_name||' tablespace');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'is '|| (v_bytes/(1024*1024))||' M in size');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	 
	 --Given that the table is large check count(*) from sys.aq$_pending_messages. To see how many entries are present

	  FOR i in pend_msg_status LOOP
		 
       UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, i.entry ||' entries are present in sys.aq$_pending_messages and these are associated with status ='||i.status||' entries.');
	  
      END LOOP;	 

      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));	  
	 
	END IF;	

	--Link dba_queue_schedules with v$process to get the spid and report memory consumption of active schedules
	
	IF db_version >= 11.1 THEN
	
	   FOR i in prop_11on LOOP
		 
       UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Process: '||i.program||' Process id: '||i.slave_os_process_id);
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA used memory: '||ROUND((i.pga_used_mem/(1024*1024)),2)||' MB PGA Allocated memory: '||ROUND((i.pga_alloc_mem/(1024*1024)),2)||' MB');
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA freeable memory: '||ROUND((i.pga_freeable_mem/(1024*1024)),2)||' MB PGA Maximum memory: '||ROUND((i.pga_max_mem/(1024*1024)),2)||' MB');
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
      END LOOP;	 
	   
	 ELSE

	  FOR i in prop_pre11 LOOP
		 
       UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Process: '||i.program||' Process id: '||i.spid);
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA used memory: '||ROUND((i.pga_used_mem/(1024*1024)),2)||' MB PGA Allocated memory: '||ROUND((i.pga_alloc_mem/(1024*1024)),2)||' MB');
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA freeable memory: '||ROUND((i.pga_freeable_mem/(1024*1024)),2)||' MB PGA Maximum memory: '||ROUND((i.pga_max_mem/(1024*1024)),2)||' MB');
	   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
      END LOOP;	 	 
	
	END IF;
   
   --Could look at v$propagation_sender information for buffered messaging and flag issues
   
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_propagation_sender checks');
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));	
   
  END hcheck_propagation_sender;
  
  --Check for issues associated with the receiver processes
  
  PROCEDURE hcheck_propagation_receiver AS
  
  --A propagation schedule can propagate both buffered and persistent messages so I assume
  --that this means that there will be a single job_queue_process on the source corresponding
  --with a single receiver process on the target.
  
  cursor prop_rec_11on is select p.spid, s.process, p.PGA_USED_MEM, p.PGA_ALLOC_MEM, p.PGA_FREEABLE_MEM, p.PGA_MAX_MEM 
                          from v$session s, v$process p 
                          where s.paddr=p.addr and action='Propagation Receiver';		  							  		  
  
  db_version   NUMBER;
  
  BEGIN
  
   prvt_get_database_version;
   db_version := prvt_get_param('APT_DATABASE_VERSION');
  
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;
   
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_propagation_receiver checks');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
   
   	IF db_version >= 11.1 THEN
	
	   FOR i in prop_rec_11on LOOP
	   
	    --This check is required for windows
		
	    IF instr(i.process,':') != 0 THEN
		 
         UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The Propagation Receiver has the os pid: '||i.spid||' and the corresponding source job_queue_process has the os pid: '||substr(i.process, instr(i.process,':')+1));
		 
		ELSE
		
		 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'The Propagation Receiver has the os pid: '||i.spid||' and the corresponding source job_queue_process has the os pid: '||substr(i.process, instr(i.process,':')+1));
		 
		END IF;
		
	    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA used memory: '||ROUND((i.pga_used_mem/(1024*1024)),2)||' MB PGA Allocated memory: '||ROUND((i.pga_alloc_mem/(1024*1024)),2)||' MB');
	    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA freeable memory: '||ROUND((i.pga_freeable_mem/(1024*1024)),2)||' MB PGA Maximum memory: '||ROUND((i.pga_max_mem/(1024*1024)),2)||' MB');
	    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
      END LOOP;	 
	   
	 ELSE

	  --Need to think about whether this is possible to do easily for pre-11g without using 
	  --a database link from the source to the target.
	  
	  NULL;
	
	END IF;
	
	--Could look at v$propagation_receiver information for buffered messaging and flag issues
   
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_propagation_receiver checks');
    UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));	
   
  END;
 
  --This procedure will check for issues relating to queue monitor processes.
  
  PROCEDURE hcheck_qmon AS
  
   cursor qmn_proc is select spid, substr(program, instr(program,'(')+1,4) process_name, PGA_USED_MEM, PGA_ALLOC_MEM, PGA_FREEABLE_MEM, PGA_MAX_MEM 
                     from v$process where background='1' and  (program like '%QMNC%' OR program like '%Q00%');
  
   db_version   NUMBER;
   
  BEGIN
  
   prvt_get_database_version;
   db_version := prvt_get_param('APT_DATABASE_VERSION');
  
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;
   
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_qmon checks');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));  
   
    FOR i in qmn_proc LOOP
	   		 
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, i.process_name || ' has the os pid: '||i.spid);			
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA used memory: '||ROUND((i.pga_used_mem/(1024*1024)),2)||' MB PGA Allocated memory: '||ROUND((i.pga_alloc_mem/(1024*1024)),2)||' MB');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA freeable memory: '||ROUND((i.pga_freeable_mem/(1024*1024)),2)||' MB PGA Maximum memory: '||ROUND((i.pga_max_mem/(1024*1024)),2)||' MB');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
    END LOOP;
	
	-- For 11.2 we have V$QMON_COORDINATOR_STATS, V$QMON_SERVER_STATS, V$QMON_TASK_STATS, V$QMON_TASKS
	-- what in particular should be monitored from a healthcheck point of view?
	
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_qmon checks');
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));  
	
  END hcheck_qmon;
  
  --This procedure will check for issues relating to notification.
 
  PROCEDURE hcheck_notification AS
  
   cursor emn_proc is select spid, substr(program, instr(program,'(')+1,4) process_name, PGA_USED_MEM, PGA_ALLOC_MEM, PGA_FREEABLE_MEM, PGA_MAX_MEM 
                     from v$process where background='1' and  (program like '%EMNC%' OR program like '%E00%');
  
	db_version   NUMBER;	
	
  BEGIN
    
	prvt_get_database_version;
   db_version := prvt_get_param('APT_DATABASE_VERSION');
  
   IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      NULL;
    ELSE
	  AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'a', 2000);
   END IF;
   
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_notification checks');
   UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));  
   
    FOR i in emn_proc LOOP
	   		 
      UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, i.process_name || ' has the os pid: '||i.spid);			
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA used memory: '||ROUND((i.pga_used_mem/(1024*1024)),2)||' MB PGA Allocated memory: '||ROUND((i.pga_alloc_mem/(1024*1024)),2)||' MB');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'PGA freeable memory: '||ROUND((i.pga_freeable_mem/(1024*1024)),2)||' MB PGA Maximum memory: '||ROUND((i.pga_max_mem/(1024*1024)),2)||' MB');
	  UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
    END LOOP;
	
	-- For 11.2 we have V$EMON
	-- what in particular should be monitored from a healthcheck point of view?
	
	-- In terms of AQ PL/SQL Notification what can we check?
	-- Expired messages in the Notification exception queue
	-- Backlog in the notification queue.
	-- In 10.2 long lasting register driver jobs.
	-- In 11.1 onwards what should we check in terms of the scheduler jobs?
	-- Check the tunable parameters that are available in 11.1 onwards for notification?
	
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_notification checks');
	UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13)); 
	
  END hcheck_notification;
  
  --The procedure will invoke all the AQ healthcheck procedures.
  
  PROCEDURE hcheck_all AS
  
   cursor qts is select owner, queue_table from dba_queue_tables;
  
  BEGIN
  
     BEGIN
        AQMON_HCREPORT_OUTPUT := UTL_FILE.FOPEN(AQMON_LOGDIR, AQMON_HCREPORT_FNAME, 'w', 2000);
      EXCEPTION
        WHEN OTHERS THEN
          prvt_echo_error('Cannot create healthcheck file in the log directory.');  
      END;   
	  
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_all checks');
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	  
	 hcheck_parameters;
	 hcheck_dictionary;
	 
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'Start of dbms_aq_monitor.hcheck_queue_tables and hcheck_queue_table_messages');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
	   
	 FOR i IN qts LOOP
	   
       hcheck_queue_table(i.owner,i.queue_table);
	   hcheck_queue_table_messages(i.owner, i.queue_table);
	   
	 END LOOP;
	 
	 UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_queue_tables and hcheck_queue_table_messages');
     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, CHR(13));
   
     hcheck_propagation_sender;
	 hcheck_propagation_receiver;
     hcheck_qmon;
     hcheck_notification;	

     UTL_FILE.put_line(AQMON_HCREPORT_OUTPUT, 'End of dbms_aq_monitor.hcheck_all checks');	 

     IF UTL_FILE.IS_OPEN(AQMON_HCREPORT_OUTPUT) THEN
      UTL_FILE.FCLOSE(AQMON_HCREPORT_OUTPUT);
     END IF;
     	 
  END hcheck_all;  
  
END dbms_aq_monitor_util;
/  
show errors
